<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_DBDriverPdopgsql.php,v 1.1 2009/01/11 05:34:32 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * DB操作クラス（PDO）
 */
require_once dirname(__FILE__) . '/../SyL_DBPdo.php';

/**
 *  DB操作クラス（PDO::Pgsql）
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_DBDriverPdopgsql.php,v 1.1 2009/01/11 05:34:32 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_DBDriverPdopgsql extends SyL_DBPdo
{
    /**
     * エンコード変換テーブル
     *
     *  PostgreSQL <=> PHP
     *  (client)       (client)
     * 
     * ※全て小文字で記述
     *
     * @access protected
     * @var string
     */
    public $encode_table = array(
      // PHP, DB
      array('eucjp-win', 'euc-jp'),
      array('euc-jp',    'euc-jp'),
      array('euc',       'euc-jp'),
      array('sjis-win',  'sjis' ),
      array('sjis-win',  'mskanji'),
      array('sjis-win',  'shiftjis'),
      array('sjis-win',  'win932'),
      array('sjis-win',  'windows932'),
      array('sjis',      'sjis'),
      array('utf-8',     'utf8'),
      array('utf-8',     'unicode')
    );

    /**
     * DB接続実行
     * 
     * @access public
     * @param string データベース名
     * @param string ユーザー名
     * @param string パスワード
     * @param string ホスト名
     * @param string ポート番号
     * @param bool 持続的接続を行うか
     * @return boolean 接続OK: true, 接続NG: false
     */
    public function openConnection($dbname, $user, $passwd, $host, $port, $persistent)
    {
        $conn_dbtype  = substr($this->dbtype, 3);
        $conn_strings = array();
        if ($user) $conn_strings[]   = 'user='     . $user;
        if ($passwd) $conn_strings[] = 'password=' . $passwd;
        if ($dbname) $conn_strings[] = 'dbname='   . $dbname;
        if ($host) $conn_strings[]   = 'host='     . $host;
        if ($port) $conn_strings[]   = 'port='     . $port;

        $conn_string = $conn_dbtype . ':' . implode(' ', $conn_strings);
        return $this->pdoConnection($conn_string);
    }

    /**
     * クライアント側文字エンコーティング設定
     * 
     * @access public
     * @param string DB文字コード
     * @param string PHP側エンコード
     */
    public function setClientEncoding($client_encode, $server_encode='')
    {
        $client_encode2 = strtolower($client_encode);
        foreach ($this->encode_table as $encode_table) {
            if ($encode_table[0] == $client_encode2) {
                $client_encode2 = $encode_table[1];
                break;
            }
        }
        $this->connection->exec("SET CLIENT_ENCODING TO {$client_encode2}");
        parent::setClientEncoding($client_encode, $server_encode);
    }

    /**
     * SQLを実行し、ページ毎にデータを取得
     *
     * @access public
     * @param string SQL文
     * @param array 実行結果格納配列
     * @param object ページオブジェクト
     * @param int 1ページの表示件数
     * @param int 表示対象ページ数 
     * @return bool 実行結果
     */
    public function queryPageRef($sql, &$result, &$pager, $limit=20, $page=1)
    {
        // ページングオブジェクト
        $pager = $this->getPager($limit, $page);
        // 総件数取得SQL作成
        $sql_count = "SELECT COUNT(*) FROM (" . $sql . ") AS SyL$";
        // 件数取得SQL実行
        if (!$this->queryRef($sql_count, $record, 'one')) return false;
        $pager->setCountAll($record);

        // ページ単位レコード取得
        $sql_page  = $sql;
        $sql_page .= " LIMIT "  . $pager->getCount();
        $sql_page .= " OFFSET " . $pager->getStartOffset();

        // データ取得SQL実行
        return $this->queryRef($sql_page, $result, 'all');
    }

    /**
     * 接続しているDBサーバーのバージョンを取得する
     * 
     * @access public
     * @return string DBのバージョン
     */
    public function getVersion()
    {
        $sql = "SELECT VERSION()";
        $this->queryRef($sql, $version, 'one');
        if (preg_match('/^PostgreSQL (.+) on .+$/', trim($version), $matches)) {
            $version = $matches[1];
        }
        return $version;
    }
}
