<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_DBDao.php,v 1.1 2009/01/11 05:34:33 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * SQL条件作成クラス
 */
require_once 'SyL_DBSqlTableConditions.php';
/**
 * SQL関連条件作成クラス
 */
require_once 'SyL_DBSqlTableRelations.php';

/**
 * DB用データアクセスクラス
 * 
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_DBDao.php,v 1.1 2009/01/11 05:34:33 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_DBDao
{
    /**
     * DBオブジェクト
     *
     * @access protected
     * @var object
     */
    var $conn = null;
    /**
     * スキーマオブジェクトの配列
     *
     * @access protected
     * @var array
     */
    var $tables = array();
    /**
     * 取得ページ数
     *
     * @access protected
     * @var int
     */
    var $page = 0;
    /**
     * 取得ページ件数
     *
     * @access protected
     * @var int
     */
    var $limit = 10;
    /**
     * ページクラス
     *
     * @access protected
     * @var object
     */
    var $pager = null;
    /**
     * SELECTしたカラム
     *
     * @access private
     * @var array
     */
    var $select_headers = array();
    /**
     * ORDER BYしたカラム
     *
     * @access private
     * @var array
     */
    var $sort_columns = array();
    /**
     * 連続DML時にエラーが発生した場合、
     * 処理を停止するフラグ
     *
     * @access protected
     * @var bool
     */
    var $error_stop = true;
    /**
     * エラーメッセージ配列
     *
     * @access protected
     * @var array
     */
    var $error_messages = array();

    /**
     * コンストラクタ
     *
     * @access public
     * @param object DBオブジェクト
     */
    function SyL_DBDao(&$conn)
    {
        $this->conn =& $conn;
    }

    /**
     * テーブル条件オブジェクト作成
     *
     * @access public
     * @return object 条件オブジェクト
     */
    function createCondition()
    {
        return new SyL_DBSqlTableConditions();
    }

    /**
     * テーブル関連オブジェクト作成
     *
     * @access public
     * @return object 関連オブジェクト
     */
    function createRelation()
    {
        return new SyL_DBSqlTableRelations();
    }

    /**
     * SELECT項目を取得
     *
     * @access public
     * @return array SELECT項目配列
     */
    function getSelectHeaders()
    {
        return $this->select_headers;
    }

    /**
     * ソート項目を取得
     *
     * @access public
     * @return array ソート項目配列
     */
    function getSortColumns()
    {
        return $this->sort_columns;
    }

    /**
     * ページ表示パラメータをセット
     *
     * @access public
     * @param int 表示ページ数
     * @param int 1ページの件数
     */
    function setPage($page=1, $limit=20)
    {
        $this->page  = $page;
        $this->limit = $limit;
    }

    /**
     * ページオブジェクトを取得
     *
     * @access public
     * @return object ページオブジェクト
     */
    function &getPager()
    {
        return $this->pager;
    }

    /**
     * DBからデータを取得する
     *
     * @access public
     * @param object or array テーブルオブジェクト 
     * @param object 関連オブジェクト
     * @return array 取得データ
     */
    function select($table, $relation=null)
    {
        // 内部処理用に変換
        $table = $this->arrangeTables($table);

        $selects   = array();
        $froms     = array();
        $wheres    = array();
        $group_bys = array();
        $order_bys = array();
        $this->select_headers = array();
        foreach(array_keys($table) as $name) {
            $selects    = array_merge($selects, $table[$name]->getSelectColumns($this->conn));
            $froms      = array_merge($froms, (array)$table[$name]->getSelectName());
            $conditions = $table[$name]->getConditions();
            if ($conditions) {
                $where = $conditions->getWhere($this->conn);
                if ($where) {
                    $wheres[] = $where;
                }
            }
            $group_bys  = array_merge($group_bys, $table[$name]->getGroups());
            $order_bys  = array_merge($order_bys, $table[$name]->getSorts());
            // SELECTカラム保持
            $this->select_headers = array_merge($this->select_headers, $table[$name]->getSelectHeaders());
        }

        // 関連条件セット
        $from = '';
        if (is_object($relation)) {
            $join = $relation->getJoin($this->conn);
            if ($join) {
                $from = $join;
            }
        }
        if (!$from) {
            $from = implode(', ', $froms);
        }
        $where    = implode(' AND ', $wheres);
        $group_by = implode(', ', $group_bys);
        $order_by = implode(', ', $order_bys);
        // カレントソートカラム保持
        $this->sort_columns = str_replace(' ', '.', $order_bys);

        $sql  = "";
        $sql .= "SELECT ";
        if (count($selects) > 0) {
        $sql .=   implode(', ', $selects) . " ";
        } else {
        $sql .=   "* ";
        }
        $sql .= "FROM ";
        $sql .=   $from . " ";
        if ($where) {
        $sql .= "WHERE ";
        $sql .=   $where . " ";
        }
        if ($group_by) {
        $sql .= "GROUP BY ";
        $sql .=   $group_by . " ";
        }
        if ($order_by) {
        $sql .= "ORDER BY ";
        $sql .=   $order_by . " ";
        }

//echo $sql;
//exit;

        if ($this->page === 0) {
            // 通常
            return $this->conn->query($sql);
        } else {
            // ページングあり
            $this->conn->queryPageRef($sql, $result, $this->pager, $this->limit, $this->page);
            return $result;
        }
    }

    /**
     * DBに登録する
     *
     * @access public
     * @param mixed テーブルオブジェクト
     * @return bool true: OK, false: エラー
     */
    function insert($table)
    {
        // 内部処理用に変換
        $table = $this->arrangeTables($table);
        // INSERT実行
        return $this->exec($table, 'insert');
    }

    /**
     * DBを更新する
     *
     * @access public
     * @param mixed テーブルオブジェクト
     * @return bool true: OK, false: エラー
     */
    function update($table)
    {
        // 内部処理用に変換
        $table = $this->arrangeTables($table);
        // UPDATE実行
        return $this->exec($table, 'update');
    }

    /**
     * DBを削除する
     *
     * @access public
     * @param mixed テーブルオブジェクト
     * @return bool true: OK, false: エラー
     */
    function delete($table)
    {
        // 内部処理用に変換
        $table = $this->arrangeTables($table);
        // DELETE実行
        return $this->exec($table, 'delete');
    }

    /**
     * テーブルオブジェクトをパラメータから内部処理用に変換
     *
     * @access private
     * @param mixed テーブルオブジェクト
     * @param string テーブルのエイリアス名
     */
    function arrangeTables($table)
    {
        $result = array();
        // 引数がテーブルオブジェクトの場合
        if (is_object($table)) {
            $alias = $table->getAliasName();
            $result[$alias] = $table;
        } else if (is_array($table)) {
            foreach ($table as $tmp) {
                $result += $this->arrangeTables($tmp);
            }
        }
        return $result;
    }

    /**
     * DB登録、更新用にカラムを整形する
     *
     * @access private
     * @param array テーブルオブジェクト
     * @param string insert or update or delete
     * @return bool true: OK、false: エラー
     */
    function exec($table, $action)
    {
        $result = true;
        foreach (array_keys($table) as $name) {
            $conditions = $table[$name]->getConditions(false);
            $where      = ($conditions) ? $conditions->getWhere($this->conn) : '';
            if (!$this->conn->execPerform($table[$name]->getName(), $table[$name]->getDataColumns(), $action, $where)) {
                $result = false;
                $this->error_messages[] = $this->conn->errorInfo();
                if ($this->error_stop) {
                    break;
                }
            }
        }
        return $result;
    }

    /**
     * テーブルレコード情報を取得する
     *
     * @access public
     * @param object テーブルオブジェクト
     * @param array 条件パラメータ
     * @return array テーブルレコード情報
     */
    function getRecord(&$table, $parameters)
    {
        $condition = $this->createCondition();
        foreach ($parameters as $name => $value) {
            $condition->addEqual($name, $value);
        }
        $table->setConditions($condition);
        $result = $this->select($table);
        return (count($result) > 0) ? $result[0] : array();
    }

    /**
     * テーブル情報を取得する
     *
     * @access public
     * @param object テーブルオブジェクト
     * @param array 条件パラメータ
     * @param array ソートパラメータ
     * @return array テーブル情報
     */
    function getRecords(&$table, $parameters=array(), $sorts=array())
    {
        if (count($parameters) > 0) {
            $condition = $this->createCondition();
            foreach ($parameters as $name => $value) {
                $condition->addEqual($name, $value);
            }
            $table->setConditions($condition);
        }
        foreach ($sorts as $name) {
            $table->addSort($name);
        }
        return $this->select($table);
    }

    /**
     * テーブルのMAX IDを取得する
     *
     * @access public
     * @param object テーブルオブジェクト
     * @param string MAX IDカラム
     * @param array 条件パラメータ
     * @return int MAX ID
     */
    function getMaxId(&$table, $column, $parameters=array())
    {
        if (count($parameters) > 0) {
            $condition = $this->createCondition();
            foreach ($parameters as $name => $value) {
                $condition->addEqual($name, $value);
            }
            $table->setConditions($condition);
        }
        $table->set($column, 'MAXID', array('MAX'));
        $result = $this->select($table);
        return (count($result) > 0) ? $result[0]['MAXID'] : null;
    }

    /**
     * 主キー存在チェック
     *
     * @access public
     * @param object テーブルオブジェクト
     * @param array 主キーデータ配列
     * @return bool true: 主キーが存在する, false: 主キーが存在しない
     */
    function hasPrimary($table, $datas)
    {
        // 条件オブジェクト作成
        $conditions = $this->createCondition();
        foreach ($table->getPrimary(false) as $tmp) {
            if (isset($datas[$tmp])) {
                $conditions->addEqual($tmp, $datas[$tmp]);
            }
        }
        if (count($conditions->gets()) == 0) {
            // 主キーが見つからない場合
            return false;
        }
        // 参照解除
        $table_tmp = version_compare(PHP_VERSION, '5.0.0', '>=')
                   ? unserialize(serialize($table))
                   : $table;
        // 条件オブジェクトセット
        $table_tmp->setConditions($conditions);
        // 主キー存在判定
        return (count($this->select($table_tmp)) > 0);
    }

    /**
     * 一意キー存在チェック
     *
     * @access public
     * @param object テーブルオブジェクト
     * @param array 一意キーデータ配列
     * @return bool true: 一意キーが存在する, false: 一意キーが存在しない
     */
    function hasUnique($table, $datas)
    {
        foreach ($table->getUniques(false) as $uniques) {
            // 条件オブジェクト作成
            $conditions = $this->createCondition();
            foreach ($uniques as $unique) {
                if (isset($datas[$unique])) {
                    if ($datas[$unique] !== '') {
                        $conditions->addEqual($unique, $datas[$unique]);
                    } else {
                        // 空 = null がある場合は、スキップ
                        continue 2;
                    }
                }
            }
            if (count($conditions->gets()) > 0) {
                // 参照解除
                $table_tmp = version_compare(PHP_VERSION, '5.0.0', '>=')
                           ? unserialize(serialize($table))
                           : $table;
                // 一意キーが存在する場合
                $table_tmp->setConditions($conditions);
                // 一意キー存在判定
                if (count($this->select($table_tmp)) > 0) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * 外部キー存在チェック
     *
     * @access public
     * @param object テーブルオブジェクト
     * @param array 外部キーデータ配列
     * @param string 対象テーブル名
     * @return bool true: 外部キーが存在する, false: 外部キーが存在しない
     */
    function hasForeign($table, $datas, &$ftabble)
    {
        foreach ($table->getForeigns(false) as $ftabble => $foreigns) {
            // 条件オブジェクト作成
            $conditions = $this->createCondition();
            foreach ($foreigns as $column1 => $column2) {
                if (isset($datas[$column1])) {
                    if ($datas[$column1] !== '') {
                        $conditions->addEqual($column2, $datas[$column1]);
                    } else {
                        // 空 = null がある場合は、スキップ
                        continue 2;
                    }
                }
            }

            if (count($conditions->gets()) > 0) {
                // 外部キーが存在する場合
                $table_tmp = new SyL_DBDaoTable($ftabble);
                $table_tmp->setConditions($conditions);
                // 外部キー存在判定
                if (count($this->select($table_tmp)) > 0) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * エラーを判定する
     *
     * @access public
     * @return bool true: エラーあり, false: エラーなし
     */
    function isError()
    {
        return (count($this->error_messages) > 0);
    }

    /**
     * エラーメッセージを全て取得する
     *
     * @access public
     * @return array エラーメッセージ配列
     */
    function getErrors()
    {
        return $this->error_messages;
    }

    /**
     * スキーマ取得オブジェクトを取得する
     *
     * @access public
     * @return object スキーマ取得オブジェクト
     */
    function &getSchema()
    {
        return $this->conn->getSchema();
    }

    /**
     * 接続オブジェクトを取得する
     *
     * @access public
     * @return object 接続オブジェクト
     */
    function &getConnection()
    {
        return $this->conn;
    }
}
