<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_DBSqlTableRelations.php,v 1.1 2009/01/11 05:34:33 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * SQL関連条件作成クラス
 * 
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_DBSqlTableRelations.php,v 1.1 2009/01/11 05:34:33 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_DBSqlTableRelations
{
    /**
     * 条件配列
     * 
     * @access private
     * @var array
     */
    var $conditions = array();

    /**
     * 結合条件文を取得する
     *
     * @access public
     * @param object コネクションオブジェクト
     * @return string SQL条件文
     */
    function getJoin(&$conn)
    {
        $from = '';
        $first = true;
        $aliases = array();
        foreach ($this->conditions as $condition) {
            if ($first) {
                $from .= $condition[0]['name'] . ' ';
                if ($condition[0]['name'] != $condition[0]['alias']) {
                    $from .= $condition[0]['alias'] . ' ';
                }
                $aliases[] = $condition[0]['alias'];
                $first = false;
            }

            list($column1, $column2)    = array_map('trim', explode('=', $condition[3][0]));
            list($alias1, $columnname1) = explode('.', $column1, 2);
            list($alias2, $columnname2) = explode('.', $column2, 2);

            // 既に結合したテーブルを検索する
            $a = $b = false;
            $join_alias = '';
            if (array_search($alias1, $aliases) !== false) {
                $join_alias = $alias2;
                $a = true;
            }
            if (array_search($alias2, $aliases) !== false) {
                $join_alias = $alias1;
                $b = true;
            }

            if ($a && $b) {
                // 既に結合済みのテーブルはスキップ
                continue;
            } else if (!$a && !$b) {
                trigger_error("[SyL error] Alias name not found", E_USER_ERROR);
            }

            switch ($condition[2]) {
            case '=':  $from .= "INNER JOIN ";      break;
            case '+':  $from .= "LEFT OUTER JOIN "; break;
            default: trigger_error("[SyL error] Not supported operator ({$condition[2]})", E_USER_ERROR);
            }

            if ($join_alias == $condition[0]['alias']) {
                $from .= $condition[0]['name'] . ' ';
                if ($condition[0]['name'] != $condition[0]['alias']) {
                    $from .= $condition[0]['alias'] . ' ';
                }
                $aliases[] = $condition[0]['alias'];
            } else if ($join_alias == $condition[1]['alias']) {
                $from .= $condition[1]['name'] . ' ';
                if ($condition[1]['name'] != $condition[1]['alias']) {
                    $from .= $condition[1]['alias'] . ' ';
                }
                $aliases[] = $condition[1]['alias'];
            }
            $from .= "ON " . implode(' AND ', $condition[3]) . ' ';
            if (is_object($condition[4])) {
                $from .= ' AND ' . $condition[4]->getWhere($conn) . ' ';
            }
        }

        return $from;
    }

    /**
     * addInnerJoinメソッドのエイリアス
     *
     * @access public
     * @param object テーブルオブジェクト1
     * @param object テーブルオブジェクト2
     * @param array 関連配列
     * @param object 条件オブジェクト
     */
    function addJoin($table1, $table2, $columns, $condition=null)
    {
        $this->addInnerJoin($table1, $table2, $columns, $condition);
    }

    /**
     * 結合条件を作成する（等価結合）
     *
     * @access public
     * @param object テーブルオブジェクト1
     * @param object テーブルオブジェクト2
     * @param array 関連配列
     * @param object 条件オブジェクト
     */
    function addInnerJoin($table1, $table2, $columns, $condition=null)
    {
        $column1 = array('name' => $table1->getName(), 'alias' => $table1->getAliasName());
        $column2 = array('name' => $table2->getName(), 'alias' => $table2->getAliasName());
        $this->add($column1, $column2, '=', $columns, $condition);
    }

    /**
     * 結合条件を作成する（左外部結合）
     *
     * @access public
     * @param object テーブルオブジェクト1
     * @param object テーブルオブジェクト2
     * @param array 関連配列
     * @param object 条件オブジェクト
     */
    function addLeftJoin($table1, $table2, $columns, $condition=null)
    {
        $column1 = array('name' => $table1->getName(), 'alias' => $table1->getAliasName());
        $column2 = array('name' => $table2->getName(), 'alias' => $table2->getAliasName());
        $this->add($column1, $column2, '+', $columns, $condition);
    }

    /**
     * 結合条件を作成する
     *
     * @access private
     * @param array カラム名1
     * @param array カラム名2
     * @param string 演算子
     */
    function add($column1, $column2, $operator, $columns, $condition)
    {
        switch ($operator) {
        case '=':
        case '+':
            break;
        default:
            trigger_error("[SyL error] Not supported operator ({$operator})", E_USER_ERROR);
        }
        $this->conditions[] = array($column1, $column2, $operator, $columns, $condition);
    }
}
