<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_MailReceivePop3.php,v 1.1 2009/01/11 05:34:31 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * ソケット接続クラス
 */
require_once dirname(__FILE__) . '/../../SyL_Socket.php';

/**
 * SMTPメール送信クラス
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_MailReceivePop3.php,v 1.1 2009/01/11 05:34:31 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_MailReceivePop3 extends SyL_MailReceive
{
    /**
     * ソケットクラス
     *
     * @access private
     * @var object
     */
    var $socket = null;
    /**
     * APOP使用判定
     *
     * @access private
     * @var bool
     */
    var $apop = false;
    /**
     * メールボックス内の総メール数
     *
     * @access private
     * @var int
     */
    var $total_nums = 0;
    /**
     * メールボックス内の総バイト数
     *
     * @access private
     * @var int
     */
    var $total_bytes = 0;
    /**
     * SMTPコマンドログ
     *
     * @access private
     * @var string
     */
    var $command_log = '';

    /**
     * 総メール件数を取得
     *
     * @access public
     * @return int 総メール件数
     */
    function getTotalNums()
    {
        return $this->total_nums;
    }

    /**
     * 総メールバイト数を取得
     *
     * @access public
     * @return int 総メールバイト数
     */
    function getTotalBytes()
    {
        return $this->total_bytes;
    }

    /**
     * APOPを有効にする
     *
     * @access public
     * @param bool APOP判定
     */
    function enableApop($apop=true)
    {
        $this->apop = (bool)$apop;
    }

    /**
     * メールサーバーに接続
     *
     * @access public
     * @return OK: true, NG: false
     */
    function connect()
    {
        if (is_object($this->socket)) {
            $this->quit();
        }

        if ($this->user == '') {
            $this->error_message = "[SyL error] User name not found";
            return false;
        }

        // ソケットオブジェクト作成
        $this->socket = new SyL_Socket($this->host, $this->port);
        $this->socket->setCommandCallBackFunc(array(&$this, 'receiveMessage'));

        // 念のため
        register_shutdown_function(array(&$this, 'quit'));

        if (!$this->socket->open()) {
            $this->error_message = $this->socket->getLastError();
            return false;
        } else {
            // APOP使用判定
            if ($this->apop) {
                // 接続メッセージ取得
                if (preg_match('/\+OK .*(<[^>]+>)/', $this->socket->receive(), $matches)) {
                    // ダイジェスト送信
                    $this->socket->send('APOP ' . $this->user . ' ' . md5($matches[1] . $this->passwd));
                    // ログインメッセージ取得
                    $this->socket->receive();
                } else {
                    $this->error_message = "[SyL error] APOP Challenge not found";
                    return false;
                }
            } else {
                // 接続メッセージ取得
                $this->socket->receive();
                // ユーザー名送信
                $this->socket->send('USER ' . $this->user);
                // ログインメッセージ取得
                $this->socket->receive();
                if (!$this->isError()) {
                    // パスワード送信
                    $this->socket->send('PASS ' . $this->passwd);
                    // ログインメッセージ取得
                    $this->socket->receive();
                    // メール数取得
                    if (!$this->isError()) {
                        $this->stat();
                    }
                }
            }
        }

        return !$this->isError();
    }

    /**
     * メール総件数と総バイト数を取得
     *
     * @access public
     */
    function stat()
    {
        $this->socket->send('STAT');
        $stat = trim($this->socket->receive());
        if (preg_match('/^\+OK/', $stat)) {
            $stats = explode(' ', $stat);
            $this->total_nums  = $stats[1];
            $this->total_bytes = $stats[2];
        }
    }

    /**
     * 各メールごとのバイト数を取得
     *
     * @access public
     * @return array 各メールごとのバイト数
     */
    function getList()
    {
        $list = array();
        $this->socket->send('LIST');
        if (preg_match('/^\+OK/', $this->socket->receive())) {
            while (true) {
                $receive = trim($this->socket->receive());
                if (!$receive || ($receive == '.')) {
                    break;
                }
                list($num, $bytes) = explode(' ', $receive, 2);
                $list[$num] = $bytes;
            }
        }
        return $list;
    }

    /**
     * メール受信実行
     *
     * @access public
     * @param int メールメッセージ番号
     * @return object メールメッセージオブジェクト
     */
    function receive($num)
    {
        if (!is_numeric($num) || ($num <= 0)) {
            $this->error_message = "[SyL error] Invalid mail message num ($num)";
            return false;
        }

        $data = '';
        $this->socket->send('RETR ' . $num);
        if (preg_match('/^\+OK/', $this->socket->receive())) {
            while (true) {
                $receive = $this->socket->receive();
                if (!$receive || (trim($receive) == '.')) {
                    break;
                }
                $data .= $receive;
            }
        } else {
            return false;
        }

        $message =& $this->createMessage();
        $message->setMessage($data);

        return $message;
    }

    /**
     * メール削除実行
     *
     * @access public
     * @param int メールメッセージ番号
     * @return bool true: 成功、false: エラー
     */
    function delete($num)
    {
        if (!is_numeric($num) || ($num <= 0)) {
            $this->error_message = "[SyL error] Invalid mail message num ($num)";
            return false;
        }
        $this->socket->send('DELE ' . $num);
        $this->socket->receive();
        return !$this->isError();
    }

    /**
     * メールサーバーとの接続を閉じる
     *
     * @access public
     */
    function quit()
    {
        if (is_object($this->socket)) {
            // 完了メッセージ送信
            $this->socket->send('QUIT');
            // 完了メッセージ取得
            $this->socket->receive();
            // ソケットを閉じる
            $this->socket->close();
        }
        $this->socket = null;
    }

    /**
     * SMTPコマンド受信メッセージ
     *
     * @access private
     * @param string send or receive
     * @param string コマンドメッセージ
     */
    function receiveMessage($type, $message)
    {
        $message = trim($message);
        $this->command_log .= "[{$type}] {$message}\n";
        if ($type == 'receive') {
            if (preg_match('/^\-ERR/', $message)) {
                $this->error_message = "[SyL error] POP3 command error ($message)";
            }
        }
    }

    /**
     * ソケットコマンド受信ログを取得
     *
     * @access public
     * @return string ソケットコマンド受信ログ
     */
    function getCommandLog()
    {
        return $this->command_log;
    }
}
