<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_MailSendSmtp.php,v 1.1 2009/01/11 05:34:32 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * ソケット接続クラス
 */
require_once dirname(__FILE__) . '/../../SyL_Socket.php';

/**
 * SMTPメール送信クラス
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_MailSendSmtp.php,v 1.1 2009/01/11 05:34:32 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_MailSendSmtp extends SyL_MailSend
{
    /**
     * ソケットクラス
     *
     * @access private
     * @var object
     */
    var $socket = null;
    /**
     * SMTPコマンドログ
     *
     * @access private
     * @var string
     */
    var $command_log = '';

    /**
     * メールサーバーに接続
     *
     * @access public
     * @return OK: true, NG: false
     */
    function connect()
    {
        if (is_object($this->socket)) {
            $this->quit();
        }
        // ソケットオブジェクト作成
        $this->socket = new SyL_Socket($this->host, $this->port);
        $this->socket->setCommandCallBackFunc(array(&$this, 'receiveMessage'));

        // 念のため
        register_shutdown_function(array(&$this, 'quit'));

        if (!$this->socket->open()) {
            $this->error_message = $this->socket->getLastError();
            return false;
        } else {
            // 接続メッセージ取得
            $this->socket->receive();
            // ログインメッセージ送信
            $this->socket->send('HELO ' . $this->host);
            // ログインメッセージ取得
            $this->socket->receive();

            // SMTP AUTH 判定
            if (!$this->isError() && ($this->user !== '')) {
                // 認証情報送信
                $this->socket->send('AUTH PLAIN ' . base64_encode("{$this->user}\0{$this->user}\0{$this->passwd}"));
                // 認証結果取得
                $this->socket->receive();
            }
        }

        return !$this->isError();
    }

    /**
     * メール送信実行
     *
     * @access public
     * @param object メールメッセージオブジェクト
     * @return OK: true, NG: false
     */
    function send($message)
    {
        $from = $message->getFrom();
        $to   = $message->getTo();

        if (count($from) == 0) {
            $this->error_message = "[SyL error] From Address not found";
            return false;
        }
        if (count($to) == 0) {
            $this->error_message = "[SyL error] To Address not found";
            return false;
        }

        // ソケット作成判定
        $once = false;
        if ($this->socket == null) {
            $once = true;
            $this->connect();
        }

        // コマンド発行 ------------------------------------------------------------

        // 送信元セットメッセージ送信
        $this->socket->send('MAIL FROM: ' . $from[0]);
        // 送信元セットメッセージ取得
        $this->socket->receive();

        foreach ($message->getRcptTo() as $to) {
            // 送信先セットメッセージ送信
            $this->socket->send('RCPT TO: ' . $to);
            // 送信先セットメッセージ取得
            $this->socket->receive();
        }

        if ($this->isError()) {
            if ($once) {
                $this->quit();
            }
            return false;
        }

        // メールヘッダー ----------------------------------------------------------

        // データセットメッセージ送信
        $this->socket->send('DATA');
        // データセットメッセージ取得
        $this->socket->receive();

        // メールヘッダを含む全文取得
        $data = $message->getMessage(true);
        // 最後に「.」を付加
        $data .= '.';

        if ($this->isError()) {
            if ($once) {
                $this->quit();
            }
            return false;
        }

        // -------------------------------------------------------------------------

        // 送信データセットメッセージ送信
        $this->socket->send($data);
        // 送信データセットメッセージ取得
        $this->socket->receive();

        if ($this->isError()) {
            if ($once) {
                $this->quit();
            }
            return false;
        }

        // リセットメッセージ送信
        $this->socket->send('RSET');
        // リセットメッセージ取得
        $this->socket->receive();

        if ($once) {
            $this->quit();
        }

        return true;
    }

    /**
     * メールサーバーとの接続を閉じる
     *
     * @access public
     */
    function quit()
    {
        if (is_object($this->socket)) {
            // 完了メッセージ送信
            $this->socket->send('QUIT');
            // 完了メッセージ取得
            $this->socket->receive();
            // ソケットを閉じる
            $this->socket->close();
        }

        $this->socket = null;
    }

    /**
     * SMTPコマンド受信メッセージ
     *
     * @access private
     * @param string send or receive
     * @param string コマンドメッセージ
     */
    function receiveMessage($type, $message)
    {
        $message = trim($message);
        $this->command_log .= "[{$type}] {$message}\n";
        if ($type == 'receive') {
            if (preg_match('/^[4|5][0-9]{2}/', $message)) {
                $this->error_message = "[SyL error] SMTP command error ($message)";
            }
        }
    }

    /**
     * ソケットコマンド受信ログを取得
     *
     * @access public
     * @return string ソケットコマンド受信ログ
     */
    function getCommandLog()
    {
        return $this->command_log;
    }
}
