<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_Cache.php,v 1.1 2009/01/11 05:34:29 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * キャッシュクラス
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_Cache.php,v 1.1 2009/01/11 05:34:29 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_Cache
{
    /**
     * キャッシュのキー
     *
     * @access protected
     * @var string
     */
    var $key = '';
    /**
     * キャッシュのキーのMD5
     *
     * @access protected
     * @var string
     */
    var $key_md5 = '';
    /**
     * キャッシュの接頭辞
     *
     * @access private
     * @var string
     */
    var $prefix = 'syl_cache.';
    /**
     * キャッシュの有効期間
     * 単位は秒[s]
     *
     * @access protected
     * @var int
     */
    var $life_time = 86400; // 1day
    /**
     * 更新時間の最小値
     * ※キャッシュが古くなっていないかの確認用
     *
     * @access protected
     * @var int
     */
    var $min_mtime = 0; // 0:無効
    /**
     * 更新時間の最大値
     *
     * @access protected
     * @var int
     */
    var $max_mtime = 0; // 0:無効
    /**
     * キャッシュファイルの確認CRCを付加するか
     *
     * @access private
     * @var bool
     */
    var $crc = true;
    /**
     * キャッシュファイルをシリアル化するか
     *
     * @access private
     * @var bool
     */
    var $serial = true;

    /**
     * コンストラクタ
     *
     * @access public
     */
    function SyL_Cache()
    {
    }

    /**
     * キャッシュオブジェクトを取得する
     *
     * @static
     * @access public
     * @param string キャッシュのキー
     * @param string キャッシュタイプ名
     */
    function &factory($type='file')
    {
        $classname = 'SyL_Cache' . ucfirst($type);
        include_once dirname(__FILE__) . "/Cache/{$classname}.php";
        $obj = new $classname();
        return $obj;
    }

    /**
     * キャッシュファイルの接頭辞をセットする
     *
     * @access public
     * @param string キャッシュファイルの接頭辞
     */
    function setPrefix($prefix)
    {
        $this->prefix = $prefix;
    }

    /**
     * キャッシュのキーをセットする
     *
     * @access public
     * @param string キャッシュキー
     */
    function setKey($key)
    {
        $this->key     = $key;
        $this->key_md5 = md5($key);
    }

    /**
     * キャッシュのキーを取得する
     *
     * @access public
     * @param bool MD5付キーフラグ
     * @return string キャッシュキー
     */
    function getKey($md5=true)
    {
        return $this->prefix . ($md5 ? $this->key_md5 : $this->key);
    }

    /**
     * キャッシュ有効時間をセットする
     *
     * @access public
     * @param int キャッシュ有効時間（秒[s]）
     */
    function setLifeTime($life_time)
    {
        $this->life_time = $life_time;
    }

    /**
     * キャッシュ有効時間を永続化（無効に）する
     *
     * @access public
     */
    function persistentLifeTime()
    {
        $this->setMinModifyTime(0);
        $this->setMaxModifyTime(0);
        $this->setLifeTime(0);
    }

    /**
     * 更新時間の最小値をセットする
     *
     * @access public
     * @param int 更新時間の最小値（秒[s]）
     */
    function setMinModifyTime($min_mtime)
    {
        $this->min_mtime = $min_mtime;
    }

    /**
     * 更新時間の最大値をセットする
     *
     * @access public
     * @param int 更新時間の最大値（秒[s]）
     */
    function setMaxModifyTime($max_mtime)
    {
        $this->max_mtime = $max_mtime;
    }

    /**
     * キャッシュの更新時間を更新する
     *
     * @access public
     * @param int 更新時間(UNIX Time)
     * @return bool true: 更新OK、false: 更新エラー
     */
    function setModifyTime($mtime=0)
    {
    }

    /**
     * ガベージコレクションを指定した確率で起動させる
     *
     * @access public
     * @param int 起動確率（0 - 100）
     */
    function gc($per)
    {
        if (is_numeric($per) && (($per >= 0) && ($per <= 100))) {
            if (($per > 0) && ($per >= mt_rand(1, 100))) {
                register_shutdown_function(array(&$this, 'clean'));
            }
        } else {
            trigger_error("[SyL error] Invalid gc method parameter ({$per})", E_USER_ERROR);
        }
    }

    /**
     * キャッシュの更新時間を取得する
     *
     * @access public
     * @return int 更新時間(UNIX Time)
     */
    function getModifyTime()
    {
    }

    /**
     * キャッシュを読み込む
     *
     * @access public
     * @param bool キャッシュ有効期間を更新
     * @param int 最低更新日時（UNIX time）
     * @return mixed キャッシュデータ
     */
    function read($modify_time_update=false)
    {
    }

    /**
     * キャッシュを保存する
     *
     * @access public
     * @param mixed キャッシュデータ
     * @return bool true: 保存成功、false: 保存エラー
     */
    function write($data)
    {
    }

    /**
     * キャッシュを削除する
     * 
     * @access public
     * @return bool true: 削除成功、false: 削除エラー
     */
    function delete()
    {
    }

    /**
     * キャッシュ中の期限切れキャッシュを削除する
     *
     * @access public
     */
    function clean()
    {
    }

    /**
     * キャッシュを全て削除する
     *
     * @access public
     */
    function cleanAll()
    {
    }

    /**
     * CRCを追加判定フラグをセット
     * 
     * @access public
     * @param bool CRCを追加判定フラグ
     */
    function setCrc($flag)
    {
        $this->crc = (bool)$flag;
    }

    /**
     * CRC有効判定
     * 
     * @access public
     * @return bool CRC有効判定
     */
    function isCrc()
    {
        return $this->crc;
    }

    /**
     * CRCを追加したデータを取得
     * 
     * @access public
     * @param string データ
     * @return string ハッシュ値
     */
    function getCrc($data)
    {
        return sprintf('%+032d', crc32($data));
    }

    /**
     * シリアル化判定フラグをセット
     * 
     * @access public
     * @param bool シリアル化判定フラグ
     */
    function setSerialize($flag)
    {
        $this->serial = (bool)$flag;
    }

    /**
     * シリアル化判定
     * 
     * @access public
     * @return bool シリアル化判定
     */
    function isSerialize()
    {
        return $this->serial;
    }
}
