<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_File.php,v 1.1 2009/01/11 05:34:29 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

set_magic_quotes_runtime(0);
clearstatcache();

/**
 * ファイル処理クラス
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_File.php,v 1.1 2009/01/11 05:34:29 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_File
{
    /**
     * ファイル名
     * 
     * @access protected
     * @var string
     */
    var $resource_name;
    /**
     * ファイルポインタ
     * 
     * @access protected
     * @var resource
     */
    var $fp = null;
    /**
     * ファイルオープンモード
     * 
     * @access protected
     * @var string
     */
    var $mode;
    /**
     * パーミッション
     * 
     * @access protected
     * @var int
     */
    var $permission = 0000;
    /**
     * 改行コード
     * 
     * @access protected
     * @var string
     */
    var $eol = "\n";
    /**
     * スクリプト側のエンコーディング
     * 
     * @access protected
     * @var string
     */
    var $encode_script = '';
    /**
     * ファイル側のエンコーディング
     * 
     * @access protected
     * @var string
     */
    var $encode_file = '';
    /**
     * カレントのマスク値
     * 
     * @access public
     * @var int
     */
    var $current_umask = null;
    /**
     * CSVファイルフラグ
     * 
     * @access public
     * @var bool
     */
    var $csv = false;
    /**
     * CSVファイルの区切り文字
     * 
     * @access protected
     * @var string
     */
    var $csv_delimiter = ',';
    /**
     * CSVファイルの囲い子
     * 
     * @access protected
     * @var string
     */
    var $csv_enclosure = '';

    /**
     * コンストラクタ
     *
     * @access public
     * @param string ファイル名
     */
    function SyL_File($resource_name)
    {
        $this->resource_name = $resource_name;
    }

    /**
     * SyL_Fileクラスのインスタンス取得
     *
     * @access public
     * @param string ファイル操作タイプ
     * @param string ファイル名
     */
    function &factory($type, $resource_name)
    {
        switch (strtolower($type)) {
        case 'r': $type = 'Reader';   break;
        case 'w': $type = 'Writer';   break;
        case 'a': $type = 'Appender'; break;
        default:  trigger_error("[SyL error] Invalid type ({$type})", E_USER_ERROR);
        }
        $classname = "SyL_File{$type}";
        include_once dirname(__FILE__) . "/File/{$classname}.php";
        $file =& new $classname($resource_name);
        return $file;
    }

    /**
     * リソース内容を取得する
     *
     * @static
     * @access public
     * @param string リソース名
     * @return string リソース内容
     */
    function readContents($resource_name)
    {
        return file_get_contents($resource_name);
    }

    /**
     * データをファイルに書き込む
     *
     * @static
     * @access public
     * @param string ファイル名
     * @param string データ
     */
    function writeContents($resource_name, $data)
    {
        $file =& SyL_File::factory('w', $resource_name);
        $file->open();
        $file->write($data);
        $file->close();
        $file = null;
    }

    /**
     * データをファイルに追記で書き込む
     *
     * @static
     * @access public
     * @param string ファイル名
     * @param string データ
     */
    function appendContents($resource_name, $data)
    {
        $file =& SyL_File::factory('a', $resource_name);
        $file->open();
        $file->write($data);
        $file->close();
        $file = null;
    }

    /**
     * ファイルを開く
     *
     * @access public
     * @param string オープンモード
     */
    function open()
    {
        if ($this->current_umask !== null) {
            $this->current_umask = umask(0000);
        }

        $this->fp = fopen($this->resource_name, $this->mode);
        if (!is_resource($this->fp)) {
            trigger_error("[SyL error] Function fopen failed. file not found or permission denied? ({$this->resource_name})", E_USER_ERROR);
        }
        if ($this->permission) {
            @chmod($this->resource_name, $this->permission);
            $this->permission = 0000;
        }
        //register_shutdown_function(array(&$this, 'close'));
    }

    /**
     * ファイルを閉じる
     *
     * @access public
     */
    function close()
    {
        if (is_resource($this->fp)) {
            fclose($this->fp);
            $this->fp = null;
        }
        if ($this->permission) {
            @chmod($this->resource_name, $this->permission);
            $this->permission = 0000;
        }
        if ($this->current_umask !== null) {
            umask($this->current_umask);
        }
    }

    /**
     * CSVファイルを使用する
     *
     * @access public
     * @param string 区切り文字
     * @param string 囲い子
     */
    function useCsv($delimiter=',', $enclosure='')
    {
        if (is_resource($this->fp)) {
            trigger_error("[SyL error] `useCsv()' method use before `open()' method", E_USER_ERROR);
        }
        $this->csv = true;
        $this->csv_delimiter = $delimiter;
        $this->csv_enclosure = $enclosure;
    }

    /**
     * ファイルサイズを取得する
     *
     * @access public
     * @return int ファイルサイズ
     */
    function getSize()
    {
        return filesize($this->resource_name);
    }

    /**
     * ファイルに関する情報を取得する
     *
     * @access public
     * @return array ファイルに関する情報
     */
    function getStat()
    {
        if (is_resource($this->fp)) {
            return fstat($this->fp);
        } else if (file_exists($this->resource_name)) {
            return stat($this->resource_name);
        } else {
            return false;
        }
    }

    /**
     * パーミッションをセットする
     *
     * @access public
     * @param int パーミッション(8進数 例: 0755)
     */
    function setPermission($permission)
    {
        $this->permission = $permission;
    }

    /**
     * 使用する改行をセットする
     *
     * @access public
     * @param string 改行
     */
    function setEol($eol)
    {
        $this->eol = $eol;
    }

    /**
     * スクリプト側のエンコーディングをセットする
     *
     * @access public
     * @param string スクリプト側のエンコーディング
     */
    function setScriptEncode($encode_script)
    {
        $this->encode_script = $encode_script;
    }

    /**
     * ファイル側のエンコーディングをセットする
     *
     * @access public
     * @param string ファイル側のエンコーディング
     */
    function setFileEncode($encode_file)
    {
        $this->encode_file = $encode_file;
    }

    /**
     * ファイルバッファを設定する
     *
     * @access public
     * @param int ファイルバッファサイズ
     */
    function setFileBuffer($buffer)
    {
        if (is_resource($this->fp)) {
            stream_set_write_buffer($this->fp, $buffer);
        }
    }

    /**
     * マスク値を使用する
     *
     * @access public
     * @param string マスク値
     */
    function useMask($current_umask=0000)
    {
        $this->current_umask = $current_umask;
    }
}
