<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_Filesystem.php,v 1.1 2009/01/11 05:34:29 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

clearstatcache();

/**
 * ファイルシステムのディレクトリクラス
 */
require_once dirname(__FILE__) . '/Filesystem/SyL_FilesystemDirectory.php';
/**
 * ファイルシステムのファイルクラス
 */
require_once dirname(__FILE__) . '/Filesystem/SyL_FilesystemFile.php';
/**
 * 実行環境クラス
 */
require_once dirname(__FILE__) . '/Util/SyL_UtilEnv.php';

/**
 * ファイルシステムクラス
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_Filesystem.php,v 1.1 2009/01/11 05:34:29 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_Filesystem
{
    /**
     * 要素名
     *
     * @access protected
     * @var string
     */
    var $file = '';
    /**
     * 要素タイプ
     *
     * @access protected
     * @var string
     */
    var $type = '';
    /**
     * 要素のメタデータ
     *
     * @access protected
     * @var array
     */
    var $stat = array();

    /**
     * コンストラクタ
     *
     * @access public
     * @param string 要素名（絶対パス）
     */
    function SyL_Filesystem($file)
    {
        if (preg_match('/^(.+)(\\\\|\/)$/', $file)) {
            $file = substr($file, 0, -1);
        }
        // ファイル名をセット
        $this->file = $file;
        // ファイルタイプ
        $this->type = filetype($file);
        // 要素情報を取得
        $this->stat = stat($file);
    }

    /**
     * 要素オブジェクトを取得する
     *
     * @static
     * @access public
     * @param string 要素名（絶対パス）
     */
    function &factory($file)
    {
        if (!file_exists($file)) {
            trigger_error("[SyL error] File not found ({$file})", E_USER_ERROR);
        }

        $file = realpath($file);

        $element = null;
        if (is_dir($file)) {
           $element =& new SyL_FilesystemDirectory($file);
        } else {
           $element =& new SyL_FilesystemFile($file);
        }
        return $element;
    }

    /**
     * ディレクトリ判定
     *
     * @access abstract
     * @return bool true: ディレクトリ、false: ディレクトリ以外
     */
    function isDir()
    {
    }

    /**
     * 書き込み判定
     *
     * @access public
     * @return bool true: ディレクトリ、false: ディレクトリ以外
     */
    function isWritable()
    {
        return is_writable($this->file);
    }

    /**
     * 要素名を取得する
     *
     * @access public
     * @param bool 要素名のみ取得フラグ
     * @return string 要素名
     */
    function getName($name_only=false)
    {
        return ($name_only) ? basename($this->file) : $this->file;
    }

    /**
     * 要素タイプを取得する
     *
     * @access public
     * @return string 要素タイプ
     */
    function getType()
    {
        return $this->type;
    }

    /**
     * 要素のメタデータを取得する
     *
     * @access public
     * @param string 要素名
     * @return mixed 要素値
     */
    function getStat($name)
    {
        return isset($this->stat[$name]) ? $this->stat[$name] : null;
    }

    /**
     * サイズを取得する
     *
     * @access protected
     * @return int サイズ
     */
    function getSize()
    {
        return isset($this->stat['size']) ? $this->stat['size'] : null;
    }

    /**
     * 最終更新日時を取得する
     *
     * @access public
     * @return string 最終更新日時
     */
    function getMtime()
    {
        return isset($this->stat['mtime']) ? $this->stat['mtime'] : null;
    }

    /**
     * 要素の所有者名を取得する
     *
     * @access public
     * @return string 所有者名
     */
    function getOwner()
    {
        return isset($this->stat['uid']) ? SyL_UtilEnv::getOsUser($this->stat['uid']) : null;
    }

    /**
     * パーミッションを整形する
     *
     * @access public
     * @param int パーミッション
     * @param bool 8進数取得フラグ
     * @return string 整形後のパーミッション
     */
    function getPermission($numeric=true)
    {
        $perm = null;
        if (isset($this->stat['mode'])) {
            $perm = $this->stat['mode'];
        } else {
            return null;
        }

        if ($numeric) {
            return substr(sprintf('%o', $perm), -4);
        } else {
            $info  = '';
            if (($perm & 0xC000) == 0xC000) {
                $info = 's'; // ソケット
            } else if (($perm & 0xA000) == 0xA000) {
                $info = 'l'; // シンボリックリンク
            } else if (($perm & 0x8000) == 0x8000) {
                $info = '-'; // 通常のファイル
            } else if (($perm & 0x6000) == 0x6000) {
                $info = 'b'; // ブロックスペシャルファイル
            } else if (($perm & 0x4000) == 0x4000) {
                $info = 'd'; // ディレクトリ
            } else if (($perm & 0x2000) == 0x2000) {
                $info = 'c'; // キャラクタスペシャルファイル
            } else if (($perm & 0x1000) == 0x1000) {
                $info = 'p'; // FIFO パイプ
            } else {
                $info = 'u'; // 不明
            }

            // 所有者
            $info .= ($perm & 0x0100) ? 'r' : '-';
            $info .= ($perm & 0x0080) ? 'w' : '-';
            $info .= ($perm & 0x0040)
                   ? (($perm & 0x0800) ? 's' : 'x')
                   : (($perm & 0x0800) ? 'S' : '-');
            // グループ
            $info .= ($perm & 0x0020) ? 'r' : '-';
            $info .= ($perm & 0x0010) ? 'w' : '-';
            $info .= ($perm & 0x0008)
                   ? (($perm & 0x0400) ? 's' : 'x')
                   : (($perm & 0x0400) ? 'S' : '-');
            // 全体
            $info .= ($perm & 0x0004) ? 'r' : '-';
            $info .= ($perm & 0x0002) ? 'w' : '-';
            $info .= ($perm & 0x0001)
                   ? (($perm & 0x0200) ? 't' : 'x')
                   : (($perm & 0x0200) ? 'T' : '-');
            return $info;
        }
    }

    /**
     * ファイルが存在するかチェックする
     *
     * @static
     * @access public 
     * @param string ファイル名
     * @return bool true: 存在する、false: 存在しない
     */
    function exists($file)
    {
        if (!file_exists($file)) {
            return false;
        }
        if (is_dir($file)) {
           return SyL_FilesystemDirectory::exists($file);
        } else {
           return SyL_FilesystemFile::exists($file);
        }
    }

    /**
     * ファイルを削除する
     *
     * @static
     * @access public 
     * @param string ファイル名
     * @return bool true: 削除OK、false: 削除エラー
     */
    function remove($file)
    {
        if (!file_exists($file)) {
            return false;
        }

        if (is_dir($file)) {
           return SyL_FilesystemDirectory::remove($file);
        } else {
           return SyL_FilesystemFile::remove($file);
        }
    }

    /**
     * ディレクトリを作成する
     *
     * @static
     * @access public
     * @param string ディレクトリ名（相対パス）
     * @param int モード
     */
    function createDirectory($dir, $mode=0777)
    {
        SyL_FilesystemDirectory::createDirectory($dir, $mode=0777);
    }

    /**
     * ファイルをコピーする
     *
     * @static
     * @access public
     * @param string 現ファイル名
     * @param string 新ファイル名
     * @return bool true: 変更OK、false: 変更エラー
     */
    function copy($current_file, $new_file)
    {
        return copy($current_file, $new_file);
    }

    /**
     * 要素名を変更する
     *
     * @static
     * @access public
     * @param string 現ファイル名
     * @param string 新ファイル名
     * @return bool true: 変更OK、false: 変更エラー
     */
    function rename($current_file, $new_file)
    {
        return rename($current_file, $new_file);
    }

    /**
     * パーミッションを変更する
     *
     * @static
     * @access public
     * @param int パーミッション（8進 例: 0755）
     * @return bool true: 変更成功、false: エラー
     */
    function changePermission($file, $perm)
    {
        return chmod($file, $perm);
    }

    /**
     * ディレクトリ内ファイルを取得する
     *
     * @static
     * @access public
     * @param string ディレクトリ
     * @param bool true: ディレクトリ取得、false: ディレクトリ取得しない
     * @param bool true: ファイル取得、false: ファイル取得しない
     * @param int 検索を行う最大階層
     * @return array ディレクトリ内ファイル
     */
    function getFiles($file, $is_get_dir=true, $is_get_file=true, $limit_depth=1)
    {
        return SyL_FilesystemDirectory::getFiles($file, $is_get_dir, $is_get_file, $limit_depth);
    }

    /**
     * ファイルを検索する
     *
     * @static
     * @access public
     * @param string ディレクトリ
     * @param string 検索ファイル名
     * @param bool true: ディレクトリ取得、false: ディレクトリ取得しない
     * @param bool true: ファイル取得、false: ファイル取得しない
     * @param int 検索を行う最大階層
     * @return array ディレクトリ内ファイル
     */
    function searchFiles($file, $word, $is_get_dir=true, $is_get_file=true, $limit_depth=1)
    {
        return SyL_FilesystemDirectory::searchFiles($file, $word, $is_get_dir, $is_get_file, $limit_depth);
    }

    /**
     * サイズ（ディレクトリ内の全ファイルサイズ）を取得する
     *
     * @static
     * @access public
     * @param string ディレクトリ
     * @param bool true: ディレクトリのサイズ含む、fase: ディレクトリのサイズ含まない
     * @return int サイズ
     */
    function getTotalSize($file, $is_dir_size=false)
    {
        if (!file_exists($file)) {
            return 0;
        }
        if (is_dir($file)) {
           return SyL_FilesystemDirectory::getTotalSize($file, $is_dir_size);
        } else {
           return filesize($file);
        }
    }

    /**
     * 指定ディレクトリ（の存在するパーテーション）に対する使用可能容量を取得する
     *
     * @static
     * @access public
     * @param string ディレクトリ
     * @return int 使用可能容量
     */
    function getFreeSpace($dir)
    {
        return disk_free_space($dir);
    }

    /**
     * 指定ディレクトリ（の存在するパーテーション）に対する使用済み容量を取得する
     *
     * @static
     * @access public
     * @param string ディレクトリ
     * @return int 使用済み容量
     */
    function getUsedSpace($dir)
    {
        $class = __CLASS__;
        $free  = call_user_func(array($class, 'getFreeSpace'),  $dir);
        $total = call_user_func(array($class, 'getTotalSpace'), $dir);
        return ($total - $free);
    }

    /**
     * 指定ディレクトリ（の存在するパーテーション）に対する総容量を取得する
     *
     * @static
     * @access public
     * @param string ディレクトリ
     * @return int 使用済み容量
     */
    function getTotalSpace($dir)
    {
        return disk_total_space($dir);
    }

    /**
     * 容量表示を整形する
     *
     * @static
     * @access public
     * @param int 容量
     * @param int 少数の桁数
     * @return string 整形後容量
     */
    function formatSize($size, $suffix='KB', $number=0)
    {
        switch (strtoupper($suffix)) {
        case 'TB': $size /= 1099511627776; break;
        case 'GB': $size /= 1073741824; break;
        case 'MB': $size /= 1048576; break;
        case 'KB': $size /= 1024; break;
        }
        $size = number_format(round($size, $number), $number);
        $sizes = explode('.', $size, 2);
        if (isset($sizes[1])) {
            for ($i=$number-1; $i>=0; $i--) {
                $tmp = substr($sizes[1], $i, 1);
                if (!$tmp) {
                    $sizes[1] = substr($sizes[1], 0, $i);
                }
            }
            $size = $sizes[1] ? "{$sizes[0]}.{$sizes[1]}" : $sizes[0];
        }
        return $size;
    }

    /**
     * 容量表示を見やすいように整形する
     *
     * @static
     * @access public
     * @param int 容量
     * @param int 少数の桁数
     * @return string 整形後容量
     */
    function formatVisualSize($size, $number=0)
    {
        $suffix = 'B';
        if ($size > 1099511627776) {
            $size /= 1099511627776;
            $suffix = 'TB';
        } else if ($size > 1073741824) {
            $size /= 1073741824;
            $suffix = 'GB';
        } else if ($size > 1048576) {
            $size /= 1048576;
            $suffix = 'MB';
        } else if ($size > 1024) {
            $size /= 1024;
            $suffix = 'KB';
        }

        $size = number_format(round($size, $number), $number);
        $sizes = explode('.', $size, 2);
        if (isset($sizes[1])) {
            for ($i=$number-1; $i>=0; $i--) {
                $tmp = substr($sizes[1], $i, 1);
                if (!$tmp) {
                    $sizes[1] = substr($sizes[1], 0, $i);
                }
            }
            $size = $sizes[1] ? "{$sizes[0]}.{$sizes[1]}" : $sizes[0];
        }
        return $size . $suffix;
    } 
}
