<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_AdmFlowLst.php,v 1.2 2009/02/14 04:32:56 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * 管理フローの基底クラス
 */
require_once SYL_FRAMEWORK_DIR . '/core/Adm/SyL_AdmFlow.php';

/**
 * 管理フロー一覧クラス
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_AdmFlowLst.php,v 1.2 2009/02/14 04:32:56 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_AdmFlowLst extends SyL_AdmFlow
{
    /**
     * オペレーション名
     *
     * @access protected
     * @var string
     */
    var $opp_name = 'lst';

    /**
     * コンストラクタ
     *
     * @access public
     */
    function SyL_AdmFlowLst()
    {
        parent::SyL_AdmFlow();
    }

    /**
     * 一覧表示フロー
     *
     * @access public
     * @param object データ管理オブジェクト
     * @param object コンテキスト管理オブジェクト
     */
    function execute(&$data, &$context)
    {
        // パラメータ取得
        $page  = $data->get('pg');
        $row   = $data->get('rw');
        $sort  = $data->get('st');

        // 表示件数判定
        if (!is_numeric($row) || ($row < 1)) {
            $row = '';
        }

        // フォーム取得
        $form =& $this->getForm();
        // データ取得オブジェクトセット
        $form->registerInput($data);
        // 検索フォーム構築
        $form->buildSearch();

        // DB接続
        $conn =& $context->getDB();

        // ADMオブジェクト作成
        $adm =& SyL_Adm::factory($form, $conn);
        // テーブル情報ロード
        $adm->loadConfig();
        // アクセス権チェック
        if (!$adm->isLst()) {
            // アクセス権限なし
            SyL_Error::trigger("[Adm error] List Opperation Forbidden");
        }

        // 表示データ取得
        $results = $adm->getList($sort, $page, $row);
        // カレントの検索パラメータ取得
        $search_parameter = implode('&', $adm->getSearchParameters());
        // ヘッダ表示項目取得
        $headers = $adm->getSelectHeaders();
        // カレントのソート項目取得
        $current_sort = $adm->getSortColumns();

        // ページオブジェクト取得
        $pager =& $adm->getPager();
        // リンクページ範囲取得
        $links        = $adm->getPageLinks($pager, $form->getStructConfig('link_range'));
        $current_page = $pager->getCurrentPage();
        $max_page     = $pager->getTotalPage();

        // デフォルト検索表示
        $search_view = $adm->isDefaultSearchView();

        // DB接続終了
        $context->closeDB($conn);

        // プライマリキー取得
        $primaries    = $adm->getPrimary(false);
        $primary_name = '';
        if (count($primaries) > 0) {
            $primary_name = $primaries[0];
        }
        // プライマリーリンクパラメータ取得
        $primary_link_parameters = $adm->getPrimaryLinkParameters($results);

        // 検索クエリリンク取得
        if ($search_parameter != '') {
            $search_parameter = '&' . $search_parameter;
            $search_view  = true;
        }

        // 検索フォーム要素データ関連
        $data->set('form', $form->getResultArray());
        $data->set('search_view', $search_view);

        // 一覧表示データ関連
        $data->set('headers', $headers);
        $data->set('results', $results);

        // 主キーパラメータ
        $data->set('key_name',     $form->getStructConfig('key_name'));
        $data->set('primary_name', $primary_name);
        $data->set('primary_link_parameters', $primary_link_parameters);

        // ページリンク関連パラメータ
        $data->set('links',        $links);
        $data->set('max_page',     $max_page);
        $data->set('current_page', $current_page);
        $data->set('current_sort', $current_sort);
        $data->set('search_parameter', $search_parameter);

       // 確認表示関連
        $data->set('view_alert', $form->getStructConfig('view_alert'));

        // リンク先URL
        $data->set('url_mnu', $this->getMenuLink());
        $data->set('url_lst', $this->getUrl($context, 'lst'));
        $data->set('url_new', $this->getUrl($context, 'new'));
        $data->set('url_upd', $this->getUrl($context, 'upd'));
        $data->set('url_vew', $this->getUrl($context, 'vew'));
        $data->set('url_del', $this->getUrl($context, 'del'));
        $data->set('url_dir', dirname($context->getScriptName()));

        // 文言関連
        $data->set('title',   $form->getStructConfig('title'));
        $data->set('summary', $form->getStructConfig('summary'));

        // 新規登録可能フラグ
        $data->set('enable_new', $adm->isNew());
        // 編集可能フラグ
        $data->set('enable_upd', $adm->isUpd());
        // 削除可能フラグ
        $data->set('enable_del', $adm->isDel());
        // 照会可能フラグ
        $data->set('enable_vew', $adm->isVew());
    }

    /**
     * Ajaxライブラリ出力用アクション
     *
     * @access public
     * @param object データ管理オブジェクト
     * @param object コンテキスト管理オブジェクト
     */
    function executeImport(&$data, &$context)
    {
        $data->addImport('ajax');
        $router =& $context->getRouter();
        $router->setViewType('js.default');
    }

    /**
     * JSON出力用アクション
     *
     * @access public
     * @param object データ管理オブジェクト
     * @param object コンテキスト管理オブジェクト
     */
    function executeJson(&$data, &$context)
    {
        $data->apply('mb_convert_encoding', SYL_ENCODE_INTERNAL, SYL_ENCODE_JS);

        $this->execute($data, $context);

        $data->apply('mb_convert_encoding', SYL_ENCODE_JS, SYL_ENCODE_INTERNAL);
        $router =& $context->getRouter();
        $router->setViewType('json');
    }

    /**
     * メニューリンク先取得
     *
     * @access public
     * @return string メニューリンク先
     */
    function getMenuLink()
    {
        // adms.xml があればメニューへのリンク取得
        $adm_config =& SyL_Config::factory('adms');
        $adm_config->parseXml();
        $config = $adm_config->getConfig();

        $menu_link = null;
        if (isset($config['adm'])) {
            foreach ($config['adm'] as $adm) {
                if ($adm['class'] == $this->classname) {
                    $menu_link = $config['base'];
                    break;
                }
            }
        }
        return $menu_link;
    }
}
