<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_CacheFile.php,v 1.1 2009/01/11 05:34:31 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * ファイルキャッシュクラス
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_CacheFile.php,v 1.1 2009/01/11 05:34:31 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_CacheFile extends SyL_Cache
{
    /**
     * キャッシュディレクトリ
     *
     * @access private
     * @var string
     */
    var $cache_dir = '/tmp/';
    /**
     * インクルードキャッシュ使用フラグ
     *
     * @access private
     * @var bool
     */
    var $include_cache = false;
    /**
     * 削除中のロックファイル名
     *
     * @access private
     * @var bool
     */
    var $lockname = '.locking';

    /**
     * キャッシュディレクトリをセットする
     *
     * @access public
     * @param string キャッシュディレクトリ
     */
    function setCacheDir($cache_dir)
    {
        if (is_dir($cache_dir)) {
            if (!is_writable($cache_dir)) {
                trigger_error("[SyL error] Cache directory permission denied ({$cache_dir})", E_USER_ERROR);
            }
        } else {
            if (!mkdir($cache_dir)) {
                trigger_error("[SyL error] Cache directory can't create ({$cache_dir})", E_USER_ERROR);
            }
            chmod($cache_dir, 0777);
        }
        if (!preg_match('/(\\\\|\/)$/', $cache_dir)) {
            $cache_dir .= '/';
        }
        $this->cache_dir = $cache_dir;
    }

    /**
     * インクルードキャッシュを使用する
     *
     * @access public
     * @param bool シリアライズフラグ
     */
    function useIncludeCache($serialize=true)
    {
        $this->include_cache = true;
        $this->setCrc(false);
        $this->setSerialize($serialize);
    }

    /**
     * キャッシュの更新時間を更新する
     *
     * @access public
     * @param int 更新時間(UNIX Time)
     * @return bool true: 更新OK、false: 更新エラー
     */
    function setModifyTime($mtime=0)
    {
        $filename = $this->getFileName();
        if (!file_exists($filename)) return false;
        return ($mtime) ? touch($filename, $mtime) : touch($filename);
    }

    /**
     * キャッシュファイルの更新時間を取得する
     *
     * @access public
     * @return int 更新時間(UNIX Time)
     */
    function getModifyTime()
    {
        $filename = $this->getFileName();
        if (!file_exists($filename)) return false;
        return filemtime($filename);
    }

    /**
     * キャッシュを読み込む
     *
     * @access public
     * @param bool キャッシュ有効期間を更新
     * @return mixed キャッシュデータ
     */
    function read($modify_time_update=false)
    {
        $filename = $this->getFileName();
        if (file_exists($filename)) {
            // キャッシュファイルOK
            $mtime = filemtime($filename);

            // 更新時間の最小値判定
            if ($this->min_mtime > 0) {
                if ($mtime < $this->min_mtime) {
                    return false;
                }
            }
            // 更新時間の最大値判定
            if ($this->max_mtime > 0) {
                if ($mtime > $this->max_mtime) {
                    return false;
                }
            }

            // 有効時間を取得
            $life_time = ($this->life_time > 0) ? $mtime + $this->life_time : null;

            if (($life_time === null) || ($life_time >= time())) {
                if ($this->include_cache) {
                    $data = include $filename;
                } else {
                    $mqr = get_magic_quotes_runtime();
                    set_magic_quotes_runtime(0);

                    $fp = fopen($filename, 'rb');
                    if (!$fp) return false;

                    $i = 0;
                    while (!flock($fp, LOCK_SH)) {
                        if ($i < 5) {
                            // 5回トライ
                            usleep(500000); // 0.5s
                            $i++;
                        } else {
                            fclose($fp);
                            return false;
                        }
                    }
                    $hash = null;
                    $size = filesize($filename);
                    if ($this->isCrc()) {
                        $hash = fread($fp, 32);
                        $data = ($size > 32) ? fread($fp, $size-32) : '';
                    } else {
                        $data = fread($fp, $size);
                    }
                    fclose($fp);

                    set_magic_quotes_runtime($mqr);

                    if (($hash === null) || ($hash == $this->getCrc($data))) {
                        if ($this->isSerialize()) {
                            $data = unserialize($data);
                        }
                    } else {
                        $this->delete();
                        return false;
                    }
                }

                if ($modify_time_update) {
                    $this->setModifyTime();
                }
                return $data;
            } else {
                // キャッシュ有効期間切れ
                $this->delete();
                return false;
            }
        } else {
            // キャッシュファイル無し
            return false;
        }
    }

    /**
     * キャッシュを保存する
     *
     * @access public
     * @param mixed キャッシュデータ
     * @return bool true: 保存成功、false: 保存エラー
     */
    function write($data)
    {
        if ($this->include_cache) {
            if ($this->isSerialize()) {
                $data = var_export($data, true);
                $data = "return {$data};";
            }
            if (substr($data, 0, 2) != '<?') {
                $data = "<?php\n" . $data;
            }
            if (substr($data, -2) != '?>') {
                $data .= "\n?>\n";
            }
        } else {
            if ($this->isSerialize()) {
                $data = serialize($data);
            }
        }
        // データのバイト数取得
        $size = strlen($data);

        // キャッシュファイル名取得
        $filename = $this->getFileName();
        $fp = fopen($filename, 'wb');
        if (!$fp) return false;

        $i = 0;
        while (!flock($fp, LOCK_EX)) {
            if ($i < 5) {
                // 5回トライ
                usleep(500000); // 0.5s
                $i++;
            } else {
                fclose($fp);
                return false;
            }
        }

        if ($this->isCrc()) {
            fwrite($fp, $this->getCrc($data), 32);
        }

        fwrite($fp, $data, $size);
        fclose($fp);

        @chmod($filename, 0777);
        return true;
    }

    /**
     * キャッシュを削除する
     * 
     * @access public
     * @return bool true: 削除成功、false: 削除エラー
     */
    function delete()
    {
        $filename = $this->getFileName();
        return file_exists($filename) ? @unlink($filename) : false;
    }

    /**
     * ディレクトリ内のキャッシュファイルを整理する
     *
     * @access private
     */
    function clean()
    {
        if (!is_dir($this->cache_dir)) {
            trigger_error("[SyL error] Cache directory property not set ({$this->cache_dir})", E_USER_ERROR);
        }
        if ($this->life_time == 0) {
            return;
        }

        $lockfile = "{$this->cache_dir}/__{$this->prefix}{$this->lockname}";
        if (is_file($lockfile)) {
            // ロックファイルがある場合は、削除処理を行わない
            return;
        }

        $fp = @fopen($lockfile, 'wb');
        if (!$fp) return;
        if (!@flock($fp, LOCK_EX)) {
            fclose($fp);
            return;
        }

        $now = time();

        $dh = opendir($this->cache_dir);
        while (($filename = readdir($dh)) !== false) {
            $filename = basename($filename);
            if (is_file("{$this->cache_dir}/{$filename}")) {
                if ($this->prefix) {
                    if (!preg_match('/^(' . preg_quote($this->prefix, '/') . ')/', $filename)) {
                        continue;
                    }
                }
                $filename = "{$this->cache_dir}/{$filename}";
                $mtime = filemtime($filename);
                // 更新時間の最小値判定
                if ($this->min_mtime && ($mtime < $this->min_mtime)) {
                    @unlink($filename);
                // 更新時間の最大値判定
                } else if ($this->max_mtime && ($mtime > $this->max_mtime)) {
                    @unlink($filename);
                // 生存時間の判定
                } else if ($now > ($mtime + $this->life_time)) {
                    @unlink($filename);
                }
            }
        }
        closedir($dh);

        fclose($fp);
        @unlink($lockfile);
    }

    /**
     * キャッシュを全て削除する
     *
     * @access public
     */
    function cleanAll()
    {
        if (!is_dir($this->cache_dir)) {
            trigger_error("[SyL error] Cache directory property not set ({$this->cache_dir})", E_USER_ERROR);
        }

        $lockfile = "{$this->cache_dir}/__{$this->prefix}{$this->lockname}";
        if (is_file($lockfile)) {
            // ロックファイルがある場合は、削除処理を行わない
            return;
        }

        $fp = @fopen($lockfile, 'w');
        if (!$fp) return;
        if (!@flock($fp, LOCK_EX)) {
            fclose($fp);
            return;
        }

        $dh = opendir($this->cache_dir);
        while (($filename = readdir($dh)) !== false) {
            if (is_file($this->cache_dir . $filename)) {
                @unlink($this->cache_dir . $filename);
            }
        }
        closedir($dh);

        fclose($fp);
        @unlink($lockfile);
    }

    /**
     * キャッシュファイル名を取得する
     * 
     * @access public
     * @param string キャッシュファイル判定キー
     * @return string キャッシュファイル名
     */
    function getFileName()
    {
        return $this->cache_dir . $this->getKey();
    }
}
