<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_DBPdo.php,v 1.1 2009/01/11 05:34:33 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 *  DB操作クラス（PDO）
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_DBPdo.php,v 1.1 2009/01/11 05:34:33 seasonstream Exp $
 * @link      http://syl.jp/
 */
abstract class SyL_DBPdo extends SyL_DB
{
    /**
     * PDO DB接続実行
     * 
     * @access public
     * @param string 接続文字列
     * @param string ユーザー名
     * @param string パスワード
     * @return boolean 接続OK: true, 接続NG: false
     */
    protected function pdoConnection($conn_string, $conn_user=null, $conn_passwd=null)
    {
        try {
            if (($conn_user !== null) && ($conn_passwd !== null)) {
                $this->connection = new PDO($conn_string, $conn_user, $conn_passwd);
            } else if ($conn_user !== null) {
                $this->connection = new PDO($conn_string, $conn_user);
            } else {
                $this->connection = new PDO($conn_string);
            }
            switch ($this->field_case) {
            case CASE_UPPER: $this->connection->setAttribute(PDO::ATTR_CASE, PDO::CASE_UPPER); break;
            case CASE_LOWER: $this->connection->setAttribute(PDO::ATTR_CASE, PDO::CASE_LOWER); break;
            }
        } catch (PDOException $e) {
            $this->setErrorMessage($e->getCode(), $e->getMessage());
            $this->triggerError();
        }

        return is_object($this->connection);
    }

    /**
     * DB接続終了
     * 
     * @access public
     */
    public function closeConnection()
    {
        $this->connection = null;
    }

    /**
     * パラメータサニタイズ（無効化）処理
     * 
     * @access public
     * @param string サニタイズ対象文字列
     * @return string サニタイズ後文字列
     */
    public function quote($parameter)
    {
        return $this->connection->quote($parameter);
    }

    /**
     * トランザクション開始
     *
     * @access public
     */
    public function beginTransaction()
    {
        $result = true;
        parent::beginTransaction();
        try {
            $this->connection->beginTransaction();
        } catch (PDOException $e) {
            $this->setErrorMessage($e->getCode(), $e->getMessage());
            $result = false;
        }
        return $result;
    }

    /**
     * トランザクション破棄
     *
     * @access public
     */
    public function rollBack()
    {
        $result = true;
        try {
            $this->connection->rollBack();
        } catch (PDOException $e) {
            $result = false;
        }
        parent::rollBack();
        return $result;
    }

    /**
     * トランザクション確定
     *
     * @access public
     */
    public function commit()
    {
        $result = true;
        try {
            $this->connection->commit();
        } catch (PDOException $e) {
            $result = false;
        }
        parent::commit();
        return $result;
    }

    /**
     * 最後に起こったエラーメッセージをセット
     *
     * @access public
     * @param string エラーコード
     * @param string エラーメッセージ
     */
    function setErrorMessage($code='', $message='')
    {
        if ($message) {
            parent::setErrorMessage($code, $message);
        } else {
            if (method_exists($this->connection, 'errorInfo')) {
                $error_info = $this->connection->errorInfo();
                if (isset($error_info[1])) {
                    $this->last_error_code    = $error_info[1] . "(" . $error_info[0] . ")";
                    $this->last_error_message = $error_info[2];
                } else {
                    $this->last_error_code    = $error_info[0];
                }
            }
        }
    }

    /**
     * SQLを実行し、DBリソースを取得
     *
     * @access public
     * @param string SQL文
     * @return mixed 実行OK: DBリソース, 実行NG: false
     */
    public function execSelect($sql)
    {
        try {
            $stmt = $this->connection->query($sql);
            if ($stmt === false) {
                $this->setErrorMessage();
                $this->triggerError();
            }
            return $stmt;
        } catch (PDOException $e) {
            $this->setErrorMessage($e->getCode(), $e->getMessage());
            $this->triggerError();
            return false;
        }
    }

    /**
     * SQLを実行し、実行結果影響件数を取得
     *
     * @access public
     * @param string SQL文
     * @return mixed 実行OK: 実行結果影響件数, 実行NG: false
     */
    public function execUpdate($sql)
    {
        try {
            $result = $this->connection->exec($sql);
            if ($result === false) {
                $this->setErrorMessage();
                $this->triggerError();
            }
            return $result;
        } catch (PDOException $e) {
            $this->setErrorMessage($e->getCode(), $e->getMessage());
            $this->triggerError();
            return false;
        }
    }

    /**
     * SQL実行のみ
     *
     * @access public
     * @param string SQL文
     * @return boolean 実行OK: true, 実行NG: false
     */
    public function execNoReturn($sql)
    {
        return $this->execUpdate($sql);
    }
  
    /**
     * SQL実行し、結果データを取得
     *
     * ・第3パラメータが、'one'の場合
     *    最初の1カラムを文字列として取得
     * ・第3パラメータが、'record'の場合
     *    最初の1レコードを配列として取得
     * ・第3パラメータが、'all'の場合
     *    全レコードを配列として取得
     *
     * @access public
     * @param string SQL文
     * @param mixed  実行結果を取得
     * @param string 結果取得フラグ
     * @return boolean 実行OK: true, 実行NG: false
     */
    public function queryRef($sql, &$data, $get='all')
    {
        // SQL実行
        $this->execRef($sql, $stmt);

        // 取得判定
        if (!is_object($stmt)) {
            $data = $stmt;
            return ($stmt !== false);
        }

        switch ($get) {
        // 1フィールド
        case 'one':
            $row = $stmt->fetch(PDO::FETCH_NUM);
            $data = $row ? $row[0] : '';
            break;
        // 1レコード
        case 'record':
            $data = $stmt->fetch(PDO::FETCH_ASSOC);
            if ($data === false) {
                $data = array();
            }
            break;
        // 全レコード
        case 'all':
            $data = $stmt->fetchAll(PDO::FETCH_ASSOC);
            break;
        }
        return $stmt->closeCursor();
    }

    /**
     * 最後に挿入された行の ID あるいはシーケンスの値を取得
     *
     * @access public
     * @return int 最後に挿入された行のID
     */
    public function lastInsertId($seq=null)
    {
        if ($seq === null) {
            return $this->connection->lastInsertId();
        } else {
            return $this->connection->lastInsertId($seq);
        }
    }
}
