<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_ValidationConfig.php,v 1.1 2009/01/11 05:34:33 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * 検証設定クラス
 * 
 * array(
 *   'def1' => array(
 *     [0] => array(
 *       'validator'    => 'require',
 *       'errorMessage' => '{name}は必須です',
 *       'options' => array(
 *         'max'   => '19',
 *         'min'   => '1'
 *       )
 *     ),
 *     [1] => ...
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_ValidationConfig.php,v 1.1 2009/01/11 05:34:33 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_ValidationConfig
{
    /**
     * ファイル名
     *
     * @access protected
     * @var string
     */
    var $filename = '';
    /**
     * 検証設定配列
     *
     * @access protected
     * @var array
     */
    var $config = array();
    /**
     * 検証設定配列（名前）
     *
     * @access protected
     * @var array
     */
    var $config_name = array();

    /**
     * コンストラクタ
     *
     * @access public
     * @param string ファイル名
     */
    function SyL_ValidationConfig($filename)
    {
        $this->filename = $filename;
    }

    /**
     * 設定オブジェクトを取得する
     *
     * @static
     * @access public
     * @param string 設定ファイル名
     * @param string 設定ファイルの拡張子
     * @return object 設定オブジェクト
     */
    function &getConfig($filename, $ext='')
    {
        if (!$ext) {
            $ext = pathinfo($filename, PATHINFO_EXTENSION);
        }

        if (!file_exists($filename)) {
            trigger_error("[SyL error] Config file not found ({$filename})", E_USER_ERROR);
        }

        if (!is_readable($filename)) {
            trigger_error("[SyL error] Config file not readable ({$filename})", E_USER_ERROR);
        }

        $classdir  = dirname(__FILE__) . '/Config/';
        $classname = 'SyL_ValidationConfig' . ucfirst($ext);
        if (!file_exists($classdir . $classname . '.php')) {
            trigger_error("[SyL error] Not Supported extension ({$ext})", E_USER_ERROR);
        }

        include_once $classdir . $classname . '.php';
        $obj =& new $classname($filename);
        $obj->parse();
        return $obj;
    }

    /**
     * 検証設定パース
     *
     * @access protected
     */
    function parse()
    {
    }

    /**
     * 検証設定値を追加する
     *
     * @access public
     * @param string 要素名
     * @param string バリデーション名
     * @param string エラーメッセージ
     * @param array 検証オプション
     * @param string 表示名
     */
    function add($name, $validation_name, $error_message, $options=array(), $display_name='')
    {
        if (!isset($this->config[$name])) {
            $this->config[$name] = array();
            $this->config_name[$name] = ($display_name !== '') ? $display_name : $name;
        }

        $i = count($this->config[$name]);
        $this->config[$name][$i] = array(
          'validator'    => $validation_name,
          'errorMessage' => $error_message,
          'options'      => ($options ? $options : array())
        );
    }

    /**
     * 設定値から検証グループオブジェクトを取得する
     *
     * @access public
     * @param string 設定キー値
     * @return object 検証グループオブジェクト
     */
    function &get($pattern)
    {
        if (!isset($this->config[$pattern])) {
            trigger_error("[SyL error] Pattern not found ({$pattern})", E_USER_ERROR);
        }
        $group =& SyL_Validators::create();
        foreach ($this->config[$pattern] as $config) {
            $options = isset($config['options']) ? $config['options'] : array();
            $group->add(SyL_Validator::create($config['validator'], $config['errorMessage'], $options));
        }
        return $group;
    }

    /**
     * 設定値から検証グループ名を取得する
     *
     * @access public
     * @param string 設定キー値
     * @return string 検証グループ名
     */
    function getName($pattern)
    {
        if (!isset($this->config_name[$pattern])) {
            trigger_error("[SyL error] Display name not found ({$pattern})", E_USER_ERROR);
        }
        if ($this->config_name[$pattern] !== '') {
            $pattern = $this->config_name[$pattern];
        }
        return $pattern;
    }

    /**
     * 設定値から全キー値を取得する
     *
     * @access public
     * @return array 設定キー値配列
     */
    function getKeys()
    {
        return array_keys($this->config);
    }

}
