package jp.sourceforge.talisman.hermes;

/*
 * $Id: HermesContextParserTest.java 198 2009-05-30 14:45:26Z tama3 $
 */

import java.io.ByteArrayInputStream;
import java.io.File;
import java.net.MalformedURLException;
import java.util.Iterator;

import jp.sourceforge.talisman.hermes.maven.DependencyScope;

import org.junit.Assert;
import org.junit.Test;

/**
 * 
 * @author Haruaki Tamada
 * @version $Revision: 198 $
 */
public class HermesContextParserTest{
    @Test
    public void testBasic() throws Exception{
        HermesContextParser parser = new HermesContextParser();
        HermesContext context = parser.parse(getClass().getResource("/hermes/hermes.xml"));

        Assert.assertNull(context.getDestination());
        Assert.assertTrue(context.isInclude(DependencyScope.COMPILE));
        Assert.assertTrue(context.isInclude(DependencyScope.RUNTIME));
        Assert.assertFalse(context.isInclude(DependencyScope.SYSTEM));
        Assert.assertFalse(context.isInclude(DependencyScope.TEST));
        Assert.assertFalse(context.isInclude(DependencyScope.PROVIDED));

        Assert.assertEquals(context.getPolicy(), UpdatingLibraryCheckPolicy.DESTINATION_CHECK);

        Assert.assertEquals(context.getRepositoryScope(), RepositoryScope.BOTH);
        Assert.assertEquals(context.getRepositoryCount(), 4);

        context.setRepositoryScope(RepositoryScope.LOCAL_ONLY);
        Assert.assertEquals(context.getRepositoryScope(), RepositoryScope.LOCAL_ONLY);
        Assert.assertEquals(context.getScopedRepositories().length, 1);

        context.setRepositoryScope(RepositoryScope.REMOTE_ONLY);
        Assert.assertEquals(context.getRepositoryScope(), RepositoryScope.REMOTE_ONLY);
        Assert.assertEquals(context.getScopedRepositories().length, 3);

        context.setDestination(System.getProperty("user.dir"));
        Assert.assertEquals(context.getDestination(), System.getProperty("user.dir"));
    }

    @Test
    public void testBasic2() throws Exception{
        HermesContextParser parser = new HermesContextParser();
        HermesContext context = parser.parse(new File("target/test-classes/hermes/hermes.xml"));

        Assert.assertFalse(parser.hasExceptions());

        Assert.assertEquals(context.getRepositoryScope(), RepositoryScope.BOTH);
        Assert.assertEquals(context.getRepositoryCount(), 4);

        context.setRepositoryScope(RepositoryScope.LOCAL_ONLY);
        Assert.assertEquals(context.getScopedRepositories().length, 1);

        context.setRepositoryScope(RepositoryScope.REMOTE_ONLY);
        Assert.assertEquals(context.getScopedRepositories().length, 3);
    }

    @Test
    public void testMalformedURLException() throws Exception{
        HermesContextParser parser = new HermesContextParser();
        HermesContext context = parser.parse(getClass().getResource("/hermes/invalid_hermes.xml"));

        Assert.assertTrue(parser.hasExceptions());
        Assert.assertEquals(parser.getExceptionCount(), 1);
        Iterator<Exception> iterator = parser.iterator();
        Assert.assertEquals(iterator.next().getClass(), MalformedURLException.class);

        Assert.assertEquals(context.getRepositoryScope(), RepositoryScope.BOTH);
        Assert.assertEquals(context.getRepositoryCount(), 4);

        context.setRepositoryScope(RepositoryScope.LOCAL_ONLY);
        Assert.assertEquals(context.getScopedRepositories().length, 1);

        context.setRepositoryScope(RepositoryScope.REMOTE_ONLY);
        Assert.assertEquals(context.getScopedRepositories().length, 3);
    }

    @Test(expected=InvalidHermesConfigException.class)
    public void testUnknownDependencyScopeInDependencyTag() throws Exception{
        String xmlString = "<hermes>" + 
        "<dependencies><dependency>" + 
        "  <groupId>jp.sourceforge</groupId>" + 
        "  <artifactId>stigmata</artifactId>" + 
        "  <version>2.0.0</version>" + 
        "  <scope>unknown</scope>" + 
        "</dependency></dependencies></hermes>";
        HermesContextParser parser = new HermesContextParser();
        parser.parse(new ByteArrayInputStream(xmlString.getBytes()));
    }

    @Test(expected=InvalidHermesConfigException.class)
    public void testUnknownDependencyScope() throws Exception{
        String xmlString = "<hermes>" + 
        "<dependency-scope>compile, unknown</dependency-scope>" +
        "<dependencies><dependency>" + 
        "  <groupId>jp.sourceforge</groupId>" + 
        "  <artifactId>stigmata</artifactId>" + 
        "  <version>2.0.0</version>" + 
        "  <scope>compile</scope>" + 
        "</dependency></dependencies></hermes>";
        HermesContextParser parser = new HermesContextParser();
        parser.parse(new ByteArrayInputStream(xmlString.getBytes()));
    }

    @Test
    public void testDestination() throws Exception{
        String xmlString = "<hermes>" +
        "<destination>unknown</destination>" + 
        "<dependency-scope>compile, runtime</dependency-scope>" +
        "<updating-policy>destination_check</updating-policy>" +
        "<dependencies><dependency>" + 
        "  <groupId>jp.sourceforge</groupId>" + 
        "  <artifactId>stigmata</artifactId>" + 
        "  <version>2.0.0</version>" + 
        "  <scope>compile</scope>" + 
        "</dependency></dependencies></hermes>";
        HermesContextParser parser = new HermesContextParser();
        HermesContext context = parser.parse(new ByteArrayInputStream(xmlString.getBytes()));

        Assert.assertEquals(context.getDestination(), "unknown");
        Assert.assertEquals(context.getPolicy(), UpdatingLibraryCheckPolicy.DESTINATION_CHECK);
    }
}
