package jp.sourceforge.talisman.hermes;

/*
 * $Id: HermesContextTest.java 198 2009-05-30 14:45:26Z tama3 $
 */

import java.io.File;
import java.net.URL;

import jp.sourceforge.talisman.hermes.maven.Artifact;
import jp.sourceforge.talisman.hermes.maven.DependencyScope;
import jp.sourceforge.talisman.hermes.maven.Repository;

import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

/**
 *
 * @author Haruaki Tamada
 * @version $Revision: 198 $
 */
public class HermesContextTest{
    private HermesContext context;

    @Before
    public void setUp() throws Exception{
        HermesContextParser parser = new HermesContextParser();
        context = parser.parse(getClass().getResource("/hermes/hermes.xml"));
    }

    @Test
    public void testBasic() throws Exception{
        Assert.assertEquals(context.getRepositoryCount(), 4);

        Repository[] repositories = context.getRepositories();

        Artifact[] dependencies = context.getDependencies();
        Assert.assertEquals(dependencies.length, 1);
        Assert.assertEquals(dependencies[0].getGroupId(), "jp.sourceforge");
        Assert.assertEquals(dependencies[0].getArtifactId(), "stigmata");
        Assert.assertEquals(dependencies[0].getVersion(), "2.0.0");

        Assert.assertTrue(context.hasRepository(repositories[0]));
        Assert.assertTrue(context.hasRepository(repositories[1]));
        Assert.assertTrue(context.hasRepository(repositories[2]));
        Assert.assertTrue(context.hasRepository(repositories[3]));

        Assert.assertTrue(context.hasRepository(new File(System.getProperty("user.home"), ".m2/repository").toURI().toURL()));
        Assert.assertTrue(context.hasRepository(new URL("http://repo1.maven.org/maven2/")));
        Assert.assertTrue(context.hasRepository(new URL("http://talisman.sourceforge.jp/maven2/")));
        Assert.assertTrue(context.hasRepository(new URL("http://stigmata.sourceforge.jp/maven2/")));

        Assert.assertFalse(context.hasRepository(new URL("http://www.google.com/")));
        Assert.assertFalse(context.hasRepository(new Repository(new URL("http://www.google.com/"))));

        context.removeRepository(new URL("http://talisman.sourceforge.jp/maven2/"));
        Assert.assertEquals(context.getRepositoryCount(), 3);

        context.removeRepository(new URL("http://www.google.com"));
        Assert.assertEquals(context.getRepositoryCount(), 3);
    }

    @Test
    public void testDependencyScope() throws Exception{
        Assert.assertTrue(context.isInclude(DependencyScope.COMPILE));
        Assert.assertTrue(context.isInclude(DependencyScope.RUNTIME));
        Assert.assertFalse(context.isInclude(DependencyScope.PROVIDED));
        Assert.assertFalse(context.isInclude(DependencyScope.SYSTEM));
        Assert.assertFalse(context.isInclude(DependencyScope.TEST));

        context.addDependencyScope(DependencyScope.SYSTEM);
        context.addDependencyScope(DependencyScope.TEST);
        context.removeDependencyScope(DependencyScope.RUNTIME);
        context.removeDependencyScope(DependencyScope.COMPILE);

        Assert.assertFalse(context.isInclude(DependencyScope.COMPILE));
        Assert.assertFalse(context.isInclude(DependencyScope.RUNTIME));
        Assert.assertFalse(context.isInclude(DependencyScope.PROVIDED));
        Assert.assertTrue(context.isInclude(DependencyScope.SYSTEM));
        Assert.assertTrue(context.isInclude(DependencyScope.TEST));
    }

    @Test
    public void testDependency() throws Exception{
        Artifact artifact1 = new Artifact("jp.sourceforge.talisman", "hermes");
        Artifact artifact2 = new Artifact("jp.sourceforge.stigmata", "digger");
        Artifact artifact3 = new Artifact("jp.sourceforge", "stigmata");

        context.addDependency(artifact1);
        context.addDependency(artifact2);

        Artifact[] dependencies1 = context.getDependencies();
        Assert.assertEquals(dependencies1.length, 3);
        Assert.assertEquals(dependencies1[0].getGroupId(), "jp.sourceforge");
        Assert.assertEquals(dependencies1[0].getArtifactId(), "stigmata");
        Assert.assertEquals(dependencies1[0].getVersion(), "2.0.0");

        Assert.assertEquals(dependencies1[1].getGroupId(), "jp.sourceforge.talisman");
        Assert.assertEquals(dependencies1[1].getArtifactId(), "hermes");
        Assert.assertNull(dependencies1[1].getVersion());

        Assert.assertEquals(dependencies1[2].getGroupId(), "jp.sourceforge.stigmata");
        Assert.assertEquals(dependencies1[2].getArtifactId(), "digger");
        Assert.assertNull(dependencies1[2].getVersion());

        context.removeDependency(artifact3);
        context.removeDependency(artifact2);

        Artifact[] dependencies2 = context.getDependencies();
        Assert.assertEquals(dependencies2.length, 1);

        Assert.assertEquals(dependencies2[0].getGroupId(), "jp.sourceforge.talisman");
        Assert.assertEquals(dependencies2[0].getArtifactId(), "hermes");
        Assert.assertNull(dependencies2[0].getVersion());
    }

    @Test
    public void testIgnores() throws Exception{
        context.addIgnore("jp.sourceforge.talisman", "hermes");
        context.addIgnore("jp.sourceforge.talisman", "i18n");
        context.addIgnore("jp.sourceforge.stigmata", "digger");
        context.addIgnore("jp.sourceforge", "stigmata");

        Assert.assertTrue(context.isIgnore("jp.sourceforge.talisman", "hermes"));
        Assert.assertTrue(context.isIgnore("jp.sourceforge.talisman", "i18n"));
        Assert.assertTrue(context.isIgnore("jp.sourceforge.stigmata", "digger"));
        Assert.assertTrue(context.isIgnore("jp.sourceforge", "stigmata"));

        Assert.assertFalse(context.isIgnore("jp.sourceforge.talisman", "xmlcli"));
        Assert.assertFalse(context.isIgnore("jp.sourceforge", "NOTstigmata"));
        Assert.assertFalse(context.isIgnore("org.apache.maven", "maven"));

        context.removeIgnores("jp.sourceforge.talisman", "hermes");
        context.removeIgnores("jp.sourceforge.talisman", "i18n");

        Assert.assertFalse(context.isIgnore("jp.sourceforge.talisman", "hermes"));
        Assert.assertFalse(context.isIgnore("jp.sourceforge.talisman", "i18n"));
        Assert.assertTrue(context.isIgnore("jp.sourceforge.stigmata", "digger"));
        Assert.assertTrue(context.isIgnore("jp.sourceforge", "stigmata"));
    }
}
