package jp.sourceforge.talisman.hermes;

/*
 * $Id: Hermes.java 200 2009-05-31 16:20:06Z tama3 $
 */

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.net.URLConnection;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import jp.sourceforge.talisman.hermes.maven.Artifact;
import jp.sourceforge.talisman.hermes.maven.DependencyScope;
import jp.sourceforge.talisman.hermes.maven.RepositoryItemNotFoundException;
import jp.sourceforge.talisman.hermes.maven.RepositoryManager;

/**
 * core class of Hermes API.
 * 
 * @author Haruaki Tamada
 * @version $Revision: 200 $
 */
public class Hermes{
    private HermesContext context;
    private List<HermesListener> listeners = new ArrayList<HermesListener>();
    private RepositoryManager manager;

    /**
     * Construct Hermes with given context.
     * @param context
     */
    public Hermes(HermesContext context){
        this.context = context;
        manager = new RepositoryManager(context);
    }

    /**
     * adds HermesListener interface for receiving notification of
     * updating process.
     */
    public void addHermesListener(HermesListener listener){
        listeners.add(listener);
    }

    /**
     * adds HermesPercentageListener interface for receiving
     * notification of updating process.
     */
    public void addHermesListener(HermesPercentageListener listener){
        addHermesListener(new DelegateHermesPercentageListener(listener));
    }

    /**
     * install given artifact to hermes destination.
     *
     * @param artifact install target
     * @param in InputStream of install target's jar file.
     * @throws IOException 
     * @throws IllegalStateException destination does not set in HermesContext
     * @see HermesContext
     */
    public void install(Artifact artifact, InputStream in) throws IOException{
        uninstall(new Artifact(artifact.getGroupId(), artifact.getArtifactId()));

        String dest = getContext().getDestination();
        if(dest == null){
            throw new IllegalStateException("destination is not specified");
        }
        File file = new File(dest, artifact.getFileName());
        OutputStream out = new FileOutputStream(file);
        byte[] data = new byte[1024];
        int len = 0;
        int size = 0;
        while((len = in.read(data)) != -1){
            out.write(data, 0, len);
            size += len;
            fireEvent(new HermesEvent(HermesEvent.Id.FILE_SIZE_UPDATED, artifact, size));
        }
        out.close();

        getContext().addDependency(artifact);
    }

    /**
     * uninstall given artifact from destination.
     */
    public void uninstall(Artifact artifact){
        getContext().removeDependency(artifact);

        if(getContext().getDestination() != null){
            File file = new File(getContext().getDestination());
            for(File f: file.listFiles()){
                if(f.getName().matches(artifact.getArtifactId() + "-[0-9.-]+.+.jar")){
                    f.renameTo(new File(f.getParentFile(), f.getName() + ".uninstalled"));
                    break;
                }
            }
        }
    }

    public boolean canUpdate() throws IOException, HermesException{
        return canUpdate(getContext().getDependencies());
    }

    public boolean canUpdate(Artifact artifact) throws IOException, HermesException{
        return canUpdate(new Artifact[] { artifact });
    }

    public boolean canUpdate(Artifact[] originalArtifacts) throws IOException, HermesException{
        return getUpdateTarget(originalArtifacts).length > 0;
    }

    /**
     * Finds given artifact from the all of repositories registered in
     * the context.
     * @see #getContext 
     * @exception IOException I/O error in communication with repositories.
     */
    public Artifact findArtifact(Artifact artifact) throws IOException{
        return manager.findArtifact(artifact);
    }

    /**
     * Finds an artifact which has the given groupId and the given
     * artifactId from the all of repositories registered in the
     * context.
     * @see #getContext
     * @exception IOException I/O error in communication with repositories.
     */
    public Artifact findArtifact(String groupId, String artifactId) throws IOException{
        return manager.findArtifact(groupId, artifactId, null);
    }

    /**
     * Finds an artifact which has the given groupId, the artifactId
     * and the version from the all of repositories registered in the
     * context.
     * @see #getContext
     * @exception IOException I/O error in communication with repositories.
     */
    public Artifact findArtifact(String groupId, String artifactId, String version) throws IOException{
        return manager.findArtifact(groupId, artifactId, version);
    }

    /**
     * returns the context.
     */
    public HermesContext getContext(){
        return context;
    }

    /**
     * {@link #getDependencies(Artifact[]) <code>getDependencies(getContext().getDependencies())</code>}.
     * @see #getContext()
     * @see #getDependencies(Artifact[])
     */
    public Artifact[] getDependencies() throws IOException, HermesException{
        return getDependencies(getContext().getDependencies());
    }

    /**
     * {@link #getDependencies(Artifact[]) <code>getDependencies(new Artifact[] { artifact })</code>}.
     * @see #getDependencies(Artifact[])
     */
    public Artifact[] getDependencies(Artifact artifact) throws IOException, HermesException{
        return getDependencies(new Artifact[] { artifact });
    }

    /**
     * finds dependencies of given artifacts and returns them as an array.
     * @exception IOException I/O error in communication with repositories.
     * @exception HermesException application error.
     */
    public Artifact[] getDependencies(Artifact[] artifacts) throws IOException, HermesException{
        Map<String, Map<String, Artifact>> map = new LinkedHashMap<String, Map<String, Artifact>>();

        findDependencies(artifacts, map);

        List<Artifact> list = new ArrayList<Artifact>();
        for(Map<String, Artifact> m: map.values()){
            for(Artifact a: m.values()){
                list.add(a);
            }
        }
        return list.toArray(new Artifact[list.size()]);
    }

    public void removeHermesListener(HermesListener listener){
        listeners.remove(listener);
    }

    public void removeHermesListener(HermesPercentageListener removeTargetListener){
        HermesListener removeTarget = null;
        for(HermesListener listener: listeners){
            if(listener instanceof DelegateHermesPercentageListener){
                if(removeTargetListener == ((DelegateHermesPercentageListener)listener).getListener()){
                    removeTarget = listener;
                    break;
                }
            }
        }
        if(removeTarget != null){
            listeners.remove(removeTarget);
        }
    }

    /**
     * {@link #update(Artifact[]) <code>update(getContext().getDependencies())</code>}
     * @see #getContext()
     */
    public void update() throws IOException, HermesException{
        update(getContext().getDependencies());
    }

    /**
     * {@link #update(Artifact[]) <code>update(new Artifact[] { artifact, })</code>}
     */
    public void update(Artifact artifact) throws IOException, HermesException{
        update(new Artifact[] { artifact, });
    }

    /**
     * Updates given artifacts and their dependent artifacts.
     * Progress of updating process is notified via HermesListener interface.
     * @see HermesListener
     */
    public void update(Artifact[] artifacts) throws IOException, HermesException{
        Artifact[] resolvedArtifacts = getUpdateTarget(artifacts);

        fireEvent(new HermesEvent(HermesEvent.Id.TARGET_RESOLVED, resolvedArtifacts));
        for(Artifact artifact: resolvedArtifacts){
            URL url = artifact.getPom().getUrl();
            URLConnection connection = url.openConnection();
            int length = connection.getContentLength();
            fireEvent(new HermesEvent(HermesEvent.Id.FILE_SIZE_GOTTEN, artifact, length));

            InputStream in = null;
            try{
                in = connection.getInputStream();
                install(artifact, in);
            } finally{
                if(in != null){
                    try{
                        in.close();
                    } catch(IOException e){
                    }
                }
            }
            fireEvent(new HermesEvent(HermesEvent.Id.DOWNLOAD_DONE, artifact));
        }
        fireEvent(new HermesEvent(HermesEvent.Id.FINISH));

        for(Artifact a: resolvedArtifacts){
            getContext().removeDependency(a.getGroupId(), a.getArtifactId());
            getContext().addDependency(a);
        }
    }

    /**
     * {@link #getUpdateTarget(Artifact[])
     * <code>getUpdateTarget(getContext().getDependencies())</code>}
     * @see #getContext
     */
    public Artifact[] getUpdateTarget() throws IOException, HermesException{
        return getUpdateTarget(getContext().getDependencies());
    }

    /**
     * {@link #getUpdateTarget(Artifact[])
     * <code>getUpdateTarget(new Artifact[] { originalArtifacts, })</code>}
     * @see #getContext
     */
    public Artifact[] getUpdateTarget(Artifact originalArtifact) throws IOException, HermesException{
        return getUpdateTarget(new Artifact[] { originalArtifact, });
    }

    /**
     * finds updating targets of given artifacts.
     * @see UpdatingLibraryCheckPolicy
     */
    public Artifact[] getUpdateTarget(Artifact[] artifacts) throws IOException, HermesException{
        // resolve dependencies of latest version target.
        Artifact[] targets = new Artifact[artifacts.length];
        for(int i = 0; i < targets.length; i++){
            targets[i] = findArtifact(artifacts[i].getGroupId(), artifacts[i].getArtifactId());
        }
        Artifact[] depResolvedArtifacts = getDependencies(targets);

        List<Artifact> artifactList;
        switch(getContext().getPolicy()){
        case DESTINATION_CHECK:
            artifactList = checkDestination(depResolvedArtifacts, artifacts);
            break;
        case REPOSITORY_CHECK:
            artifactList = checkRepository(depResolvedArtifacts, artifacts);
            break;
        case NO_CHECK:
        default:
            artifactList = new ArrayList<Artifact>();
            for(Artifact artifact: depResolvedArtifacts){
                artifactList.add(artifact);
            }
            break;
        }

        return artifactList.toArray(new Artifact[artifactList.size()]);
    }

    private List<Artifact> checkDestination(Artifact[] resolvedArtifacts, Artifact[] original) throws IOException{
        File file = new File(getContext().getDestination());
        if(!file.exists()){
            throw new FileNotFoundException(file.getPath() + " is not found");
        }

        List<Artifact> artifactList = new ArrayList<Artifact>();
        File[] files = file.listFiles();

        for(Artifact artifact: resolvedArtifacts){
            Artifact target = artifact;
            for(int i = 0; i < files.length; i++){
                if(files[i] != null){
                    String name = files[i].getName();
                    if(name.startsWith(artifact.getArtifactId() + "-") && name.endsWith(".jar")){
                        if(name.equals(artifact.getFileName())){
                            target = null;
                        }
                        files[i] = null;
                    }
                }
            }
            if(target != null){
                artifactList.add(target);
            }
        }

        return artifactList;
    }

    private List<Artifact> checkRepository(Artifact[] depResolvedArtifacts, Artifact[] artifacts) throws IOException, HermesException{
        List<Artifact> artifactList = new ArrayList<Artifact>(); 
        // resolve dependencies of given target.
        Artifact[] origDepArtifacts = getDependencies(artifacts);

        // list updated libraries for matching version.
        for(int j = 0; j < depResolvedArtifacts.length; j++){
            Artifact a1 = depResolvedArtifacts[j];
            int index = -1;
            for(int i = 0; i < origDepArtifacts.length; i++){
                Artifact a2 = origDepArtifacts[i];
                if(a2 != null && a1.getGroupId().equals(a2.getGroupId())
                        && a1.getArtifactId().equals(a2.getArtifactId())){
                    String version1 = a1.getVersion();
                    String version2 = a2.getVersion();

                    index = i;
                    // new version is available.
                    if(!version1.equals(version2)){
                        artifactList.add(a1);
                    }
                    break;
                }
            }
            if(index >= 0){
                origDepArtifacts[index] = null;
            }
        }
        return artifactList;
    }

    /**
     * traversing dependencies tree of given artifacts, and storing
     * dependencies into given map.
     * @exception an artifact of dependencies tree is not found in
     * registered repositories.
     */
    private void findDependencies(Artifact[] artifacts, Map<String, Map<String, Artifact>> map) throws IOException, HermesException{
        for(Artifact artifact: artifacts){
            if(!isTargetableScope(artifact)){
                continue;
            }
            if(!isTarget(artifact)){
                continue;
            }
            if(isAlreadyFound(artifact, map)){
                continue;
            }
            checkVersion(artifact, map);

            Artifact resolvedArtifact = findArtifact(artifact);
            if(resolvedArtifact == null){
                throw new RepositoryItemNotFoundException(String.format(
                    "%s:%s:%s not found", artifact.getGroupId(),
                    artifact.getArtifactId(), artifact.getVersion())
                );
            }
            String groupId = resolvedArtifact.getGroupId();
            Map<String, Artifact> submap = map.get(groupId);
            if(submap == null){
                submap = new LinkedHashMap<String, Artifact>();
                map.put(groupId, submap);
            }
            submap.put(resolvedArtifact.getArtifactId(), resolvedArtifact);

            Artifact[] newDependencies = resolvedArtifact.getDependencies();
            findDependencies(newDependencies, map);
        }
    }

    /**
     * If version conflict is founds in artifacts of depdendencies
     * tree, this method throws {@link VersionMismatchException
     * <code>VersionMismatchException</code>}.  Because, two artifacts
     * of different version, but same groupId, and artifactId is not
     * correct dependencies tree.
     * 
     * @param artifact target artifact.
     * @param map stored artifacts in dependencies tree.
     * @exception VersionMismatchException version mismatch artifact is found.
     */
    private void checkVersion(Artifact artifact, Map<String, Map<String, Artifact>> map) throws HermesException{
        if(map.get(artifact.getGroupId()) != null
                && map.get(artifact.getGroupId()).get(artifact.getArtifactId()) != null){
            Artifact artifact2 = map.get(artifact.getGroupId()).get(artifact.getArtifactId());
            if(!artifact.getVersion().equals(artifact2.getVersion())){
                throw new VersionMismatchException(String.format(
                    "%s:%s, version %s and %s",
                    artifact2.getGroupId(), artifact2.getArtifactId(),
                    artifact.getVersion(), artifact2.getVersion())
                );
            }
        }
    }

    /**
     * returns true when the given artifact is already found in the map.
     */
    private boolean isAlreadyFound(Artifact artifact, Map<String, Map<String, Artifact>> map){
        Map<String, Artifact> submap = map.get(artifact.getGroupId());
        return submap != null && submap.get(artifact.getArtifactId()) != null;
    }

    private boolean isTarget(Artifact artifact){
        return !getContext().isIgnore(artifact.getGroupId(), artifact.getArtifactId());
    }

    private boolean isTargetableScope(Artifact artifact){
        DependencyScope scope = artifact.getScope();
        return scope == null || getContext().isInclude(scope);
    }

    private void fireEvent(HermesEvent e){
        for(HermesListener listener: listeners){
            switch(e.getId()){
            case DOWNLOAD_DONE:
                listener.downloadDone(e);
                break;
            case FILE_SIZE_GOTTEN:
                listener.fileSizeGotten(e);
                break;
            case FILE_SIZE_UPDATED:
                listener.fileSizeUpdated(e);
                break;
            case FINISH:
                listener.finish(e);
                break;
            case TARGET_RESOLVED:
                listener.targetResolved(e);
            }
        }
    }
}
