package jp.sourceforge.talisman.hermes;

/*
 * $Id: HermesContext.java 198 2009-05-30 14:45:26Z tama3 $
 */

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.Serializable;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import jp.sourceforge.talisman.hermes.maven.Artifact;
import jp.sourceforge.talisman.hermes.maven.DependencyScope;
import jp.sourceforge.talisman.hermes.maven.Repository;

/**
 * 
 * @author Haruaki Tamada
 * @version $Revision: 198 $
 */
public class HermesContext implements Serializable, Iterable<Artifact>{
    private static final long serialVersionUID = 3130612309057960466L;

    private String destination;
    private UpdatingLibraryCheckPolicy policy;
    private RepositoryScope scope;

    private List<Artifact> dependencies;
    private List<PropertyChangeListener> propertyChangeListeners;
    private List<Repository> repositories;
    private Set<DependencyScope> dependencyScopes;
    private Map<String, List<String>> ignores;

    /**
     * Basic constructor.
     */
    public HermesContext(){
        this(Repository.getLocalRepository());
    }

    /**
     * constructor for testing.
     */
    HermesContext(Repository localRepository){
        repositories = new ArrayList<Repository>();
        propertyChangeListeners = new ArrayList<PropertyChangeListener>();
        ignores = new HashMap<String, List<String>>();
        dependencyScopes = new HashSet<DependencyScope>();
        dependencies = new ArrayList<Artifact>();

        scope = RepositoryScope.BOTH;
        policy = UpdatingLibraryCheckPolicy.DESTINATION_CHECK;

        repositories.add(localRepository);

        dependencyScopes.add(DependencyScope.COMPILE);
        dependencyScopes.add(DependencyScope.RUNTIME);
    }

    /**
     * Adding dependency
     */
    public void addDependency(Artifact artifact){
        if(!dependencies.contains(artifact)){
            Artifact[] oldValue = dependencies.toArray(new Artifact[dependencies.size()]);
            dependencies.add(artifact);
            Artifact[] newValue = dependencies.toArray(new Artifact[dependencies.size()]);
            fireEvent(new PropertyChangeEvent(this, "dependency", oldValue, newValue));
        }
    }

    /**
     * Adding dependency scope.
     */
    public synchronized void addDependencyScope(DependencyScope scope){
        if(!dependencyScopes.contains(scope)){
            DependencyScope[] oldValue = dependencyScopes.toArray(new DependencyScope[dependencyScopes.size()]);
            dependencyScopes.add(scope);
            DependencyScope[] newValue = dependencyScopes.toArray(new DependencyScope[dependencyScopes.size()]);
            fireEvent(new PropertyChangeEvent(this, "dependency-scope", oldValue, newValue));
        }
    }

    /**
     * Adding ignoreing artifact.
     */
    public void addIgnore(String groupId, String artifactId){
        Artifact[] oldValue = getIgnores();
        List<String> list = ignores.get(groupId);
        if(list == null){
            list = new ArrayList<String>();
            ignores.put(groupId, list);
        }
        if(!list.contains(artifactId)){
            list.add(artifactId);
            Artifact[] newValue = getIgnores();
            fireEvent(new PropertyChangeEvent(this, "ignores", oldValue, newValue));
        }
    }

    /**
     * Adding <code>PropertyChangeListener</code>.
     */
    public void addPropertyChangeListener(PropertyChangeListener listener){
        propertyChangeListeners.add(listener);
    }

    /**
     * Adding repository.
     */
    public void addRepository(Repository repository){
        if(!repositories.contains(repository)){
            Repository[] oldValue = repositories.toArray(new Repository[repositories.size()]);
            repositories.add(repository);
            Repository[] newValue = repositories.toArray(new Repository[repositories.size()]);
            fireEvent(new PropertyChangeEvent(this, "repositories", oldValue, newValue));
        }
    }

    /**
     * returns all of dependencies as an array.
     */
    public Artifact[] getDependencies(){
        return dependencies.toArray(new Artifact[dependencies.size()]);
    }

    /**
     * returns an Artifact which has given groupId and artifactId.  If
     * this context do no have an artifact which has given groupId and
     * artifactId, this method returns null.
     */
    public Artifact getDependency(String groupId, String artifactId){
        for(Artifact artifact: dependencies){
            if(artifact.getGroupId().equals(groupId) && artifact.getArtifactId().equals(artifactId)){
                return artifact;
            }
        }
        return null;
    }

    /**
     * returns destination.
     */
    public String getDestination(){
        return destination;
    }

    /**
     * returns all of ignores as an array.
     */
    public Artifact[] getIgnores(){
        List<Artifact> ignoreList = new ArrayList<Artifact>();
        for(Map.Entry<String, List<String>> entry: ignores.entrySet()){
            String groupId = entry.getKey();
            List<String> list = entry.getValue();
            for(String artifactId: list){
                ignoreList.add(new Artifact(groupId, artifactId));
            }
        }
        return ignoreList.toArray(new Artifact[ignoreList.size()]);
    }

    /**
     * returns updating library check policy.
     */
    public UpdatingLibraryCheckPolicy getPolicy(){
        return policy;
    }

    /**
     * returns all of repositories as an array.
     */
    public synchronized Repository[] getRepositories(){
        return Collections.unmodifiableList(repositories).toArray(new Repository[repositories.size()]);
    }

    /**
     * returns the count of repositories this context has.
     */
    public int getRepositoryCount(){
        return repositories.size();
    }

    /**
     * returns repository scope.
     */
    public RepositoryScope getRepositoryScope(){
        return scope;
    }

    /**
     * returns repositories corresponding repository scope.
     * @see #getRepositoryScope
     */
    public synchronized Repository[] getScopedRepositories(){
        List<Repository> repositoryList = new ArrayList<Repository>();
        for(Repository repository: repositories){
            if(repository.isAllowed(scope)){
                repositoryList.add(repository);
            }
        }
        return repositoryList.toArray(new Repository[repositoryList.size()]);
    }

    /**
     * returns true when this context has given repository.
     */
    public synchronized boolean hasRepository(Repository repo){
        return repositories.contains(repo);
    }

    /**
     * returns true when this context has repository which location is given url.
     */
    public synchronized boolean hasRepository(URL url){
        for(Repository repo: repositories){
            if(repo.isSameLocation(url)){
                return true;
            }
        }
        return false;
    }

    /**
     * returns true when given groupId and artifactId is registered as ignores.
     */
    public boolean isIgnore(String groupId, String artifactId){
        List<String> list = ignores.get(groupId);
        return list != null && list.contains(artifactId);
    }

    /**
     * returns true when given scope is included in dependency scope.
     */
    public boolean isInclude(DependencyScope scope){
        return dependencyScopes.contains(scope);
    }

    /**
     * returns iterator of dependencies.
     */
    public Iterator<Artifact> iterator(){
        return Collections.unmodifiableList(dependencies).iterator();
    }

    /**
     * remove given artifact from dependency list.
     * @throws NullPointerException given artifact is null.
     */
    public void removeDependency(Artifact artifact){
        if(artifact == null){
            throw new NullPointerException();
        }
        removeDependency(artifact.getGroupId(), artifact.getArtifactId());
    }

    /**
     * remove an artifact which has given groupId and artifactId from dependency list.
     * @throws NullPointerException groupId or artifactId are null.
     */
    public void removeDependency(String groupId, String artifactId){
        if(groupId == null || artifactId == null){
            throw new NullPointerException();
        }
        Artifact[] oldValue = dependencies.toArray(new Artifact[dependencies.size()]);
        Artifact targetArtifact = null;
        for(Artifact a: this){
            if(a.getGroupId().equals(groupId) && a.getArtifactId().equals(artifactId)){
                targetArtifact = a;
                break;
            }
        }
        if(targetArtifact != null){
            dependencies.remove(targetArtifact);
            Artifact[] newValue = dependencies.toArray(new Artifact[dependencies.size()]);
            fireEvent(new PropertyChangeEvent(this, "dependency", oldValue, newValue));
        }
    }

    /**
     * remove given dependency scope.
     */
    public void removeDependencyScope(DependencyScope scope){
        if(dependencyScopes.contains(scope)){
            DependencyScope[] oldValue = dependencyScopes.toArray(new DependencyScope[dependencyScopes.size()]);
            dependencyScopes.remove(scope);
            DependencyScope[] newValue = dependencyScopes.toArray(new DependencyScope[dependencyScopes.size()]);
            fireEvent(new PropertyChangeEvent(this, "dependency-scope", oldValue, newValue));
        }
    }

    /**
     * remove given artifact from ignore list.
     * @throws NullPointerException given artifact is null.
     */
    public void removeIgnores(Artifact artifact){
        if(artifact == null){
            throw new NullPointerException();
        }
        removeIgnores(artifact.getGroupId(), artifact.getArtifactId());
    }

    /**
     * remove an artifact which has given groupId and artifactId from ignore list.
     * @throws NullPointerException groupId or artifactId are null.
     */
    public void removeIgnores(String groupId, String artifactId){
        if(groupId == null || artifactId == null){
            throw new NullPointerException();
        }
        Artifact[] oldValue = getIgnores();

        List<String> list = ignores.get(groupId);
        if(list != null && list.contains(artifactId)){
            list.remove(artifactId);
            if(list.size() == 0){
                ignores.remove(groupId);
            }
            Artifact[] newValue = getIgnores();
            fireEvent(new PropertyChangeEvent(this, "ignores", oldValue, newValue));
        }
    }

    /**
     * remove given PropertyChangeListener.
     */
    public void removePropertyChangeListener(PropertyChangeListener listener){
        propertyChangeListeners.remove(listener);
    }

    /**
     * remove given repository.
     */
    public synchronized boolean removeRepository(Repository repository){
        Repository[] oldValue = getRepositories();
        boolean flag = repositories.remove(repository);
        if(flag){
            Repository[] newValue = getRepositories();
            fireEvent(new PropertyChangeEvent(this, "repositories", oldValue, newValue));
        }
        return flag;
    }

    /**
     * remove repository which location is given url.
     */
    public synchronized boolean removeRepository(URL url){
        Repository[] oldValue = getRepositories();
        Repository removeRepository = null;
        for(Repository repo: repositories){
            if(repo.isSameLocation(url)){
                removeRepository = repo;
                break;
            }
        }
        if(removeRepository != null){
            repositories.remove(removeRepository);
        }
        if(removeRepository != null){
            Repository[] newValue = getRepositories();
            fireEvent(new PropertyChangeEvent(this, "repositories", oldValue, newValue));
        }
        return removeRepository != null;
    }

    /**
     * set destination.
     */
    public void setDestination(String dest){
        String oldValue = getDestination();
        this.destination = dest;
        fireEvent(new PropertyChangeEvent(this, "destination", oldValue, dest));
    }

    /**
     * set updating library check policy.
     */
    public void setPolicy(UpdatingLibraryCheckPolicy policy){
        UpdatingLibraryCheckPolicy oldValue = getPolicy();
        this.policy = policy;
        fireEvent(new PropertyChangeEvent(this, "policy", oldValue, policy));
    }

    /**
     * set repository scope.
     */
    public void setRepositoryScope(RepositoryScope scope){
        RepositoryScope oldValue = getRepositoryScope();
        this.scope = scope;
        fireEvent(new PropertyChangeEvent(this, "repository-scope", oldValue, scope));
    }

    /**
     * fire property change event.
     */
    private void fireEvent(PropertyChangeEvent e){
        for(PropertyChangeListener listener: propertyChangeListeners){
            listener.propertyChange(e);
        }
    }
}
