#!/usr/local/bin/ruby

#
# browser - Data browser with TapKit
#
# usage: browser modelfile
#

$LOAD_PATH.unshift '../lib'

require 'tapkit'

# like, cslike, ==, !=, >, <, >=, <=, and, or, not
class Browser
	include TapKit

	USAGE = 'usage: browser modelfile'
	WELCOME = "Welcome to the data browser with TapKit.\n" +
	          "Type 'help' for help, 'quit' for quit the program.\n\n"
	PROMPT = 'browser> '
	QUIT = 'quit'
	HELP = "Commands:\n" +
	"    help            show help (this message)\n" +
	"    quit            quit the program\n" +
	"    entity          show current entity\n" +
	"    use [entity]    change current entity\n" +
	"    entities        show list of entities\n" +
	"    attributes      show list of attributes for current entity\n" +
	"    relationships   show list of relationships for current entity\n" +
	"\nOperators:\n" +
	"    like            like\n" +
	"    cilike          case insensitive like\n" +
	"    ==              equal\n" +
	"    !=              not equal\n" +
	"    >               greater than\n" +
	"    <               less than\n" +
	"    >=              greater than or equal\n" +
	"    <=              less than or equal\n" +
	"    and             and\n" +
	"    or              or\n" +
	"    not             not\n"

	def initialize( modelfiles )
		if modelfiles.empty? then
			show_usage
		end

		@tapkit = Application.new modelfiles
		@group  = @tapkit.model_group
		@ec     = @tapkit.shared_editing_context
	end

	def show_usage
		puts USAGE
		exit
	end

	def run
		input = nil

		show_welcome
		begin
			if input then
				case input
				when 'help'
					show_help
				when 'quit'
					exit
				when /use\s+(.*)/
					use $1
				when 'entity'
					show_entity
				when 'entities'
					show_entities
				when 'attributes'
					show_attributes
				when 'relationships'
					show_relationships
				else
					fetch input
				end
			end
			show_prompt
		end while input = $stdin.gets.chomp
	end

	def show_welcome
		puts WELCOME
	end

	def show_prompt
		print PROMPT
	end

	def show_help
		puts HELP
	end

	def use( entity_name )
		if entity = @tapkit.model_group.entity(entity_name) then
			puts "entity changed"
			@entity = entity
		else
			puts "#{entity_name}: no such entity"
		end
	end

	def show_entity
		unless @entity then
			puts "entity is not selected"
		else
			name = @entity.name
			puts "current entity is '#{name}'"
		end
	end

	def show_entities
		entities = []
		@group.models.each do |model|
			model.entity_names.each do |name|
				entities << {'entities' => name}
			end
		end
		puts format(['entities'], entities)
	end

	def show_attributes
		attributes = []
		@entity.attributes.each do |attr|
			attributes << {'attributes' => attr.name}
		end
		puts format(['attributes'], attributes)
	end

	def show_relationships
		relationships = []
		@entity.relationships.each do |re|
			relationships << {'relationships' => re.name}
		end
		puts format(['relationships'], relationships)
	end

	def fetch( format )
		begin
			qualifier  = Qualifier.new_with_format format
			fetch_spec = FetchSpec.new(@entity.name, qualifier)
			start      = Object::Time.new
			objects    = @ec.fetch fetch_spec
			sec        = Object::Time.new - start

			names = []
			@entity.attributes.each do |attr|
				names << attr.name
			end

			puts format(names, objects, sec)
		rescue
			puts "error: can't fetch"
		end
	end

	def format( headers, rows, sec = nil )
		str   = ''
		sizes = {}

		headers.each do |header|
			sizes[header] = header.size
		end

		rows.each do |row|
			headers.each do |header|
				value = row[header]
				size  = value.to_s.size
				if sizes[header] then
					if sizes[header] < size then
						sizes[header] = size
					end
				else
					sizes[header] = size
				end
			end
		end

		th   = ''
		line = ''
		headers.each do |header|
			th   << ('| ' + header.ljust(sizes[header]+1))
			line << ('+-' + ('-' * (sizes[header] +1)))
		end
		th   << "|\n"
		line << "+\n"

		rows_str = ''
		rows.each do |row|
			row_str = ''
			headers.each do |header|
				size  = sizes[header]
				value = row[header]
				row_str << '| '

				case value
				when Integer
					row_str << value.to_s.rjust(size)
				else
					row_str << value.to_s.ljust(size)
				end
				row_str << " "
			end
			row_str  << "|\n"
			rows_str << row_str
		end

		str << line
		str << th
		str << line
		str << rows_str
		str << line
		if rows.size == 1 then
			str << "#{rows.size} row in set"
		else
			str << "#{rows.size} rows in set"
		end
		str << " (#{sec} sec)" if sec
		str << "\n\n"

		str
	end
end


if __FILE__ == $0 then
	browser = Browser.new ARGV
	browser.run
end
