module TapKit

	class DatabaseChannel
		attr_writer :locking, :editing_context, :entity, :refresh
		attr_reader :database_context, :adapter_channel

		def initialize( database_context )
			@database_context = database_context
			name = @database_context.database.adapter_name
			@adapter_channel = \
				AdapterChannel.new_with_name(name, @database_context.adapter_context)
			@fetched_objects = []
			@refresh = false
		end

		def locking?
			@locking
		end

		def lock
			@locking = true
		end

		def unlock
			@locking = false
		end

		def open?
			@adapter_channel.open?
		end

		def open
			@adapter_channel.open
		end

		def close
			@adapter_channel.close
		end

		def refresh?
			@refresh
		end

		def select( fetch_spec, editing_context )
			db = @database_context.database
			app = db.application

			@editing_context = editing_context
			@entity = app.model_group.entity fetch_spec.entity_name
			attrs = @entity.attributes
			lock = true
			@refresh = fetch_spec.refresh?

			factory = db.adapter.expression_factory
			sql = factory.select_statement(attrs, lock, fetch_spec, @entity)
			@adapter_channel.select_attributes(attrs, lock, fetch_spec, @entity)
			open unless open?
			@adapter_channel.evaluate sql
		end

		def fetch_all
			db = @database_context.database
			nc = db.application.notification_center
			objects = []

			@adapter_channel.fetch_all.each do |row|
				gid = @entity.gid row

				snapshot = Snapshot.new
				snapshot.update row

				# register the snapshot
				# don't I regist the snapshot into DatabaseContext?
				if registered = db.snapshot(gid) then
					# update refetched objects
					if refresh? then
						userinfo = {:gid => gid, :snapshot => snapshot}
						db.record_snapshot(gid, snapshot)
						nc.post(ObjectStore::OBJECTS_CHANGED_IN_STORE_NOTIFICATION,
							@database_context, userinfo)
					else
						snapshot = registered
					end
				else
					db.record_snapshot(gid, snapshot)
				end

				if @database_context.own? gid then
					object = @editing_context.object gid
				end

				# create a database object
				if object then
					# resolve faulting
					if object.fault? then
						# userinfo = {gid => gid}
						# nc.post(GlobalID::GID_CHANGED_NOTIFICATION, gid, userinfo)

						# ready fault objects for a faulted object
						@editing_context.initialize_object(object, gid)

						object.update_from_snapshot snapshot
						object.clear_fault
					end
				else
					description = EntityClassDescription.new @entity
					object = description.create(@editing_context, gid)
					@editing_context.record(gid, object)
					@editing_context.initialize_object(object, gid)
					nc.add(@editing_context,
					       :handle_notification,
					       ObjectStore::OBJECTS_CHANGED_IN_STORE_NOTIFICATION,
					       @database_context)
				end

				# refault
				if refresh? then
					@editing_context.refault(object, gid)
				end

				objects << object
			end

			objects
		end
	end

end
