module TapKit

	class DatabaseContext < CooperatingObjectStore
		attr_reader :adapter_context, :database, :coordinator, :channels

		class << self
			def handle_notification( notification )
				case notification.name
				when ObjectStoreCoordinator::COS_NEEDED_NOTIFICATION
					coordinator = notification.object
					entity_name = notification.userinfo[:object].entity_name
					application = notification.userinfo[:application]

					entity   = application.model_group.entity entity_name
					model    = entity.model
					database = application.database model
					context  = DatabaseContext.new database
					coordinator.add_cooperating_object_store context
				end
			end
		end

		def initialize( database )
			@database            = database
			@application         = @database.application
			@coordinator         = @application.object_store
			@observer_center     = @application.observer_center
			@notification_center = @application.notification_center
			@database.register_context self
			@adapter_context = \
				AdapterContext.new_with_name(@database.adapter_name, @database.adapter)

			@channels            = []
			@inserted_objects    = []
			@deleted_objects     = []
			@updated_objects     = []
			@database_operations = []
			@pk_values           = {} # gid => {key => values}
			@temp2key            = nil
			@snapshot_store = SnapshotStore.new
		end

		def initialize_object( object, gid, editing_context )
			# do I use snapshots in DatabaseContext?
			object.application = @application
			entity = @database.entity gid.entity_name
			snapshot = @database.snapshot gid

			# use a snapshot to set default value for object
			entity.class_property_attributes.each do |attribute|
				value = snapshot[attribute.name]
				object._init_stored_value(attribute.name, value) # without will_read
			end

			entity.class_property_relationships.each do |relation|
				dest_object = nil
				if relation.to_many? then
					dest_object = array_fault(gid, relation.name, editing_context)
				else
					destinations = {}

					relation.joins.each do |join|
						# joining PKs to PKs
						names = relation.entity.primary_key_attribute_names
						if names.include? join.source.name then
						end

						if snapshot[join.source.name] then
							destinations[join.destination.name] = snapshot[join.source.name]
						end
					end

					unless destinations.empty? then
						# If the application already has an object for the relation,
						# use it instead of creating fault.
						fault_gid = KeyGlobalID.new(relation.destination_entity.name,
							destinations)
						dest_object = editing_context.object fault_gid

						unless dest_object then
							dest_object = fault(fault_gid, editing_context)
						end
					end
				end

				object._init_stored_value(relation.name, dest_object)
			end
		end

		# Create a new channel if all existed channels are busy.
		def available_channel
			available = nil
			@channels.each do |channel|
				unless channel.locking? then
					available = channel
				end
			end

			unless available then
				channel = DatabaseChannel.new self
				register_channel channel
				available = channel
			end

			available
		end

		def register_channel( channel )
			@channels << channel
		end

		def unregister_channel( channel )
			@channels.delete channel
		end

		def objects_for_source_gid( src_gid, name, editing_context )
			objects = []

			if gids = @database.snapshot_for_source(src_gid, name) then
				gids.each do |gid|
					if object = editing_context.object(gid) then
						objects << object
					end
				end
			else
				entity   = @application.entity src_gid.entity_name
				rel      = entity.relationship name
				snapshot = @database.snapshot src_gid

				unless snapshot then
					raise "Can't find a snapshot for global ID: #{src_gid}"
				end

				values = {}
				rel.joins.each do |join|
					values[join.destination.name] = snapshot[join.source.name]
				end
				qualifier = Qualifier.new_to_match_all_values values
				fetchspec = FetchSpec.new(rel.destination, qualifier)
				objects = fetch(fetchspec, editing_context)
			end

			objects
		end

		def fetch( fetch_spec, editing_context )
			channel = available_channel
			channel.lock
			channel.select(fetch_spec, editing_context)
			objects = channel.fetch_all
			channel.unlock
			objects
		end

		def handle_fetch_spec?( fetch_spec )
			if @database.entity fetch_spec.entity_name then
				true
			else
				false
			end
		end

		def prepare_for_save( coordinator, editing_context )
			@coordinatior    = coordinator
			@editing_context = editing_context
			@snapshot_store  = SnapshotStore.new
			@inserted_objects.clear
			@deleted_objects.clear
			@updated_objects.clear
			@database_operations.clear
			@temp2key = {}

			db_channel = available_channel
			@adapter_channel = db_channel.adapter_channel

			@editing_context.inserted_objects.each do |object|
				entity       = object.class_description.entity
				gid          = @editing_context.gid object
				primary_keys = @adapter_channel.primary_keys_for_new_row(1, entity)

				# compound key
				if primary_keys.nil? then
					values = {}
					entity.primary_key_attribute_names.each do |name|
						values[name] = object[name]
					end
					@pk_values[gid] = values
				else
					@pk_values[gid] = primary_keys.first
				end

				new_gid         = KeyGlobalID.new(entity.name, @pk_values[gid])
				@temp2key[gid]  = new_gid
			end

			@editing_context.updated_objects.each do |object|
				gid = @editing_context.gid object
				unless own? gid then
					next
				end

				entity   = object.class_description.entity
				snapshot = @database.snapshot gid
				values   = {}

				# sets or updates values for primary keys
				entity.primary_key_attribute_names.each do |name|
					if entity.class_property? name then
						values[name] = object[name]
					else
						values[name] = snapshot[name]
					end
				end

				@pk_values[gid] = values
			end
		end

		def record_changes
			# insert, delete, update
			_record_changes(@editing_context.inserted_objects, @inserted_objects, \
				DatabaseOperation::DATABASE_INSERT_OPERATOR)
			_record_changes(@editing_context.deleted_objects, @deleted_objects, \
				DatabaseOperation::DATABASE_DELETE_OPERATOR)
			_record_changes(@editing_context.updated_objects, @updated_objects, \
				DatabaseOperation::DATABASE_UPDATE_OPERATOR)
		end

		private

		def _record_changes( changes, registers, db_operator )
			to_many_snapshots = {}
			changes.each do |object|
				if own? object then
					registers << object
					gid, _to_many_snapshots = _record_operations(object, db_operator)
					to_many_snapshots[gid] = _to_many_snapshots
				end
			end
		end

		def _record_operations( object, db_operator )
			gid = @editing_context.gid object
			entity = @database.entity object

			if (db_operator == DatabaseOperation::DATABASE_INSERT_OPERATOR) or \
				gid.temporary? then
				keyname = entity.primary_key_attribute_names.first
				value   = @pk_values[gid][keyname]
				gid     = KeyGlobalID.new(entity.name, @pk_values[gid])
			end

			dbop = DatabaseOperation.new(gid, object, entity)
			dbop.database_operator = db_operator

			# set primary key after creating snapshots
			snapshot, to_many_snapshots = _snapshot_to_db_operator_snapshots(
				object.snapshot, gid, object.to_many_relationship_keys, entity)
			if (db_operator == DatabaseOperation::DATABASE_INSERT_OPERATOR) then
				snapshot[keyname] = value
			elsif (db_operator == DatabaseOperation::DATABASE_UPDATE_OPERATOR) then
				snapshot.update @pk_values[gid]
			end

			record_snapshot(gid, snapshot)
			record_to_many_snapshots to_many_snapshots
			dbop.new_row = snapshot
			dbop.record_to_many_snapshots to_many_snapshots

			if db_operator == DatabaseOperation::DATABASE_INSERT_OPERATOR then
				dbop.db_snapshot = {}
			else
				dbop.db_snapshot = @database.snapshot gid
			end

			if db_operator == DatabaseOperation::DATABASE_UPDATE_OPERATOR then
				if dbop.row_diffs.empty? then
					dbop.database_operator = DatabaseOperation::DATABASE_NOTHING_OPERATOR
				end
			end

			@database_operations << dbop
			[gid, to_many_snapshots]
		end

		def _snapshot_to_db_operator_snapshots( object_snapshot,
		                                        gid, 
		                                        to_many_relationship_keys,
		                                        entity )
			to_many_snapshots = {}
			snapshot = object_snapshot.dup

			# replace all temporary global ID for all to-many objects
			to_many_relationship_keys.each do |key|
				to_many_snapshot = object_snapshot[key]

				unless FaultingDelayEvaluationArray === to_many_snapshot then
					to_many_snapshots[key] = to_many_snapshot

					to_many_snapshot_gids = to_many_snapshots[key].dup
					to_many_snapshot_gids.each do |temp_gid|
						if temp_gid.temporary? then
							to_many_snapshots[key].delete temp_gid
							to_many_snapshots[key] << @temp2key[temp_gid]
						end
					end
				end
				snapshot.delete key
			end

			snapshot.each do |key, value|
				if re = entity.relationship(key) then
					unless re.to_many? then
						snapshot.delete key
						snap_gid       = value
						src_attrs      = re.source_attributes
						src_attr_names = []
						src_attrs.each do |attr|
							src_attr_names << attr.name
						end
						pk_values = {}

						if snap_gid.nil? then
							src_attrs.each do |attr|
								dest_name = re.destination_attribute(attr).name
								pk_values[dest_name] = nil
							end
						elsif snap_gid.temporary? then
							pk_values = @pk_values[snap_gid]
						else
							pk_values = snap_gid.key_values
						end

						src_attrs.each_with_index do |attr, index|
							name = re.destination_attribute(attr).name
							snapshot[attr.name] = pk_values[name]
						end
					end
				end
			end

			# check all attributes
			entity.attributes.each do |attr|
				unless snapshot[attr.name] then
					snapshot[attr.name] = nil
				end
			end

			[snapshot, to_many_snapshots]
		end

		public

		def record_update( object, changes )
			dbop = nil
			if @database_operations.empty? then
				entity = @database.entity object.entity_name
				dbop = DatabaseOperation.new(object.gid, object, entity)
				dbop.database_operator = DATABASE_UPDATE_OPERATOR
				dbop.new_row = changes
				return
			end

			@database_operations.each do |op|
				if op.gid == object.gid then
					dbop = op
					break
				end
			end

			unless dbop then
				entity = @database.entity object.entity_name
				dbop = DatabaseOperation.new(object.gid, object, entity)
				dbop.database_operator = DATABASE_UPDATE_OPERATOR
				@database_operations << dbop

				snapshot = object.snapshot
				snapshot.update changes

				record_snapshot(object.gid, snapshot)

				dbop.new_row = snapshot
				dbop.db_snapshot = Snapshot.new
			else
				dbop.new_row.update changes
			end
		end

		# Returns true if the receiver is responsible for fetching and saving
		# the object (or GlobalID), false otherwise.
		def own?( object )
			if GlobalID === object then
				_own_gid? object
			else
				_own_object? object
			end
		end

		private

		def _own_gid?( gid )
			if gid.temporary? then
				false
			elsif @database.entity gid.entity_name then
				true
			else
				false
			end
		end

		def _own_object?( object )
			if @database.entity object.entity_name then
				true
			else
				false
			end
		end

		public

		def perform_changes
			adapter_operations = []
			@database_operations.each do |dbop|
				adops = _create_adapter_operations dbop
				dbop.adapter_operations = adops
				adapter_operations.concat adops
			end

			@adapter_channel.adapter_context.begin_transaction
			@adapter_channel.perform_adapter_operations adapter_operations
		end

		private

		def _create_adapter_operations( database_operation )
			adops = nil
			case database_operation.database_operator
			when DatabaseOperation::DATABASE_NOTHING_OPERATOR
				adops = []
			when DatabaseOperation::DATABASE_INSERT_OPERATOR
				adop = AdapterOperation.new(database_operation.entity, \
				       DatabaseOperation::ADAPTER_INSERT_OPERATOR)
				adop.changed_values = database_operation.new_row
				adops = [adop]
			when DatabaseOperation::DATABASE_DELETE_OPERATOR
				adop = AdapterOperation.new(database_operation.entity, \
				       DatabaseOperation::ADAPTER_DELETE_OPERATOR)
				entity         = database_operation.entity
				gid            = database_operation.gid
				qualifier      = entity.qualifier_for_primary_key gid.key_values
				adop.qualifier = qualifier
				adops = [adop]
			when DatabaseOperation::DATABASE_UPDATE_OPERATOR
				if row_diffs = database_operation.row_diffs then
					adop = AdapterOperation.new(database_operation.entity, \
				          DatabaseOperation::ADAPTER_UPDATE_OPERATOR)
					adop.changed_values = row_diffs
					entity              = database_operation.entity
					gid                 = database_operation.gid
					qualifier           = entity.qualifier_for_primary_key gid.key_values
					adop.qualifier      = qualifier
					adops               = [adop]
				end
			else
				raise 'Unknown operation.'
			end

			adops
		end

		public

		def commit
			@adapter_channel.adapter_context.commit_transaction
		end

		def rollback
			@adapter_channel.adapter_context.rollback_transaction
		end

		# * Delete snapshots of deleted objects
		# * Set timestamp to now
		# * Register new snapshots
		# * Update global IDs for new records
		# * Notificate GID_CHANGED_NOTIFICATION
		# * End the transaction
		def finalize_commit_changes
			@database.forget_snapshots _deleted_gids
			@database.set_timestamp_to_now
			@database.record_snapshots @snapshot_store.snapshots
			@database.record_to_many_snapshots @snapshot_store.to_many_snapshots

			@inserted_objects.each do |object|
				gid      = @editing_context.gid object
				key_gid  = @temp2key[gid]
				snapshot = @snapshot_store.snapshot key_gid
				userinfo = {gid => key_gid, :snapshot => snapshot}
				@notification_center.post(GlobalID::GID_CHANGED_NOTIFICATION, gid, userinfo)
			end

			# _end_transaction
		end

		def set_timestamp_to_now
			@snapshot_store.set_timestamp_to_now
		end

		private

		def _deleted_gids
			gids = []
			@deleted_objects.each do |object|
				if gid = @editing_context.gid(object) then
					gids << gid
				end
			end
			gids
		end

		public

		##
		## locking thread
		##

		# Not yet implemented.
		def lock; end

		# Not yet implemented.
		def unlock; end

		##
		## faulting
		##

		def fault( gid, editing_context )
			description = ClassDescription.new_with_entity_name(@application,
				gid.entity_name)

			if object = editing_context.object(gid) then
				return object
			end

			object = description.create editing_context
			handler = AccessFaultHandler.new(gid, self, editing_context)
			object.turn_into_fault handler

			editing_context.record(gid, object)
			@notification_center.add(editing_context, :handle_notification,
				GlobalID::GID_CHANGED_NOTIFICATION, gid)

			object
		end

		def array_fault( gid, relationship_name, editing_context )
			FaultingDelayEvaluationArray.new(gid, relationship_name, self, editing_context)
		end

		def fault_for_raw_row( row, entity_name, editing_context )
			entity = @database.entity entity_name
			pks = {}
			entity.primary_key_names.each do |name|
				pks[name] = row[name]
			end
			gid = KeyGlobalID.new(entity_name, pks)
			fault(gid, editing_context)
		end

		def refault( object, gid, editing_context )
			object.to_one_relationship_keys.each do |key|
				if destination = object[key] then
					# forget snapshots
					dest_gid = editing_context.gid destination
					forget_snapshot dest_gid
					@database.forget_snapshot dest_gid

					handler = AccessFaultHandler.new(dest_gid, self, editing_context)
					destination.turn_into_fault handler
				end
			end

			# restruct objects
			object.to_many_relationship_keys.each do |key|
				destinations = object[key]
				if FaultingDelayEvaluationArray === destinations then
					destinations.each do |destination|
						dest_gid = editing_context.gid destination
						forget_snapshot dest_gid
						@database.forget_snapshot dest_gid
						editing_context.forget destination
					end

					handler = AccessArrayFaultHandler.new(gid, key, self, editing_context)
					destinations.turn_into_fault handler
				end
			end
		end


		##
		## snapshots
		##

		def record_snapshot( gid, snapshot )
			@snapshot_store.record_snapshot(gid, snapshot)
		end

		def record_to_many_snapshots( snapshots )
			@snapshot_store.record_to_many_snapshots snapshots
		end

		def snapshot( gid, time = nil )
			@snapshot_store.snapshot(gid, time)
		end

		def snapshot_for_source( gid, name, time = nil )
			@snapshot_store.snapshot_for_source(gid, name, time)
		end

		def forget_snapshots( gids )
			@snapshot_store.forget_snapshots gids
		end

		def forget_snapshot( gid )
			@snapshot_store.forget_snapshot gid
		end
	end

end
