module TapKit
	class SQLExpression
		NULL = 'NULL'

		attr_reader :aliases
		attr_reader :entity
		attr_reader :join_semantic
		attr_reader :join_entities
		attr_reader :join_on
		attr_reader :join_relationships
		attr_reader :list_string
		attr_reader :lock_clause
		attr_reader :group_by_string
		attr_reader :order_by_string
		attr_reader :where_clause
		attr_reader :value_list
		attr_reader :bind_variables
		attr_writer :use_aliases
		attr_writer :use_bind_variables
		attr_accessor :statement
		attr_accessor :encoding

		NAME_KEY        = :name_key
		PLACEHOLDER_KEY = :placeholder_key
		ATTRIBUTE_KEY   = :attribute_key
		VALUE_KEY       = :value_key
		READ_FORMAT     = '%P'
		WRITE_FORMAT    = '%V'

		def initialize( entity, encoding = nil )
			@encoding           = encoding
			@entity             = entity
			@list_string        = ''
			@value_list         = ''
			@bind_variables     = []
			@aliases            = {'' => 't0'} # path -> another_name (alias)
			@table_size         = 0
			@where_clause       = ''
			@join_clause        = ''
			@join_on            = ''
			@join_relationships = {} # path -> relationship
			@join_entities      = [] # item -> [entity, another_name]
			@use_aliases        = true
			@use_bind_variables = true
		end


		##
		## bind variables
		##

		def add_bind_variable( binding )
			@bind_variables << binding
		end

		# abstract
		def bind_variable( attribute, value ); end
		def must_use_bind_variable?( attribute ); end
		def should_use_bind_variable?( attribute ); end

		def use_bind_variables?
			@use_bind_variables
		end

		def use_aliases?
			@use_aliases
		end

		##
		## list string
		##

		def append_item( string, list = nil )
			list ||= @list_string
			unless list.empty? then
				list << ", #{string}"
			else
				list << string
			end
		end

		def table_list( entity, join = true )
			list = nil
			if use_aliases? then
				if join == true then
					name = entity.external_name
					if @entity = entity then
						another = @aliases['']
					else
						another = @aliases[name]
					end
					list = "#{name} #{another}"
				else
					list = _table_list_by_aliases entity
				end
			else
				list = entity.external_name
			end

			list
		end

		private

		def _table_list_by_aliases( entity )
			list = ''
			externals = {}

			@aliases.each do |keypath, another|
				if keypath == '' then
					name = entity.external_name
				else
					last_entity = entity.destination_entity keypath
					name = last_entity.external_name
				end

				externals[name] = another
			end

			external_array = externals.sort do |a, b|
				alias1 = a[0]
				alias2 = a[1]

				if alias1 > alias2 then
					-1
				elsif alias1 < alias2 then
					1
				else
					0
				end
			end

			external_array.each do |item|
				table = item[0]
				name  = item[1]

				list << "#{table} #{name}, "
			end
			list.chop!
			list.chop!

			list
		end

		public

		##
		## order by string
		##

		def add_order( sort_ordering )
			attr = @entity.attribute sort_ordering.key
			name = sql_for_attribute attr

			case sort_ordering.symbol
			when SortOrdering::ASC
				ordering = 'ASC'
			when SortOrdering::DESC
				ordering = 'DESC'
			when SortOrdering::CI_ASC
				ordering = 'ASC'
				name = "UPPER(#{name})"
			when SortOrdering::CI_DESC
				ordering = 'DESC'
				name = "UPPER(#{name})"
			end

			if @order_by_string then
				@order_by_string << ", #{name} #{ordering}"
			else
				@order_by_string = "#{name} #{ordering}"
			end
		end


		##
		## delete
		##

		# Generates a DELETE statement by the following steps:
		#
		# 1. Generates where_clause.
		# 2. Invokes table_list to get the table name for the FROM clause..
		# 3. Invokes assemble_delete.
		def prepare_delete( qualifier )
			@use_aliases  = false
			@where_clause = sql_for_qualifier qualifier
			tables        = table_list @entity
			@statement    = assemble_delete(qualifier, tables, where_clause)
			@use_aliases  = true
		end

		#  DELETE FROM table_list WHERE where_clause
		def assemble_delete( qualifier, table_list, where_clause )
			"DELETE FROM #{table_list} WHERE #{where_clause}"
		end


		##
		## join
		##

		def join_clause
			if @join_entities.empty? then
				return ''
			end

			clause = "#@join_semantic JOIN "
			@join_entities.each do |entity|
				name    = entity[0].external_name
				another = entity[1]
				clause << "#{name} #{another}, "
			end
			clause.chop!
			clause.chop!

			unless @join_on.empty? then
				clause << " ON #@join_on"
			end

			clause
		end

		def join_expression
			if @aliases.size == 1 then
				return
			end

			@aliases.each do |path, another|
				if relationship = @join_relationships[path] then
					relationship.joins.each do |join|
						_join_expression(relationship, join)
					end
				end
			end
		end

		private

		def _join_expression( relationship, join )
			left = right = nil

			if set = @join_entities.assoc(join.source.entity) then
				another = set[1]
				left = "#{another}.#{join.source.name}"
			elsif @entity == join.source.entity then
				left = "#{@aliases['']}.#{join.source.name}"
			end

			if set = @join_entities.assoc(join.destination.entity) then
				another = set[1]
				right = "#{another}.#{join.destination.name}"
			elsif @entity == join.destination.entity then
				right = "#{@aliases['']}.#{join.destination.name}"
			end

			add_join(left, right, relationship.join_semantic)
		end

		public

		def add_join( left, right, semantic )
			clause = assemble_join(left, right, semantic)

			unless @join_semantic then
				case semantic
				when Relationship::INNER_JOIN
					@join_semantic = "INNER"
				when Relationship::FULL_OUTER_JOIN
					@join_semantic = "FULL OUTER"
				when Relationship::LEFT_OUTER_JOIN
					@join_semantic = "LEFT OUTER"
				when Relationship::RIGHT_OUTER_JOIN
					@join_semantic = "RIGHT OUTER"
				end
			end

			if @join_on.empty? then
				@join_on = clause
			else
				@join_on << " AND #{clause}"
			end

			clause
		end

		def assemble_join( left, right, semantic )
			"#{left} = #{right}"
		end


		##
		## select
		##

		# Generates a SELECT statement by the following steps:
		#
		# 1. Invokes add_select_list() for each in the attributes.
		# 2. Uses qualifier of the fetch_spec to generate where_clause.
		# 3. Invokes add_order() for each AttributeOrdering object in the fetch_spec.
		# 4. Invokes join_expression() to generate join_clause.
		# 5. Invokes table_list() to generate FROM clause.
		# 6. If the lock is true, invokes lock_clause to get the SQL string
		#    for locking selected rows.
		# 7. Invokes assemble_select(). After this, you can get
		#    the complete SELECT statement by calling statement().
		def prepare_select( attributes, lock, fetch_spec )
			attributes.each do |attr|
				add_select_list attr
			end

			@where_clause = sql_for_qualifier fetch_spec.qualifier

			fetch_spec.sort_orderings.each do |order|
				add_order order
			end

			join_expression
			tables  = table_list @entity
			locking = lock_clause # unsupported
			select  = nil # 'DISTINCT', etc.

			sql = assemble_select(attributes, lock, fetch_spec.qualifier,
				fetch_spec.sort_orderings, select, @list_string, tables, join_clause,
				where_clause, order_by_string, lock_clause)

			@statement = sql
		end

		# attributes, lock, qualifier, fetchorder are unsupported.
		#
		#  SELECT column_list FROM table_list lock_clause
		#  join_semantic JOIN join_clause WHERE where_clause ORDER BY order_clause
		def assemble_select( attributes, lock, qualifier, fetch_order, select,
			column_list, table_list, join_clause, where_clause, order_by, lock_clause )
			sql =  "SELECT"
			sql << " #{select}" if select
			sql << " #{column_list}"
			sql << " FROM #{table_list}"
			sql << " #{join_clause}"        unless join_clause.empty?
			sql << " WHERE #{where_clause}" if where_clause
			sql << " ORDER BY #{order_by}"  if order_by
			sql
		end

		def add_select_list( attribute, as = nil, grouping = false )
			column = sql_for_attribute attribute
			item   = format_sql_string(column, attribute.read_format)
			@group_by_string = item.dup if grouping
			item << " #{as}" if as
			append_item item
		end


		##
		## aggregate
		##

		def prepare_aggregate( aggregate_spec )
			if aggregate_spec.group_by then
				attr = @entity.attribute aggregate_spec.group_by
				add_select_list(attr, attr.name, true)
			else
				group_by = nil
			end

			aggregate_spec.attributes.each do |agg_attr|
				attr = @entity.attribute agg_attr[:name]
				add_select_list_with_function(attr, agg_attr[:function], agg_attr[:key])
			end

			@where_clause = sql_for_qualifier aggregate_spec.qualifier

			# not support yet
			# having_clause = sql_for_qualifier aggregate_spec.having
			having_clause = nil

			aggregate_spec.sort_orderings.each do |order|
				add_order order
			end

			join_expression
			tables  = table_list @entity
			select  = nil # 'DISTINCT', etc.

			sql = assemble_aggregate(aggregate_spec.qualifier,
				aggregate_spec.having, aggregate_spec.sort_orderings, select,
				@list_string, tables, join_clause,
				where_clause, group_by_string, having_clause, order_by_string)

			@statement = sql
		end

		def add_select_list_with_function( attribute, function, as = nil )
			column = sql_for_attribute attribute
			item   = format_sql_string(column, attribute.read_format)
			item   = sql_for_function(function, item)
			item << " #{as}" if as
			append_item item
		end

		#  SELECT group_by column_list FROM table_list join_semantic JOIN join_clause
		#  WHERE where_clause GROUP BY group_by HAVING having_clause ORDER BY order_clause
		def assemble_aggregate( qualifier, having_qualifier, fetch_order,
			select, column_list, table_list, join_clause, where_clause, group_by,
			having_clause, order_by )
			sql =  "SELECT"
			sql << " #{select}" if select
			sql << " #{column_list}"
			sql << " FROM #{table_list}"
			sql << " #{join_clause}"          unless join_clause.empty?
			sql << " WHERE #{where_clause}"   if where_clause
			sql << " GROUP BY #{group_by}"    if group_by
			sql << " HAVING #{having_clause}" if having_clause
			sql << " ORDER BY #{order_by}"    if order_by
			sql
		end


		##
		## update
		##

		# Generates a UPDATE statement by the following steps:
		#
		# 1. Invokes add_update_list_attribute for each entry in row
		#    to prepare the comma-separated list of "attribute = value" assignments.
		# 2. Generates where_clause.
		# 3. Invokes table_list to get the table name for the FROM clause.
		# 4. Invokes assemble_update.
		def prepare_update( row, qualifier )
			@use_aliases = false

			row.each do |key, value|
				attr = @entity.attribute key
				add_update_list(attr, value)
			end

			@where_clause = sql_for_qualifier qualifier
			tables        = table_list @entity
			@statement    = assemble_update(
				row, qualifier, tables, @list_string, where_clause)
			@use_aliases  = true
		end

		def add_update_list( attribute, value )
			column    = sql_for_attribute attribute
			sql_name  = format_sql_string(column, attribute.write_format)
			sql_value = sql_for_value(attribute.name, value)
			str       = "#{sql_name} = #{sql_value}"
			append_item str
		end

		#  UPDATE table_list SET update_list WHERE where_clause
		def assemble_update( row, qualifier, table_list, update_list, where_clause )
			"UPDATE #{table_list} SET #{update_list} WHERE #{where_clause}"
		end


		##
		## insert
		##

		# Generates a INSERT statement by the following steps:
		#
		# 1. Invokes add_insert_list_attribute for each entry in row
		#    to prepare the comma-separated list of attributes and
		#    the corresponding list of values.
		# 2. Invokes table_list to get the tables name for the FROM clause.
		# 3. Invokes assemble_insert.
		def prepare_insert( row )
			@use_aliases = false

			row.each do |key, value|
				attr = @entity.attribute key
				add_insert_list(attr, value)
			end

			tables       = table_list @entity
			@statement   = assemble_insert(row, tables, @list_string, value_list)
			@use_aliases = true
		end

		def assemble_insert( row, table_list, column_list, value_list )
			sql = "INSERT INTO #{table_list} (#{column_list}) VALUES (#{value_list})"
			@statement = sql
		end

		def add_insert_list( attribute, value )
			column = sql_for_attribute attribute
			item   = format_sql_string(column, attribute.write_format)
			append_item item

			sql_value = sql_for_value(attribute.name, value)
			unless @value_list.empty? then
				@value_list << ", #{sql_value}"
			else
				@value_list = sql_value
			end
		end


		##
		## format
		##

		# abstract
		def format_value( value, attribute ); end

		def format_sql_string( sql, format = nil )
			unless format then
				return sql
			end

			formatted = format.dup
			formatted.gsub!(READ_FORMAT, sql)
			formatted.gsub!(WRITE_FORMAT, sql)
			formatted
		end

		##
		## SQL string
		##

		def sql_escape_char
			'\\'
		end

		# attribute or name
		def sql_for_attribute( attribute )
			column = attribute.column_name
			if use_aliases? then
				"#{@aliases['']}.#{column}"
			else
				column
			end
		end

		def sql_for_attribute_named( name )
			if name.include? '.' then
				paths = name.split '.'
				sql_for_attribute_path paths
			else
				attr   = @entity.attribute name
				column = attr.column_name
				if use_aliases? then
					"#{@aliases['']}.#{column}"
				else
					column
				end
			end
		end

		def sql_for_attribute_path( paths )
			last_entity = @entity
			last_attr = nil
			re_path = ''
			re = nil
			paths.each_with_index do |path, index|
				if (index + 1) == paths.size then
					last_attr = last_entity.attribute path
					re_path.chop!
					_add_new_alias(re_path, re) if use_aliases?
					_add_join_relationships(re_path, re)
				else
					re = last_entity.relationship(path)
					last_entity = re.destination_entity

					re_path << "#{path}."
					_add_new_alias(path, re) if use_aliases?
					_add_join_relationships(re_path.chop, re)
				end
			end

			if use_aliases? then
				another = @aliases[re_path]
			else
				another = nil
			end

			if another then
				"#{another}.#{last_attr.column_name}"
			else
				"#{@last_entity.external_name}.#{last_attr.column_name}"
			end
		end

		private

		def _add_join_relationships( path, relationship )
			@join_relationships[path] = relationship
		end

		def _add_new_alias( path, relationship )
			if @aliases[path].nil? and (@entity.name != path) then
				entity = relationship.destination_entity
				another = "t#@table_size"

				unless @join_entities.assoc entity then
					@table_size += 1
					another = "t#@table_size"
					@join_entities << [entity, another]
				end
				@aliases[path] = another
			end
		end

		public

		def sql_for_value( keypath, object )
			attribute = @entity.destination_attribute keypath

			if must_use_bind_variable?(attribute) or \
				should_use_bind_variable?(attribute) then
				binding = bind_variable(attribute, object)
				binding[NAME_KEY] = keypath
				add_bind_variable binding
				return binding[PLACEHOLDER_KEY]
			end

			format_value(object, attribute)
		end

		def sql_for_string( string )
			if string.nil? then
				'NULL'
			else
				if @encoding then
					string = Utilities.encode(string, @encoding)
				end
				escape = Regexp.escape sql_escape_char
				string = string.to_s.gsub(/'/) { "#{escape}'" }
				"'#{string}'"
			end
		end

		def sql_for_date( date )
			if date.nil? then
				'NULL'
			else
				date.to_s
			end
		end

		def sql_for_number( number )
			if number.nil? then
				'NULL'
			else
				number.to_s
			end
		end

		def sql_for_case_insensitive_like( value, key )
			"UPPER(#{key}) LIKE UPPER(#{value})"
		end

		def sql_for_function( function, column_name )
			case function
			when AggregateSpec::AVG then
				sql_for_average_function(column_name)
			when AggregateSpec::MAX then
				sql_for_max_function(column_name)
			when AggregateSpec::MIN then
				sql_for_min_function(column_name)
			when AggregateSpec::COUNT then
				sql_for_count_function(column_name)
			when AggregateSpec::COUNT_ALL then
				sql_for_count_all_function(column_name)
			end
		end

		def sql_for_average_function( column_name )
			"AVG(#{column_name})"
		end

		def sql_for_max_function( column_name )
			"MAX(#{column_name})"
		end

		def sql_for_min_function( column_name )
			"MIN(#{column_name})"
		end

		def sql_for_count_function( column_name )
			"COUNT(#{column_name})"
		end

		def sql_for_count_all_function( column_name )
			"COUNT(*)"
		end


		##
		## qualifier
		##

		#  *  -> %
		#  ?  -> _
		#  %  -> \%
		#  _  -> \_
		#  \* -> *
		#  \? -> ?
		def sql_pattern( pattern, escape = nil )
			unless escape then
				escape = sql_escape_char
			end

			translated = pattern.dup
			translated.gsub!(/(\\\*|\*|\\\?|\?|%|_)/) do
				case $1
				when '\*' then '*'
				when '*'  then '%'
				when '\?' then '?'
				when '?'  then '_'
				when '%'  then "#{escape}%"
				when '_'  then "#{escape}_"
				end
			end

			translated
		end

		def sql_for_symbol( symbol, value )
			case symbol
			when Qualifier::EQUAL
				if value.nil? then 'IS'
				else               '='
				end
			when Qualifier::NOT_EQUAL
				if value.nil? then 'IS NOT'
				else               '<>'
				end
			when Qualifier::GREATER          then '>'
			when Qualifier::GREATER_OR_EQUAL then '>='
			when Qualifier::LESS             then '<'
			when Qualifier::LESS_OR_EQUAL    then '<='
			when Qualifier::LIKE             then 'LIKE'
			else
				raise ArgumentError, "Symbol #{symbol} is an unknown operator"
			end
		end

		# key - sql_for_attribute_named, format_sql_string
		# value - sql_for_value, format_value(sql_pattern)
		#  <key> <operator> <value>
		def sql_for_key_value_qualifier( qualifier )
			key   = qualifier.key
			value = qualifier.value
			op    = qualifier.symbol

			# operator
			unless op == Qualifier::CI_LIKE then
				op_str = sql_for_symbol(op, value)
			end

			# key
			key_str = sql_for_attribute_named key			
			key_str = format_sql_string key_str

			# value
			if (op == Qualifier::LIKE) or (op == Qualifier::CI_LIKE) then
				value = sql_pattern value
			end
			value_str = sql_for_value(key, value)

			unless op == Qualifier::CI_LIKE then
				"#{key_str} #{op_str} #{value_str}"
			else
				sql_for_case_insensitive_like(value_str, key_str)
			end
		end

		# left - sql_for_attribute_named, format_sql_string
		#  <left> <operator> <right>
		def sql_for_key_comparison_qualifier( qualifier )
			left  = qualifier.left
			right = qualifier.right
			op    = qualifier.symbol

			# operator
			unless op == Qualifier::CI_LIKE then
				op_str = sql_for_symbol(op, true) # second argument is dummy
			end

			left_str  = sql_for_attribute_named left
			right_str = sql_for_attribute_named right

			unless op == Qualifier::CI_LIKE then
				"#{left_str} #{op_str} #{right_str}"
			else
				sql_for_case_insensitive_like(right_str, left_str)
			end
		end

		#  (<qualifier> AND <qualifier> AND ... <qualifier>)
		def sql_for_conjoined_qualifiers( qualifiers )
			sql = '('
			qualifiers.each do |qualifier|
				sql << "#{sql_for_qualifier(qualifier)} AND "
			end
			sql = sql[0, (sql.size-5)] + ')'
			sql
		end

		#  (<qualifier> OR <qualifier> OR ... <qualifier>)
		def sql_for_disjoined_qualifiers( qualifiers )
			sql = '('
			qualifiers.each do |qualifier|
				sql << "#{sql_for_qualifier(qualifier)} OR "
			end
			sql = sql[0, (sql.size-4)] + ')'
			sql
		end

		#  NOT (<qualifier>)
		def sql_for_negated_qualifier( qualifier )
			"NOT (#{sql_for_qualifier(qualifier)})"
		end

		def sql_for_qualifier( qualifier )
			case qualifier
			when KeyValueQualifier
				sql_for_key_value_qualifier qualifier
			when KeyComparisonQualifier
				sql_for_key_comparison_qualifier qualifier
			when AndQualifier
				sql_for_conjoined_qualifiers qualifier.qualifiers
			when OrQualifier
				sql_for_disjoined_qualifiers qualifier.qualifiers
			when NotQualifier
				sql_for_negated_qualifier qualifier.qualifier
			else
				nil
			end
		end

		def to_s
			str = "#{statement}"

			str << " [" unless @bind_variables.empty?
			@bind_variables.each do |binding|
				value = binding[VALUE_KEY]
				str << "#{value.inspect}, "
			end

			unless @bind_variables.empty? then
				str.chop!
				str.chop!
				str << ']' 
			end

			str
		end
	end


	class SQLExpressionFactory
		attr_reader :adapter

		def initialize( adapter )
			@adapter = adapter
			if @adapter.model then
				@encoding = @adapter.model.connection['encoding']
			end
		end

		def expression_class
			SQLExpression
		end

		def create( entity )
			expression_class.new entity
		end

		def expression( string )
			expr = expression_class.new nil
			expr.statement = string
			expr
		end

		def delete_statement( qualifier, entity )
			expr = expression_class.new entity
			expr.prepare_delete qualifier
			expr
		end

		def insert_statement( row, entity )
			expr = expression_class.new(entity, @encoding)
			expr.prepare_insert row
			expr
		end

		def select_statement( attributes, lock, fetch_spec, entity )
			expr = expression_class.new(entity, @encoding)
			expr.prepare_select(attributes, lock, fetch_spec)
			expr
		end

		def update_statement( row, qualifier, entity )
			expr = expression_class.new(entity, @encoding)
			expr.prepare_update(row, qualifier)
			expr			
		end

		def aggregate_statement( entity, aggregate_spec )
			expr = expression_class.new(entity, @encoding)
			expr.prepare_aggregate aggregate_spec
			expr
		end
	end

end

