module TapKit

	class EditingContext < ObjectStore
		include Utilities, Aliasable

		attr_reader :inserted_objects, :deleted_objects, :registered_objects, :updated_objects
		attr_reader :parent_object_store, :root_object_store
		attr_accessor :shared_editing_context
		attr_reader :gid_store

		def initialize( parent )
			@parent_object_store = parent
			@inserted_objects    = [] # GlobalID -> object
			@deleted_objects     = []
			@updated_objects     = []
			@registered_objects  = []
			@gid_store           = GlobalIDStore.new

			if EditingContext === @parent_object_store then
				@root_object_store = @parent_object_store.root_object_store
			else
				@root_object_store = @parent_object_store
			end
			@application = @root_object_store.application
		end

		def changed?
			if @inserted_objects.empty? and @deleted_objects.empty? and \
				@updated_objects.empty? then
				false
			else
				true
			end
		end

		def initialize_object( object, gid, editing_context = nil )
			editing_context ||= self
			@parent_object_store.initialize_object(object, gid, editing_context)
		end

		def object_will_change( object )
			if gid(object) then
				unless @inserted_objects.include?(object) or \
					@deleted_objects.include?(object) or \
					@updated_objects.include?(object) then
					@updated_objects << object
				end
			end
		end

		def insert( object, gid = nil )
			unless @inserted_objects.include? object then
				if @deleted_objects.include? object then
					@deleted_objects.delete object
					forget object
				else
					unless gid then
						unless gid = self.gid(object) then
							gid = TemporaryGlobalID.new
							object.application = @application

							record(gid, object)
							nc.add(self, :handle_notification,
								ObjectStore::OBJECTS_CHANGED_IN_STORE_NOTIFICATION)
						end
					end

					@inserted_objects << object
					record(gid, object)
					object.awake_from_insertion self
				end
			end
		end

		def delete( object )
			unless gid = self.gid(object) then
				raise "Object is not registered"
			end

			if object.fault? then
				object.will_read
			end

			unless @deleted_objects.include? object then
				if @inserted_objects.include? object then
					@inserted_objects.delete object
					forget object
				else
					@deleted_objects << object
				end
			end
		end

		def record( gid, object )
			@registered_objects << object
			@gid_store.add(gid, object)

			# observer, notification
			@application.observer_center.add(self, object)
			@application.notification_center.add(self,
				:handle_notification, GlobalID::GID_CHANGED_NOTIFICATION)
		end

		def forget( object )
			@registered_objects.delete object
			@gid_store.forget object
		end

		def handle_notification( notification )
			case notification.name
			when GlobalID::GID_CHANGED_NOTIFICATION
				_handle_gid_changed_notification notification
			when ObjectStore::OBJECTS_CHANGED_IN_STORE_NOTIFICATION
				_handle_objects_changed_in_store_notification notification
			else
				raise "EditingContext got an unhandled notification: #{notification.name}"
			end
		end

		private

		def _handle_gid_changed_notification( notification )
			gid    = notification.object
			object = self.object gid

			if gid.temporary? and object then
				key_gid = notification.userinfo[gid]
				forget object
				record(key_gid, object)

				snapshot = notification.userinfo[:snapshot]
				object.update_from_snapshot snapshot
			else
				new_gid = notification.userinfo[gid]

				if object and (object.fault? == false) then
					return
				elsif object and new_gid then
					# @resolved_faults[gid] = new_gid
					forget gid
					record(new_gid, object)
					nc.remove(self, GID_CHANGED_NOTIFICATION, gid)
				end
			end
		end

		def _handle_objects_changed_in_store_notification( notification )
			context  = notification.object
			gid      = notification.userinfo[:gid]
			snapshot = notification.userinfo[:snapshot]
			object   = self.object gid
			object.update_from_snapshot(snapshot)
		end

		public

		def save
			unless changed? then
				return
			end

			process_recent_changes

			# validate changes
			@inserted_objects.each { |object| object.validate_for_insert }
			@updated_objects.each  { |object| object.validate_for_update }
			@deleted_objects.each  { |object| object.validate_for_delete }

			@parent_object_store.save self

			# finalize for save
			@deleted_objects.each { |object| forget object }
			@inserted_objects.clear
			@deleted_objects.clear
			@updated_objects.clear
		end

		def process_recent_changes
			@deleted_objects.each do |object|
				@inserted_objects.delete object
				@updated_objects.delete object
				object.propagate_delete self
			end

			_remove_objects_with_0_for_primary_key @inserted_objects
			_remove_objects_with_0_for_primary_key @deleted_objects
			_remove_objects_with_0_for_primary_key @updated_objects
		end

		private

		def _remove_objects_with_0_for_primary_key( objects )
			zero_objects = []
			objects.each do |object|
				gid = self.gid object

				unless gid.temporary? then
					pk_values = gid.key_values
					if (pk_values.keys.size == 1) and (pk_values.values[0] == 0) then
						zero_objects << object
					end
				end
			end

			zero_objects.each do |object|
				objects.delete object
			end
		end

		public

		def fetch( fetch_spec, editing_context = nil )
			if AggregateSpec === fetch_spec then
				return fetch_with_aggregate_spec(fetch_spec)
			end

			editing_context ||= self
			objects = @parent_object_store.fetch(fetch_spec, editing_context)

			if fetch_spec.qualifier then
				objects = Qualifier.filter(objects, fetch_spec.qualifier)
			end

			sorts = fetch_spec.sort_orderings
			unless sorts.empty? then
				objects = SortOrdering.sort(objects, sorts)
			end

			objects
		end

		def fetch_with_aggregate_spec( agg_spec )
			store = osc.object_store_for_fetch_spec agg_spec
			dbchannel = store.available_channel
			adchannel = dbchannel.adapter_channel

			entity = mg.entity agg_spec.entity_name
			factory = store.database.adapter.expression_factory
			expr = factory.aggregate_statement(entity, agg_spec)

			dbchannel.open unless dbchannel.open?
			adchannel.evaluate_for_aggregate_spec(agg_spec, expr)
		end

		def gid( object )
			@gid_store.gid object
		end

		def object( gid )
			@gid_store.object gid
		end


		##
		## faulting
		##

		def fault( gid, editing_context = self )
			@parent_object_store.fault(gid, editing_context)
		end

		def array_fault( gid, relationship_name, editing_context = self )
			@parent_object_store.array_fault(gid, relationship_name, editing_context)
		end

		def fault_for_raw_row( row, entity_name, editing_context = self )
			@parent_object_store.fault_for_raw_row(row, entity_name, editing_context)
		end

		def refault( object, gid, editing_context = self )
			@parent_object_store.refault(object, gid, editing_context)
		end

		def objects_for_source_gid( gid, name, editing_context = self )
			@parent_object_store.objects_for_source_gid(gid, name, editing_context)
		end
	end

	# EditingContext for read only.
	class SharedEditingContext < EditingContext
		def shared_editing_context
			nil
		end

		def save
			_raise_error
		end

		private

		def _raise_error
			raise "Can't modify objects in a SharedEditingContext object."
		end
	end
end
