
class TestAttribute < Test::Unit::TestCase
	include TapKit

	def setup
		@app = Application.new MODELS
		@ec  = @app.create_editing_context
	end

	def test_validate_value
		entity = @app.model_group.entity 'Book'
		attr = entity.attribute 'title'

		# allow_null
		assert_raises(ValidationError) do
			attr.validate_value nil
		end

		# read_only
		attr.read_only = true
		assert_raises(ValidationError) { attr.validate_value 'test' }
		attr.read_only = false

		# width
		attr.width = 1
		assert_raises(ValidationError) { attr.validate_value 'test' }
	end

	def test_validate_required_attributes
		attribute = Attribute.new
		assert_raises(ValidationError) { attribute.validate_required_attributes }
	end

	# mapping test
	def test_convert_attribute_value
		entity = @app.model_group.entity 'Book'
		attr = entity.attribute 'title'
		origin = attr.class_name

		tests = []
		tests << {:class_name => 'String', :value => 'test', :expected => 'test'}
		tests << {:class_name => 'String', :value => 100, :expected => '100'}
		tests << {:class_name => 'Integer', :value => 100, :expected => 100}
		tests << {:class_name => 'Integer', :value => '100', :expected => 100}
		tests << {:class_name => 'Float', :value => 0.5, :expected => 0.5}
		tests << {:class_name => 'Float', :value => '0.5', :expected => 0.5}
		tests << {:class_name => 'BigDecimal', :value => '0.5', :expected => '0.5'}

		date  = Date.new(2003,1,1)
		_date = ::Date.new(2003,1,1)
		tests << {:class_name => 'Date', :value => date, :expected => date}
		tests << {:class_name => 'Date', :value => _date, :expected => date}

		time  = Time.new(12,30,30)
		_time = ::Time.local(2000,1,1,12,30,30)
		tests << {:class_name => 'Time', :value => time, :expected => time}
		tests << {:class_name => 'Time', :value => _time, :expected => time}

		stamp   = Timestamp.new(2003,1,1,12,30,30)
		_stamp1 = DateTime.new(2003,1,1,12,30,30)
		_stamp2 = ::Time.local(2003,1,1,12,30,30)
		tests << {:class_name => 'Timestamp', :value => stamp, :expected => stamp}
		tests << {:class_name => 'Timestamp', :value => _stamp1, :expected => stamp}
		tests << {:class_name => 'Timestamp', :value => _stamp2, :expected => stamp}

		tests << {:class_name => 'Boolean', :value => true, :expected => true}
		tests << {:class_name => 'Boolean', :value => 'true', :expected => true}
		tests << {:class_name => 'Boolean', :value => 1, :expected => true}
		tests << {:class_name => 'Boolean', :value => false, :expected => false}
		tests << {:class_name => 'Boolean', :value => 'false', :expected => false}
		tests << {:class_name => 'Boolean', :value => 0, :expected => false}
		tests << {:class_name => 'Boolean', :value => 'dummy', :expected => nil}

		e = ArgumentError
		tests << {:class_name => 'Date', :raise => true, :value => 'dummy', :expected => e}
		tests << {:class_name => 'Time', :raise => true, :value => 'dummy', :expected => e}
		tests << {:class_name => 'Timestamp', :raise => true, :value => '', :expected => e}
		tests << {:class_name => 'Custom', :raise => true, :value => '', :expected => e}

		tests.each do |test|
			attr.class_name = test[:class_name]

			if test[:raise] == true then
				assert_raises(test[:expected]) do
					attr.convert_attribute_value test[:value]
				end
			else
				assert_equal(test[:expected], attr.convert_attribute_value(test[:value]))
			end
		end

		attr.class_name = origin
	end

	def test_to_h
		list = {'name'=>'title', 'class_name'=>'String', 'column_name'=>'title',
			'allow_null'=>true, 'external_type'=>'char', 'width'=>nil, 'read_only'=>false}
		attr = Attribute.new list
		expected = {'name'=>'title', 'class_name'=>'String', 'column_name'=>'title',
			'allow_null'=>true, 'external_type'=>'char', 'read_only'=>false}

		assert_equal(expected, attr.to_h)
	end


	# converting

	def test_beautify_name
		attribute = Attribute.new

		attribute.column_name = 'ATTRIBUTE_NAME'
		attribute.beautify_name
		assert_equal('attribute_name', attribute.name)
	end

end
