class TestEntity < Test::Unit::TestCase
	include TapKit

	def setup
		@app    = Application.new MODELS
		@ec     = @app.create_editing_context
		@entity = @app.model_group.entity 'Book'

		@row    = {'book_id' => 1, 'title' => 'test', 'publisher_id' => 1}
	end

	# add and remove
	def test_set_attribute
		attr = Attribute.new({'name'=>'test'})

		@entity.add_attribute attr
		assert_equal('test', @entity.attribute('test').name)

		@entity.remove_attribute attr
		assert_nil @entity.attribute('test')
	end

	def test_attribute_named
		assert_not_nil @entity.attribute('title')
	end

	# add and remove
	def test_set_relationship
		re = Relationship.new({'name'=>'test'})

		@entity.add_relationship re
		assert_equal('test', @entity.relationship('test').name)

		@entity.remove_relationship re
		assert_nil @entity.relationship('test')
	end

	def test_relationship_named
		assert_not_nil @entity.relationship('publisher')
	end

	def test_gid
		gid = @entity.gid @row
		assert_instance_of(KeyGlobalID, gid)
	end

	def test_class_property?
		name = 'title'
		attr = @entity.attribute name
		re   = @entity.relationship 'publisher'

		assert @entity.class_property?(attr)
		assert @entity.class_property?(re)
		assert @entity.class_property?(name)
	end

	def test_primary_key_attribute?
		name = 'book_id'
		attr = @entity.attribute name

		assert @entity.primary_key_attribute?(attr)
		assert @entity.primary_key_attribute?(name)
	end

	def test_primary_key_attribute_names
		names = ['book_id']
		assert_equal(names, @entity.primary_key_attribute_names)
	end

	def test_primary_key
		# for gid
		gid    = @entity.gid @row
		values = @entity.primary_key gid
		value  = {'book_id' => @row['book_id']}
		assert_equal(value, values)

		# for row
		values = @entity.primary_key @row
		assert_equal(value, values)

		# for instance of other classes
		values = @entity.primary_key nil
		assert_nil values
	end

	def test_qualifier_for_primary_key
		row = {'book_id' => 1}
		q = @entity.qualifier_for_primary_key row
		assert_instance_of(AndQualifier, q)
	end

	def test_qualifier_for_primary_key?
		row = {'book_id' => 1}
		q = @entity.qualifier_for_primary_key row
		assert @entity.qualifier_for_primary_key?(q)
	end


	# destination

	def test_destination_entity
		tests = [
			{:keypath => 'title', :expected => 'Book'},
			{:keypath => 'publisher', :expected => 'Publisher'},
			{:keypath => 'publisher.name', :expected => 'Publisher'}
		]

		tests.each do |test|
			entity = @entity.destination_entity test[:keypath]

			if test[:expected] then
				assert_equal(test[:expected], entity.name)
				assert_instance_of(Entity, entity)
			else
				assert_nil entity
			end
		end
	end

	def test_destination_attribute
		tests = [
			{:keypath => 'title', :expected => 'title'},
			{:keypath => 'publisher', :expected => nil},
			{:keypath => 'publisher.name', :expected => 'name'}
		]

		tests.each do |test|
			attribute = @entity.destination_attribute test[:keypath]

			if test[:expected] then
				assert_equal(test[:expected], attribute.name)
				assert_instance_of(Attribute, attribute)
			else
				assert_nil attribute
			end
		end
	end

	def test_destination_relationship
		tests = [
			{:keypath => 'title', :expected => nil},
			{:keypath => 'publisher', :expected => 'publisher'},
			{:keypath => 'publisher.name', :expected => 'publisher'}
		]

		tests.each do |test|
			relationship = @entity.destination_relationship test[:keypath]

			if test[:expected] then
				assert_equal(test[:expected], relationship.name)
				assert_instance_of(Relationship, relationship)
			else
				assert_nil relationship
			end
		end
	end


	# validation

	def test_validate_required_attributes
		entity = Entity.new
		assert_raises(ValidationError) { entity.validate_required_attributes }
	end


	def test_to_h
		list = {'name'=>'Test', 'external_name'=>'test', 'attributes'=>[],
		   'class_properties'=>[], 'primary_key_attributes'=>[], 'relationships'=>[]}
		entity = Entity.new list
		expected = {'name'=>'Test', 'external_name'=>'test', 'attributes'=>[],
		   'class_properties'=>[], 'primary_key_attributes'=>[],
		   'class_name'=>'GenericRecord'}

		assert_equal(expected, entity.to_h)
	end


	# converting

	def test_beautify_name
		entity = Entity.new

		entity.external_name = 'ENTITY_NAME'
		entity.beautify_name
		assert_equal('EntityName', entity.name)
	end

end
