unit Base_Set;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms,
  Dialogs, StdCtrls, ExtCtrls, DBCtrls, DB, IBCustomDataSet,
  IBTable, StrUtils, IBQuery, Provider, DBClient, DBLocal, DBLocalI,
  MainDatastore, IB, IBErrorCodes, IBSQL, IBDatabase;

type
  TSetTemplate = class(TForm)
    Transaction: TIBTransaction;
    Query: TIBSQL;

    bvlBorder: TBevel;

    btnOK: TButton;
    btnCancel: TButton;

    procedure Form_Create(Sender: TObject);
    procedure Form_Destroy(Sender: TObject);

    procedure Form_Show(Sender: TObject);
    procedure Form_CloseQuery(Sender: TObject; var CanClose: Boolean);

    procedure btnOK_Click(Sender: TObject);
  private
    { Private 錾 }
    FTableName : String;
    FModifiable : Boolean;
    FCancelable : Boolean;
    FPrimaryKey : Integer;
    FRealizedKey : Integer;
    FPreparing : Boolean;
    FSelecting : Boolean;
    FAppending : Boolean;
    FErrorMessages : TStrings;
    FSilentAction : Boolean;

    function GetRealizedKey() : Integer;
    function GetBrowsing() : Boolean;

    function VerifyError() : Boolean;

    function ShowConsole() : Integer;

    procedure HandleException( err : Exception);
  protected
    { Protected 錾 }
    property TableName : String read FTableName;
    property Modifiable : Boolean read FModifiable;
    property Cancelable : Boolean read FCancelable;
    property PrimaryKey : Integer read FPrimaryKey;
    property RealizedKey : Integer read GetRealizedKey;
    property Preparing : Boolean read FPreparing;
    property Selecting : Boolean read FSelecting;
    property Appending : Boolean read FAppending;
    property Browsing : Boolean read GetBrowsing;
    property SilentAction : Boolean read FSilentAction;
    property ErrorMessages : TStrings read FErrorMessages;

    procedure ShowError( const msg, title : String); overload;
    procedure ShowError( const msg : String); overload;
    procedure ShowNotify( const msg, title : String); overload;
    procedure ShowNotify( const msg : String); overload;
    function ShowConfirmation( const msg, title : String; flags : Integer) : Integer; overload;
    function ShowConfirmation( const msg : String; flags : Integer) : Integer; overload;

    function VerifyRecords( nameOfTable : String; whereSQL : String) : Boolean;
    function VerifyRecord( nameOfTable : String; keyInstance : Integer) : Boolean;
    function CancelRecords( nameOfTable : String; whereSQL : String) : Boolean;
    function CancelRecord( nameOfTable : String; keyInstance : Integer) : Boolean;
    function RestoreRecords( nameOfTable : String; whereSQL : String) : Boolean;
    function RestoreRecord( nameOfTable : String; keyInstance : Integer) : Boolean;

    procedure DefineTable( var nameOfTable : String; var modifyFlag, cancelFlag : Boolean); virtual; abstract;
    procedure BeforeInput(); virtual; abstract;
    procedure AfterInput(); virtual; abstract;
    procedure BeforeSelect(); virtual; abstract;
    procedure AfterSelect(); virtual; abstract;
    procedure BeforeAppend(); virtual; abstract;
    procedure AfterAppend(); virtual; abstract;
    procedure BeforePopup(); virtual; abstract;
    procedure AfterPopup(); virtual; abstract;

  public
    { Public 錾 }
    constructor Prepare; overload; virtual;

    procedure Browse( keyInstance : Integer; Silently : Boolean = false); overload; virtual;
    procedure Select( keyInstance : Integer; Silently : Boolean = false); overload; virtual;
    procedure Append( Silently : Boolean = false); overload; virtual;
    procedure Delete( keyInstance : Integer; Silently : Boolean = false); overload; virtual;
    procedure Restore( keyInstance : Integer; Silently : Boolean = false); overload; virtual;
  end;

  ETransactionRejected = class(Exception);
  EOperationForbidden = class(Exception);
  EOperationRejected = class(Exception);
  EOperationAborted = class(Exception);

var
  SetTemplate: TSetTemplate;

implementation

{$R *.dfm}

uses Misc_Constants, Misc_Utilities;

{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

constructor TSetTemplate.Prepare;
begin
	inherited Create( Screen.ActiveForm);

	if Owner is TSetTemplate then
	begin
		if ( Width >= ( Owner as TForm).Width - 5) and ( Height >= ( Owner as TForm).Height - 5) then
		begin
			Position := poDesigned;
			Left := ( Owner as TForm).Left + 160;
			Top := ( Owner as TForm).Top + 64;
		end;
	end;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TSetTemplate.Browse( keyInstance : Integer; Silently : Boolean);
begin
	try
		if keyInstance <= 0 then exit;
		if not Silently then try

			Transaction.StartTransaction;
			if not Transaction.InTransaction then raise ETransactionRejected.Create( '');

			FPrimaryKey := keyInstance;
			FRealizedKey := keyInstance;
			FPreparing := true;
			FSelecting := false;
			FAppending := false;

			BeforeInput;
			if not VerifyError then raise EOperationAborted.Create( '');

			BeforeSelect;
			if not VerifyError then raise EOperationAborted.Create( '');

			ShowConsole();
		except
			on err : Exception do HandleException( err);
		end;
	finally
		Release;
	end;
end;

procedure TSetTemplate.Select( keyInstance : Integer; Silently : Boolean);
begin
	try
		if keyInstance <= 0 then exit;

		FSilentAction := Silently;

		Transaction.StartTransaction;
		if not Transaction.InTransaction then raise ETransactionRejected.Create( '');

		FPrimaryKey := keyInstance;
		FRealizedKey := keyInstance;
		FPreparing := true;
		FSelecting := Modifiable and VerifyRecord( TableName, PrimaryKey);
		FAppending := false;

		BeforeInput;
		if not VerifyError then raise EOperationAborted.Create( '');

		BeforeSelect;
		if not VerifyError then raise EOperationAborted.Create( '');

		if ShowConsole() <> mrOk then raise EOperationAborted.Create( '');

		AfterSelect;
		if not VerifyError then raise EOperationAborted.Create( '');

		AfterInput;
		if not VerifyError then raise EOperationAborted.Create( '');

		Transaction.Commit;
	except
		on err : Exception do HandleException( err);
	end;
end;

procedure TSetTemplate.Append( Silently : Boolean);
begin
	try
		try
			FSilentAction := Silently;

			Transaction.StartTransaction;
			if not Transaction.InTransaction then raise ETransactionRejected.Create( '');

			FPreparing := true;
			FSelecting := false;
			FAppending := true;
			FPrimaryKey := -1;

			BeforeInput;
			if not VerifyError then raise EOperationAborted.Create( '');

			BeforeAppend;
			if not VerifyError then raise EOperationAborted.Create( '');

			if ShowConsole() <> mrOk then raise EOperationAborted.Create( '');

			AfterAppend;
			if not VerifyError then raise EOperationAborted.Create( '');

			AfterInput;
			if not VerifyError then raise EOperationAborted.Create( '');

			Transaction.Commit;
		except
			on err : Exception do HandleException( err);
		end;
	finally
		Release;
	end;
end;

procedure TSetTemplate.Delete( keyInstance : Integer; Silently : Boolean);
begin
	try
		if keyInstance <= 0 then exit;
		try
			FSilentAction := Silently;

			if not Cancelable then raise EOperationForbidden.Create( '');

			Transaction.StartTransaction;
			if not Transaction.InTransaction then raise ETransactionRejected.Create( '');

			FPrimaryKey := keyInstance;
			FRealizedKey := keyInstance;
			if not VerifyRecord( TableName, PrimaryKey) then raise EOperationRejected.Create( '');
			if not CancelRecord( TableName, PrimaryKey) then raise EOperationRejected.Create( '');

			Transaction.Commit;
		except
			on err : Exception do HandleException( err);
		end;
	finally
		Release;
	end;
end;

procedure TSetTemplate.Restore( keyInstance : Integer; Silently : Boolean);
begin
	try
		if keyInstance <= 0 then exit;
		try
			FSilentAction := Silently;

			Transaction.StartTransaction;
			if not Transaction.InTransaction then raise ETransactionRejected.Create( '');

			FPrimaryKey := keyInstance;
			FRealizedKey := keyInstance;
			if not VerifyRecord( TableName, PrimaryKey) then raise EOperationRejected.Create( '');
			if not RestoreRecord( TableName, PrimaryKey) then raise EOperationRejected.Create( '');

			Transaction.Commit;
		except
			on err : Exception do HandleException( err);
		end;
	finally
		Release;
	end;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function TSetTemplate.GetRealizedKey() : Integer;
begin
	if PrimaryKey > 0 then
		result := PrimaryKey
	else
	begin
		if FRealizedKey > 0 then
			result := FRealizedKey
		else
			result := DataStore.ReserveRevision( 1);
	end;

	FRealizedKey := result;
end;

function TSetTemplate.GetBrowsing() : Boolean;
begin
	result :=  not ( Appending or Selecting);
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function TSetTemplate.VerifyError() : Boolean;
begin
	if FErrorMessages.Count > 0 then
	begin
		if SIlentAction then raise Exception.Create( 'verification error');
		ShowMessage( FErrorMessages.Text);
		FErrorMessages.Clear;
		result := false;
	end
	else
		result := true;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function TSetTemplate.ShowConsole() : Integer;
begin
	if SilentAction then
	begin
		try
			Form_Show( self);
			if btnOK.Enabled then
			begin
				btnOK.Click;
				result := mrOk;
			end
			else
			begin
				btnCancel.Click;
				result := mrCancel;
			end;
		except
			on err : Exception do result := mrCancel;
		end;
	end
	else
		result := ShowModal();
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TSetTemplate.HandleException( err : Exception);
begin
	if Transaction.InTransaction then Transaction.Rollback;

	if err is EIBInterBaseError then
	begin
		case ( err as EIBInterBaseError).IBErrorCode of
		isc_lock_conflict : ShowNotify( '֘Af[^̃Rs[^ŎgpłB');
		isc_foreign_key : ShowNotify( '̃Rs[^Ŏgp̃f[^邽ߏł܂ł');
		else raise err;
		end;
		exit;
	end;

	if err is ETransactionRejected then
	begin
		ShowNotify( 'gUNVJnł܂ł');
		exit;
	end;

	if err is EOperationForbidden then
	begin
		ShowNotify( '̃f[^͍폜֎~Ă܂');
		exit;
	end;

	if err is EOperationRejected then
	begin
		ShowNotify( 'gp̃f[^Ȃ̂ŏł܂ł');
		exit;
	end;

	if err is EOperationAborted then exit;

	ShowNotify( err.Message);
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TSetTemplate.ShowError( const msg, title : String);
begin
	if SilentAction then raise Exception.Create( title);
	Application.MessageBox( PChar( msg), PChar( title));
end;

procedure TSetTemplate.ShowError( const msg : String);
begin
	ShowError( msg, Application.Title);
end;

procedure TSetTemplate.ShowNotify( const msg, title : String);
begin
	if SilentAction then exit;
	Application.MessageBox( PChar( msg), PChar( title));
end;

procedure TSetTemplate.ShowNotify( const msg : String);
begin
	ShowNotify( msg, Application.Title);
end;

function TSetTemplate.ShowConfirmation( const msg, title : String; flags : Integer) : Integer;
begin
	if SilentAction then raise Exception.Create( title);
	result := Application.MessageBox( PChar( msg), PChar( title), flags);
end;

function TSetTemplate.ShowConfirmation( const msg : String; flags : Integer) : Integer;
begin
	result := ShowConfirmation( msg, Application.Title, flags);
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function TSetTemplate.VerifyRecords( nameOfTable : String; whereSQL : String) : Boolean;
begin
	result := MainDatastore.VerifyRecords( Query, nameOfTable, whereSQL);
end;

function TSetTemplate.VerifyRecord( nameOfTable : String; keyInstance : Integer) : Boolean;
begin
	result := MainDatastore.VerifyRecord( Query, nameOfTable, keyInstance);
end;

function TSetTemplate.CancelRecords( nameOfTable : String; whereSQL : String) : Boolean;
begin
	result := MainDatastore.CancelRecords( Query, nameOfTable, whereSQL);
end;

function TSetTemplate.CancelRecord( nameOfTable : String; keyInstance : Integer) : Boolean;
begin
	result := MainDatastore.CancelRecord( Query, nameOfTable, keyInstance);
end;

function TSetTemplate.RestoreRecords( nameOfTable : String; whereSQL : String) : Boolean;
begin
	result := MainDatastore.RestoreRecords( Query, nameOfTable, whereSQL);
end;

function TSetTemplate.RestoreRecord( nameOfTable : String; keyInstance : Integer) : Boolean;
begin
	result := MainDatastore.RestoreRecord( Query, nameOfTable, keyInstance);
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TSetTemplate.Form_Create(Sender: TObject);
begin
	FModifiable := false;
	FCancelable := false;
	FPrimaryKey := -1;
	FRealizedKey := -1;
	FPreparing := false;
	FSelecting := false;
	FAppending := false;
	FSilentAction := false;

	FErrorMessages := TStringList.Create;

	DefineTable( FTableName, FModifiable, FCancelable);
end;

procedure TSetTemplate.Form_Destroy(Sender: TObject);
begin
	FErrorMessages.Free;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TSetTemplate.Form_Show(Sender: TObject);
begin
	ClientWidth := btnCancel.Left + btnCancel.Width + 8;
	ClientHeight := btnCancel.Top + btnCancel.Height + 8;

	btnOK.Enabled := not Browsing;

	BeforePopup;

	if Browsing then Caption := Caption + ' - ֎~';

	FPreparing := false;
end;

procedure TSetTemplate.Form_CloseQuery(Sender: TObject; var CanClose: Boolean);
begin
	CanClose := VerifyError;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TSetTemplate.btnOK_Click(Sender: TObject);
begin
	AfterPopup;
end;

end.
