unit Chooser_Schedule;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms,
  Dialogs, StdCtrls, StrUtils, ExtCtrls, Provider, DB, DBClient, DBLocal,
  MainDatastore, DBLocalI, IB, IBErrorCodes, IBSQL, IBDatabase, IBCustomDataSet,
  IBQuery, DBCtrls, ComCtrls, DateUtils, Grids, DBGrids, Math, Buttons,
  Base_Chooser, Misc_Constants, Misc_Utilities;

type
  TScheduleInfo = record
    typProgress, refService, refExchange : Integer;
    intMinutesAt, intMinutesFor : Integer;
    refAddressFrom, refAddressTo : Integer;
    refServer : Integer;
    strNotice : String;
  end;

function QuoteSchedule( _tran : TIBTransaction; datAt : TDate; keyClient, typSchedule : Integer; var info : TScheduleInfo) : Boolean;
function QuoteSchedulePrevMonth( _tran : TIBTransaction; datAt : TDate; keyClient, typSchedule : Integer; var info : TScheduleInfo) : Boolean;

type
  TScheduleChooser = class(TChooserTemplate)
    SourceOf_Schedule: TIBQuery;
    ProviderOf_Schedule: TDataSetProvider;
    Schedule: TClientDataSet;
    DataSourceOf_Schedule: TDataSource;

    gridSchedule: TDBGrid;

    btnBrowseSchedule: TButton;
    btnNextMonth: TSpeedButton;
    btnPrevMonth: TSpeedButton;
    pnlMonth: TPanel;
    btnSelectMonth: TButton;
    cboOption: TComboBox;
    chkQuoteMinutes: TCheckBox;
    chkQuoteAddress: TCheckBox;
    chkQuoteService: TCheckBox;
    chkQuoteServer: TCheckBox;

    procedure Selector_Change(Sender : TObject);
    procedure btnPrevMonth_Click(Sender: TObject);
    procedure btnNextMonth_Click(Sender: TObject);
    procedure btnSelectMonth_Click(Sender: TObject);

    procedure gridSchedule_DrawColumnCell(Sender: TObject; const Rect: TRect; DataCol: Integer; Column: TColumn; State: TGridDrawState);

    procedure btnBrowseSchedule_Click(Sender : TObject);

  private
    { Private 錾 }
    FScheduleLimit : TDate;
    FMonthFirstday : TDate;
    FMonthLastday : TDate;
  private
    { Private 錾 }
    procedure Retrieve( var info : TScheduleInfo);
  protected
    { Protected 錾 }
    procedure BeforePopup(); override;
    procedure AfterPopup(); override;
  end;

var
  ScheduleChooser: TScheduleChooser;

implementation

{$R *.dfm}

uses Chooser_Date, Set_Care, Set_Menage, Set_Migration, Set_Transfer;

{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function QuoteSchedule( _tran : TIBTransaction; datAt : TDate; keyClient, typSchedule : Integer; var info : TScheduleInfo) : Boolean;
begin
	if keyClient <= 0 then exit;
	with TScheduleChooser.Prepare do
	begin
		SourceOf_Schedule.ParamByName( 'keyClient').AsInteger := keyClient;
		SourceOf_Schedule.ParamByName( 'typSchedule').AsInteger := typSchedule;

		FScheduleLimit := datAt;
		FMonthFirstday := StartOfTheMonth( datAt);
		FMonthLastday := EndOfTheMonth( datAt);

		if Select( _tran, result) then Retrieve( info);
		Release;
	end;
end;

function QuoteSchedulePrevMonth( _tran : TIBTransaction; datAt : TDate; keyClient, typSchedule : Integer; var info : TScheduleInfo) : Boolean;
begin
	if keyClient <= 0 then exit;
	with TScheduleChooser.Prepare do
	begin
		SourceOf_Schedule.ParamByName( 'keyClient').AsInteger := keyClient;
		SourceOf_Schedule.ParamByName( 'typSchedule').AsInteger := typSchedule;

		FScheduleLimit := datAt;
		FMonthFirstday := StartOfTheMonth( IncMonth( datAt, -1));
		FMonthLastday := EndOfTheMonth( IncMonth( datAt, -1));

		if Select( _tran, result) then Retrieve( info);
		Release;
	end;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TScheduleChooser.Retrieve( var info : TScheduleInfo);
begin
	with info, Schedule do
	begin
		if chkQuoteService.Checked then
		begin
			typProgress := FieldByName( 'typProgress').AsInteger;
			refService := FieldByName( 'refService').AsInteger;
			refExchange := FieldByName( 'refExchange').AsInteger;
			strNotice := Trim( FieldByName( 'strNotice').AsString);
		end
		else
		begin
			typProgress := UNCLEAR_PROGRESS;
			refService := -1;
			refExchange := -1;
			strNotice := '';
		end;

		if chkQuoteMinutes.Checked then
		begin
			intMinutesAt := FieldByName( 'intMinutesAt').AsInteger;
			intMinutesFor := FieldByName( 'intMinutesFor').AsInteger;
		end
		else
		begin
			intMinutesAt := -1;
			intMinutesFor := -1;
		end;

		if chkQuoteAddress.Checked then
		begin
			refAddressFrom := FieldByName( 'refAddressFrom').AsInteger;
			refAddressTo := FieldByName( 'refAddressTo').AsInteger;
		end
		else
		begin
			refAddressFrom := -1;
			refAddressTo := -1;
		end;

		if chkQuoteServer.Checked then
			refServer := FieldByName( 'refServer').AsInteger
		else
			refServer := -1;
	end;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TScheduleChooser.BeforePopup();
begin
	if SourceOf_Schedule.ParamByName( 'typSchedule').AsInteger = TRANSFER_SCHEDULE then
	begin
		chkQuoteMinutes.Checked := false;
		chkQuoteServer.Checked := false;
	end;

	Selector_Change( self);
end;

procedure TScheduleChooser.AfterPopup();
begin
	if Schedule.IsEmpty then
		ErrorMessages.Add( 'QƂłXPW[܂B');
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TScheduleChooser.Selector_Change(Sender : TObject);
begin
	Schedule.Close;
	SourceOf_Schedule.ParamByName( 'datFrom').AsDate := FMonthFirstday;
	SourceOf_Schedule.ParamByName( 'datTo').AsDate := Min( FMonthLastday, FScheduleLimit);
	if cboOption.ItemIndex = 0 then
		SourceOf_Schedule.ParamByName( 'typWeekday').AsInteger := -1
	else
		SourceOf_Schedule.ParamByName( 'typWeekday').AsInteger := DayOfWeek( FScheduleLimit) - 1;
	Schedule.Open;
	// update query

	pnlMonth.Caption := FormatDateTime( 'gg ee "N" yyyy" N "MM" "', FMonthFirstday);
	btnNextMonth.Enabled := FMonthLastday < FScheduleLimit;

	btnBrowseSchedule.Enabled := not Schedule.IsEmpty;
	// update control
end;

procedure TScheduleChooser.btnPrevMonth_Click(Sender: TObject);
begin
	FMonthFirstday := IncMonth( DateOf( FMonthFirstday), -1);
	FMonthLastday := IncDay( IncMonth( FMonthFirstday), -1);
	Selector_Change( btnPrevMonth);
end;

procedure TScheduleChooser.btnNextMonth_Click(Sender: TObject);
begin
	FMonthFirstday := IncMonth( DateOf( FMonthFirstday));
	FMonthLastday := IncDay( IncMonth( FMonthFirstday), -1);
	Selector_Change( btnNextMonth);
end;

procedure TScheduleChooser.btnSelectMonth_Click(Sender: TObject);
begin
	if ChooseMonthBefore( FScheduleLimit, FMonthFirstday, FMonthLastday) then
	begin
		Selector_Change( btnSelectMonth);
	end;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TScheduleChooser.gridSchedule_DrawColumnCell(Sender: TObject; const Rect: TRect; DataCol: Integer; Column: TColumn; State: TGridDrawState);
begin
	gridSchedule.Canvas.Font.Size := gridSchedule.Font.Size div 2;
	case DayOfTheWeek( Schedule.FieldByName( 'datSchedule').AsDateTime) of
	DayMonday..DayFriday :
		begin
			gridSchedule.Canvas.Font.Color := IfThen( gdSelected in State, clHighlightText, Column.Font.Color);
			gridSchedule.Canvas.Brush.Color := IfThen( gdSelected in State, clHighlight, Column.Color);
		end;
	DaySaturday :
		begin
			gridSchedule.Canvas.Font.Color := IfThen( gdSelected in State, clWhite, Column.Font.Color);
			gridSchedule.Canvas.Brush.Color := IfThen( gdSelected in State, clNavy, TColor( $DCC0C0));
		end;
	DaySunday :
		begin
			gridSchedule.Canvas.Font.Color := IfThen( gdSelected in State, clWhite, Column.Font.Color);
			gridSchedule.Canvas.Brush.Color := IfThen( gdSelected in State, clMaroon, TColor( $C0C0DC));
		end;
	end;

	if Column.Field.FieldName = 'DATSCHEDULE' then with Schedule do
	begin
		gridSchedule.Canvas.TextRect( Rect, Rect.Left + 2, Rect.Top + ( Rect.Bottom - Rect.Top) div 4,
			FormatDateTime( 'dd"" (ddd)', Column.Field.AsDateTime)
		);
		exit;
	end;

	if Column.Field.FieldName = 'INTMINUTESAT' then with Schedule do
	begin
		case FieldByName( 'typSchedule').AsInteger of
		CARE_SCHEDULE, MENAGE_SCHEDULE :
			gridSchedule.Canvas.TextRect( Rect,
				Rect.Left + 2,
				Rect.Top + ( Rect.Bottom - Rect.Top) div 4,
				RangeToText( Column.Field.AsInteger, FieldByName( 'intMinutesFor').AsInteger)
			);
		else
			gridSchedule.Canvas.TextRect( Rect,
				Rect.Left + 2,
				Rect.Top + ( Rect.Bottom - Rect.Top) div 4,
				MinutesToText( Column.Field.AsInteger)
			);
		end;
		exit;
	end;

	if Column.Field.FieldName = 'STRSERVICE' then with Schedule do
	begin
		gridSchedule.Canvas.TextRect( Rect,
			Rect.Left + 2,
			Rect.Top + ( Rect.Bottom - Rect.Top) div 4,
			Column.Field.AsString
		);
		exit;
	end;

	if Column.Field.FieldName = 'STRADDRESSFROM' then
	begin
		case Schedule.FieldByName( 'typSchedule').AsInteger of
		TRANSFER_SCHEDULE, MIGRATION_SCHEDULE :
			begin
				gridSchedule.Canvas.TextRect( Rect, Rect.Left + 2, Rect.Top + 2, Column.Field.AsString);
				gridSchedule.Canvas.TextOut(
					Rect.Left + 2,
					Rect.Top + ( Rect.Bottom - Rect.Top) div 2 + 1,
					'  ' + Schedule.FieldByName( 'strAddressTo').AsString
				);
			end;
		CARE_SCHEDULE, MENAGE_SCHEDULE :
			gridSchedule.Canvas.TextRect( Rect,
				Rect.Left + 2,
				Rect.Top + ( Rect.Bottom - Rect.Top) div 4,
				Column.Field.AsString
			);
		end;
		exit;
	end;

	if ( Column.PickList.Count > 0) and ( not Column.Field.IsNull) then
	begin
		gridSchedule.Canvas.TextRect( Rect,
			Rect.Left + 2,
			Rect.Top + ( Rect.Bottom - Rect.Top) div 4,
			Column.PickList[ Column.Field.AsInteger]
		);
		exit;
	end;

	gridSchedule.Canvas.TextRect( Rect,
		Rect.Left + 2,
		Rect.Top + ( Rect.Bottom - Rect.Top) div 4,
		Column.Field.AsString
	);
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TScheduleChooser.btnBrowseSchedule_Click(Sender : TObject);
begin
	if Schedule.IsEmpty then exit;

	with Schedule do case FieldByName( 'typSchedule').AsInteger of
	     CARE_SCHEDULE : TCareSet.Prepare.Browse( FieldByName( 'keyInstance').AsInteger);
	   MENAGE_SCHEDULE : TMenageSet.Prepare.Browse( FieldByName( 'keyInstance').AsInteger);
	MIGRATION_SCHEDULE : TMigrationSet.Prepare.Browse( FieldByName( 'keyInstance').AsInteger);
	 TRANSFER_SCHEDULE : TTransferSet.Prepare.Browse( FieldByName( 'keyInstance').AsInteger);
	end;
end;

end.
