unit FlexReport;

interface

uses
  Windows, Messages, SysUtils, Classes, Controls, ExtCtrls, Graphics, Printers,
  StdCtrls, DateUtils, Math;

type
  TFlexErrorCode = (
    FR_ERROR_TYPE_MISMATCH,
    FR_ERROR_INVALID_COPY_COUNT,
    FR_ERROR_INVALID_SCALE_VALUE,
    FR_ERROR_INVALID_SIZE_VALUE,
    FR_ERROR_INVALID_ITERATION,
    FR_ERROR_INVALID_ROW_COUNT,
    FR_ERROR_INVALID_COL_COUNT,
    FR_ERROR_UNKNOWN_ORIENTATION,
    FR_ERROR_UNKNOWN_PAPER_FORM,
    FR_ERROR_UNKNOWN_BAND_ALIGN,
    FR_ERROR_NOT_FREEFORM,
    FR_ERROR_NOT_DESIGNING,
    FR_ERROR_NOW_PRINTING,
    FR_ERROR_NOW_WAITING,
    FR_ERROR_NO_REPORT,
    FR_ERROR_NO_PAPER,
    FR_ERROR_UNKNOWN
  );

  TFlexPaperForm = ( pfA6, pfA5, pfA4, pfA3, pfB6, pfB5, pfB4, pfB3, pfPostcard, pfFreeform);

  TFlexBandAlign = ( baTop, baBottom);

  TFlexReportFrameKind = ( frTop, frLeft, frRight, frBottom);
  TFlexReportFrames = set of TFlexReportFrameKind;

  TFlexReportPen = class;
  TFlexReportBrush = class;
  TFlexReportFont = class;
  TFlexCustomReport = class;

  TFlexReportHorzAlignment = ( haLeft, haRight, haCenter);
  TFlexReportVertAlignment = ( vaTop, vaBottom, vaCenter);

  TFlexReportScanDirection = ( gdRowToCol, gdColToRow);

  TFlexReportArrangement = record
    Left : Integer;
    Top : Integer;
    Right : Integer;
    Bottom : Integer;
    Width : Integer;
    Height : Integer;
    Pen : TFlexReportPen;
    Brush : TFlexReportBrush;
    Font : TFlexReportFont;
    Frames : TFlexReportFrames;
  end;

  TFlexReportCell = record
    Left : Integer;
    Top : Integer;
    Right : Integer;
    Bottom : Integer;
    Width : Integer;
    Height : Integer;
    Pen : TFlexReportPen;
    Brush : TFlexReportBrush;
    Font : TFlexReportFont;
    Frames : TFlexReportFrames;
    HorzAlignment : TFlexReportHorzAlignment;
    VertAlignment : TFlexReportVertAlignment;
    HorzSpacing : Integer;
    VertSpacing : Integer;
    LineSpacing : Integer;
    Lines : TStrings;
  end;

  TFlexPrintEvent = procedure ( Sender: TObject; Report: TFlexCustomReport) of object;
  TFlexParseEvent = procedure ( Sender: TObject; var Text : String) of object;
  TFlexArrangeEvent = procedure ( Sender: TObject; ARow, ACol : Integer; var Arrangement : TFlexReportArrangement) of object;
  TFlexFilloutEvent = procedure ( Sender: TObject; ARow, ACol : Integer; var Cell : TFlexReportCell) of object;

  EFlexReport = class(Exception)
  private
    { Private 錾 }
    FErrorCode : TFlexErrorCode;
  protected
    { Protected 錾 }
  public
    { Public 錾 }
    constructor Create( AErrorCode : TFlexErrorCode); virtual;

    property ErrorCode : TFlexErrorCode read FErrorCode;
  end;

  TFlexReportPen = class(TPersistent)
  private
    { Private 錾 }
    FStyle : TPenStyle;
    FColor : TColor;
    FWidth : Integer;
    FOnChange : TNotifyEvent;

    procedure SetStyle( AStyle : TPenStyle);
    procedure SetColor( AColor : TColor);
    procedure SetWidth( AWidth : Integer);
  public
    { Public 錾 }
    constructor Create;

    procedure Assign( Source : TPersistent); override;
    procedure AssignTo( Dest : TPersistent); override;
  published
    { Published 錾 }
    property Style : TPenStyle read FStyle write SetStyle;
    property Color : TColor read FColor write SetColor;
    property Width : Integer read FWidth write SetWidth;
    property OnChange : TNotifyEvent read FOnChange write FOnChange;
  end;

  TFlexReportBrush = class(TPersistent)
  private
    { Private 錾 }
    FStyle : TBrushStyle;
    FColor : TColor;
    FOnChange : TNotifyEvent;

    procedure SetStyle( AStyle : TBrushStyle);
    procedure SetColor( AColor : TColor);
  public
    { Public 錾 }
    constructor Create;

    procedure Assign( Source : TPersistent); override;
    procedure AssignTo( Dest : TPersistent); override;
  published
    { Published 錾 }
    property Style : TBrushStyle read FStyle write SetStyle;
    property Color : TColor read FColor write SetColor;

    property OnChange : TNotifyEvent read FOnChange write FOnChange;
  end;

  TFlexReportFont = class(TPersistent)
  private
    { Private 錾 }
    FName : TFontName;
    FCharset : TFontCharset;
    FPitch : TFontPitch;
    FStyle : TFontStyles;
    FColor : TColor;
    FHeight : Integer;
    FOnChange : TNotifyEvent;

    procedure SetName( AName : TFontName);
    procedure SetCharset( ACharset : TFontCharset);
    procedure SetPitch( APitch : TFontPitch);
    procedure SetStyle( AStyle : TFontStyles);
    procedure SetColor( AColor : TColor);
    procedure SetHeight( AHeight : Integer);
  public
    { Public 錾 }
    constructor Create;

    procedure Assign( Source : TPersistent); override;
    procedure AssignTo( Dest : TPersistent); override;
  published
    { Published 錾 }
    property Name : TFontName read FName write SetName;
    property Charset : TFontCharset read FCharset write SetCharset;
    property Pitch : TFontPitch read FPitch write SetPitch;
    property Style : TFontStyles read FStyle write SetStyle;
    property Color : TColor read FColor write SetColor;
    property Height : Integer read FHeight write SetHeight;

    property OnChange : TNotifyEvent read FOnChange write FOnChange;
  end;

  TFlexReportPaper = class(TCustomPanel)
  private
    { Private 錾 }
    FPaperForm : TFlexPaperForm;
    FPaperWidth : Integer;
    FPaperHeight : Integer;
    FLeftMargin : Integer;
    FTopMargin : Integer;
    FRightMargin : Integer;
    FBottomMargin : Integer;
    FOrientation : TPrinterOrientation;
    FDesignScale : Integer;

    FCascadeTo : TFlexReportPaper;

    FBeforePrint : TFlexPrintEvent;
    FAfterPrint : TFlexPrintEvent;

    procedure Adjust;

    procedure SetPaperForm( APaperForm : TFlexPaperForm);
    procedure SetOrientation( AOrientation : TPrinterOrientation);
    procedure SetPaperWidth( AWidth : Integer);
    procedure SetPaperHeight( AHeight : Integer);
    procedure SetLeftMargin( AMargin : Integer);
    procedure SetTopMargin( AMargin : Integer);
    procedure SetRightMargin( AMargin : Integer);
    procedure SetBottomMargin( AMargin : Integer);
    procedure SetDesignScale( AScale : Integer);
  protected
    { Protected 錾 }
    procedure SetName( const NewName : TComponentName); override;

    procedure ConstrainedResize( var MinWidth, MinHeight, MaxWidth, MaxHeight: Integer); override;
    procedure AlignControls( AControl : TControl; var ARect : TRect); override;
    procedure Paint; override;
  public
    { Public 錾 }
    constructor Create( AOwner : TComponent); override;
  published
    { Published 錾 }
    property PaperForm : TFlexPaperForm read FPaperForm write SetPaperForm default pfA4;
    property Orientation : TPrinterOrientation read FOrientation write SetOrientation default poPortrait;
    property PaperWidth : Integer read FPaperWidth write SetPaperWidth default 210;
    property PaperHeight : Integer read FPaperHeight write SetPaperHeight default 297;
    property LeftMargin : Integer read FLeftMargin write SetLeftMargin default 15;
    property TopMargin : Integer read FTopMargin write SetTopMargin default 15;
    property RightMargin : Integer read FRightMargin write SetRightMargin default 15;
    property BottomMargin : Integer read FBottomMargin write SetBottomMargin default 15;
    property DesignScale : Integer read FDesignScale write SetDesignScale default 2;

    property CascadeTo : TFlexReportPaper read FCascadeTo write FCascadeTo;

    procedure Print( AReport : TFlexCustomReport);

    property BeforePrint : TFlexPrintEvent read FBeforePrint write FBeforePrint;
    property AfterPrint : TFlexPrintEvent read FAfterPrint write FAfterPrint;
  end;

  TFlexCustomReport = class(TComponent)
  private
    { Private 錾 }
    FPen : TFlexReportPen;
    FBrush : TFlexReportBrush;
    FFont : TFlexReportFont;

    FTitle : String;

    FNowPrinting : Boolean;
    FPageNumber : Integer;
    FCurrentPaper : TFlexReportPaper;
    FCurrentBounds : TRect;
    FAvailableWidth : Integer;
    FAvailableHeight : Integer;

    FBeforePrint : TFlexPrintEvent;
    FAfterPrint : TFlexPrintEvent;

    FBeforePage : TNotifyEvent;
    FAfterPage : TNotifyEvent;

  protected
    { Protected 錾 }
    procedure SetPen( APen : TFlexReportPen); virtual;
    procedure SetBrush( ABrush : TFlexReportBrush); virtual;
    procedure SetFont( AFont : TFlexReportFont); virtual;

    procedure SetTitle( ATitle : String);

    procedure SetCurrentBounds( ABounds : TRect); virtual;

    procedure StartDocument( APaper : TFlexReportPaper); virtual;
    procedure EndDocument; virtual;

    procedure StartPage( APaper : TFlexReportPaper); virtual;
    procedure EndPage; virtual;
  public
    { Public 錾 }
    constructor Create( AOwner : TComponent); override;

    property Pen : TFlexReportPen read FPen write SetPen;
    property Brush : TFlexReportBrush read FBrush write SetBrush;
    property Font : TFlexReportFont read FFont write SetFont;

    property Title : String read FTitle write SetTitle;

    property NowPrinting : Boolean read FNowPrinting;
    property PageNumber : Integer read FPageNumber;
    property CurrentPaper : TFlexReportPaper read FCurrentPaper;
    property CurrentBounds : TRect read FCurrentBounds;
    property AvailableWidth : Integer read FAvailableWidth;
    property AvailableHeight : Integer read FAvailableHeight;

    procedure Print( APaper : TFlexReportPaper); virtual;

    procedure BeginClip( X1, Y1, X2, Y2 : Integer); virtual; abstract;
    procedure EndClip(); virtual; abstract;

    function TextExtent( const Text : String) : TSize; virtual; abstract;
    function TextHeight( const Text : String) : Integer; virtual; abstract;
    function TextWidth( const Text : String) : Integer; virtual; abstract;

    procedure Line( X1, Y1, X2, Y2 : Integer); virtual; abstract;

    procedure Arc( X1, Y1, X2, Y2, X3, Y3, X4, Y4 : Integer); virtual; abstract;
    procedure Pie( X1, Y1, X2, Y2, X3, Y3, X4, Y4 : Integer); virtual; abstract;
    procedure Chord( X1, Y1, X2, Y2, X3, Y3, X4, Y4 : Integer); virtual; abstract;
    procedure Ellipse( X1, Y1, X2, Y2 : Integer); overload; virtual; abstract;
    procedure Ellipse( const Rect : TRect); overload; virtual; abstract;

    procedure Rectangle( X1, Y1, X2, Y2 : Integer); overload; virtual; abstract;
    procedure Rectangle( const Rect : TRect); overload; virtual; abstract;
    procedure RoundRect( X1, Y1, X2, Y2, CornerWidth, CornerHeight : Integer); overload; virtual; abstract;
    procedure RoundRect( const Rect : TRect; CornerWidth, CornerHeight : Integer); overload; virtual; abstract;
    procedure FillRect( X1, Y1, X2, Y2 : Integer); overload; virtual; abstract;
    procedure FillRect( const Rect : TRect); overload; virtual; abstract;

    procedure Polyline( Points : array of TPoint); virtual; abstract;
    procedure PolyBezier( const Points : array of TPoint); virtual; abstract;
    procedure Polygon( Points : array of TPoint); virtual; abstract;

    procedure TextOut( X, Y : Integer; const Text : String; _Transparent : Boolean); virtual; abstract;
    procedure TextRect( Rect : TRect; X, Y : Integer; const Text : String; _Transparent : Boolean); virtual; abstract;

    procedure Draw( X1, Y1, X2, Y2 : Integer; Graphic : TGraphic); overload; virtual; abstract;
    procedure Draw( const Rect : TRect; Graphic : TGraphic); overload; virtual; abstract;
    procedure Draw( X, Y : Integer; Graphic : TGraphic); overload; virtual; abstract;
  published
    { Published 錾 }
    property BeforePrint : TFlexPrintEvent read FBeforePrint write FBeforePrint;
    property AfterPrint : TFlexPrintEvent read FAfterPrint write FAfterPrint;

    property BeforePage : TNotifyEvent read FBeforePage write FBeforePage;
    property AfterPage : TNotifyEvent read FAfterPage write FAfterPage;
  end;

  TFlexReport = class(TFlexCustomReport)
  private
    { Private 錾 }
    FPageEMFs : array of TMetafile;
    FCanvas : TMetafileCanvas;
  protected
    { Protected 錾 }
    function GetPageCount() : Integer; virtual;
    function GetPageMetafiles( index : Integer) : TMetafile; virtual;

    procedure StartDocument( APaper : TFlexReportPaper); override;
    procedure EndDocument; override;

    procedure StartPage( APaper : TFlexReportPaper); override;
    procedure EndPage; override;

    procedure MatiereSetup;

    function DesignToPaperX( X : Integer) : Integer;
    function PaperToDesignX( X : Integer) : Integer;
    function DesignToPaperY( Y : Integer) : Integer;
    function PaperToDesignY( Y : Integer) : Integer;
  public
    { Public 錾 }
    constructor Create( AOwner : TComponent); override;
    destructor Destroy; override;

    property PageCount : Integer read GetPageCount;
    property PageMetafiles[ index : Integer] : TMetafile read GetPageMetafiles;

    procedure BeginClip( X1, Y1, X2, Y2 : Integer); override;
    procedure EndClip(); override;

    function TextExtent( const Text : String) : TSize; override;
    function TextHeight( const Text : String) : Integer; override;
    function TextWidth( const Text : String) : Integer; override;

    procedure Line( X1, Y1, X2, Y2 : Integer); override;

    procedure Arc( X1, Y1, X2, Y2, X3, Y3, X4, Y4 : Integer); override;
    procedure Pie( X1, Y1, X2, Y2, X3, Y3, X4, Y4 : Integer); override;
    procedure Chord( X1, Y1, X2, Y2, X3, Y3, X4, Y4 : Integer); override;
    procedure Ellipse( X1, Y1, X2, Y2 : Integer); overload; override;
    procedure Ellipse( const Rect : TRect); overload; override;

    procedure Rectangle( X1, Y1, X2, Y2 : Integer); overload; override;
    procedure Rectangle( const Rect : TRect); overload; override;
    procedure RoundRect( X1, Y1, X2, Y2, CornerWidth, CornerHeight : Integer); overload; override;
    procedure RoundRect( const Rect : TRect; CornerWidth, CornerHeight : Integer); overload; override;
    procedure FillRect( X1, Y1, X2, Y2 : Integer); overload; override;
    procedure FillRect( const Rect : TRect); overload; override;

    procedure Polyline( Points : array of TPoint); override;
    procedure PolyBezier( const Points : array of TPoint); override;
    procedure Polygon( Points : array of TPoint); override;

    procedure TextOut( X, Y : Integer; const Text : String; _Transparent : Boolean); override;
    procedure TextRect( Rect : TRect; X, Y : Integer; const Text : String; _Transparent : Boolean); override;

    procedure Draw( X1, Y1, X2, Y2 : Integer; Graphic : TGraphic); overload; override;
    procedure Draw( const Rect : TRect; Graphic : TGraphic); overload; override;
    procedure Draw( X, Y : Integer; Graphic : TGraphic); overload; override;
  end;

  TFlexReportCustomItem = class(TCustomPanel)
  private
    { Private 錾 }
    FPrinted : Boolean;
    FArrangementX : Integer;
    FArrangementY : Integer;

    function GetAbsoluteLeft : Integer;
    function GetAbsoluteTop : Integer;
    function GetAbsoluteRight : Integer;
    function GetAbsoluteBottom : Integer;
    function GetAbsoluteRect : TRect;
  protected
    { Protected 錾 }
    FBeforePrint : TFlexPrintEvent;
    FAfterPrint : TFlexPrintEvent;

    function ContainerPaper : TFlexReportPaper;

    procedure Material_Changed( Sender : TObject); virtual;

    procedure Print( X, Y : Integer; AReport : TFlexCustomReport); overload;
  public
    { Public 錾 }
    property AbsoluteLeft : Integer read GetAbsoluteLeft;
    property AbsoluteTop : Integer read GetAbsoluteTop;
    property AbsoluteRight : Integer read GetAbsoluteRight;
    property AbsoluteBottom : Integer read GetAbsoluteBottom;
    property AbsoluteRect : TRect read GetAbsoluteRect;
    property Printed : Boolean read FPrinted;

    procedure Print( AReport : TFlexCustomReport); overload; virtual; abstract;
  published
    { Published 錾 }
    property Enabled;

    property BeforePrint : TFlexPrintEvent read FBeforePrint write FBeforePrint;
    property AfterPrint : TFlexPrintEvent read FAfterPrint write FAfterPrint;
  end;

  TFlexReportLayout = class(TFlexReportCustomItem)
  private
    { Private 錾 }
    FAlign : TFlexBandAlign;
    FIteration : Boolean;
    FIterator : Integer;
  protected
    { Protected 錾 }
    procedure SetName( const NewName : TComponentName); override;
    procedure SetParent( AParent: TWinControl); override;
    procedure SetAlign( AAlign : TFlexBandAlign);

    procedure Paint; override;
  public
    { Public 錾 }
    constructor Create( AOwner : TComponent); override;
    destructor Destroy; override;

    property Iterator : Integer read FIterator;

    procedure Print( AReport : TFlexCustomReport); override;
  published
    { Published 錾 }
    property Align : TFlexBandAlign read FAlign write SetAlign;
    property Iteration : Boolean read FIteration write FIteration;
  end;

  TFlexReportLayouts = array of TFlexReportLayout;

  TFlexReportBand = class(TFlexReportLayout)
  private
    { Private 錾 }
    FFrames : TFlexReportFrames;
    FBrush : TFlexReportBrush;
    FPen : TFlexReportPen;
    FTransparent : Boolean;
  protected
    { Protected 錾 }
    procedure SetFrames( AFrames : TFlexReportFrames);
    procedure SetBrush( ABrush : TFlexReportBrush);
    procedure SetPen( APen : TFlexReportPen);

    procedure Paint; override;
  public
    { Public 錾 }
    constructor Create( AOwner : TComponent); override;
    destructor Destroy; override;

    procedure Print( AReport : TFlexCustomReport); override;
  published
    { Published 錾 }
    property Frames : TFlexReportFrames read FFrames write SetFrames;
    property Brush : TFlexReportBrush read FBrush write SetBrush;
    property Pen : TFlexReportPen read FPen write SetPen;
    property Transparent : Boolean read FTransparent write FTransparent default true;
  end;

  TFlexReportCustomPanel = class(TFlexReportCustomItem)
  private
    { Private 錾 }
  protected
    { Protected 錾 }
  public
    { Public 錾 }
  published
    { Published 錾 }
    property Align;
    property Anchors;
  end;

  TFlexReportArranger = class(TFlexReportCustomPanel)
  private
    { Private 錾 }
    FPen : TFlexReportPen;
    FBrush : TFlexReportBrush;
    FFont : TFlexReportFont;
    FFrames : TFlexReportFrames;
    FRowCount : Integer;
    FColCount : Integer;
    FDirection : TFlexReportScanDirection;
    FTransparent : Boolean;

    FBeforeArrange : TFlexArrangeEvent;
    FAfterArrange : TFlexArrangeEvent;
  protected
    { Protected 錾 }
    procedure SetParent( AParent : TWinControl); override;
    procedure SetPen( APen : TFlexReportPen);
    procedure SetBrush( ABrush : TFlexReportBrush);
    procedure SetFont( AFont : TFlexReportFont);
    procedure SetFrames( AFrames : TFlexReportFrames);
    procedure SetRowCount( ACount : Integer);
    procedure SetColCount( ACount : Integer);

    procedure AlignControls( AControl : TControl; var ARect : TRect); override;

    procedure Paint; override;
  public
    { Public 錾 }
    constructor Create( AOwner : TComponent); override;
    destructor Destroy; override;

    procedure Print( AReport : TFlexCustomReport); override;
  published
    { Published 錾 }
    property Pen : TFlexReportPen read FPen write SetPen;
    property Brush : TFlexReportBrush read FBrush write SetBrush;
    property Font : TFlexReportFont read FFont write SetFont;
    property Frames : TFlexReportFrames read FFrames write SetFrames;
    property RowCount : Integer read FRowCount write SetRowCount;
    property ColCount : Integer read FColCount write SetColCount;
    property Direction : TFlexReportScanDirection read FDirection write FDirection;
    property Transparent : Boolean read FTransparent write FTransparent default true;

    property BeforeArrange : TFlexArrangeEvent read FBeforeArrange write FBeforeArrange;
    property AfterArrange : TFlexArrangeEvent read FAfterArrange write FAfterArrange;
  end;

  TFlexReportPanel = class(TFlexReportCustomPanel)
  private
    { Private 錾 }
    FPen : TFlexReportPen;
    FBrush : TFlexReportBrush;
    FFont : TFlexReportFont;
    FFrames : TFlexReportFrames;
    FHorzAlignment : TFlexReportHorzAlignment;
    FVertAlignment : TFlexReportVertAlignment;
    FHorzSpacing : Integer;
    FVertSpacing : Integer;
    FLineSpacing : Integer;
    FLines : TStrings;
    FTransparent : Boolean;

    FParseString : TFlexParseEvent;
  protected
    { Protected 錾 }
    procedure SetParent( AParent : TWinControl); override;
    procedure SetPen( APen : TFlexReportPen);
    procedure SetBrush( ABrush : TFlexReportBrush);
    procedure SetFont( AFont : TFlexReportFont);
    procedure SetFrames( AFrames : TFlexReportFrames);
    procedure SetHorzAlignment( AnAlignment : TFlexReportHorzAlignment);
    procedure SetVertAlignment( AnAlignment : TFlexReportVertAlignment);
    procedure SetHorzSpacing( ASpacing : Integer);
    procedure SetVertSpacing( ASpacing : Integer);
    procedure SetLineSpacing( ASpacing : Integer);
    procedure SetLines( ALines : TStrings);

    function ParseString( const Source : String) : String; virtual;

    procedure Paint; override;
  public
    { Public 錾 }
    constructor Create( AOwner : TComponent); override;
    destructor Destroy; override;

    procedure Print( AReport : TFlexCustomReport); override;
  published
    { Published 錾 }
    property Pen : TFlexReportPen read FPen write SetPen;
    property Brush : TFlexReportBrush read FBrush write SetBrush;
    property Font : TFlexReportFont read FFont write SetFont;
    property Frames : TFlexReportFrames read FFrames write SetFrames;
    property HorzAlignment : TFlexReportHorzAlignment read FHorzAlignment write SetHorzAlignment;
    property VertAlignment : TFlexReportVertAlignment read FVertAlignment write SetVertAlignment;
    property HorzSpacing : Integer read FHorzSpacing write SetHorzSpacing;
    property VertSpacing : Integer read FVertSpacing write SetVertSpacing;
    property LineSpacing : Integer read FLineSpacing write SetLineSpacing;
    property Lines : TStrings read FLines write SetLines;
    property Transparent : Boolean read FTransparent write FTransparent default true;

    property OnParseString : TFlexParseEvent read FParseString write FParseString;
  end;

  TFlexReportInteger = class(TFlexReportPanel)
  private
    { Private 錾 }
    FValue : Integer;
  protected
    { Protected 錾 }
    procedure SetValue( AValue : Integer);

    function ParseString( const Source : String) : String; override;
  published
    { Published 錾 }
    property Value : Integer read FValue write SetValue;
  end;

  TFlexReportPageNumber = class(TFlexReportPanel)
  private
    { Private 錾 }
    FCurrentReport : TFlexCustomReport;
  protected
    { Protected 錾 }
    function ParseString( const Source : String) : String; override;
  published
    { Published 錾 }
    procedure Print( AReport : TFlexCustomReport); override;
  end;

  TFlexReportDecimal = class(TFlexReportPanel)
  private
    { Private 錾 }
    FValue : Double;
  protected
    { Protected 錾 }
    procedure SetValue( AValue : Double);

    function ParseString( const Source : String) : String; override;
  published
    { Published 錾 }
    property Value : Double read FValue write SetValue;
  end;

  TFlexReportDateTime = class(TFlexReportPanel)
  private
    { Private 錾 }
    FAutoToday : Boolean;
    FValue : TDateTime;
  protected
    { Protected 錾 }
    procedure SetAutoToday( AValue : Boolean);
    procedure SetValue( AValue : TDateTime);

    function ParseString( const Source : String) : String; override;
  published
    { Published 錾 }
    property AutoToday : Boolean read FAutoToday write SetAutoToday;
    property Value : TDateTime read FValue write SetValue;
  end;

  TFlexReportScoreboard = class(TFlexReportCustomItem)
  private
    { Private 錾 }
    FPen : TFlexReportPen;
    FDivider : TFlexReportPen;
    FBrush : TFlexReportBrush;
    FFont : TFlexReportFont;
    FFrames : TFlexReportFrames;
    FPartitions : Integer;
    FLetters : String;
    FTransparent : Boolean;
  protected
    { Protected 錾 }
    procedure SetParent( AParent : TWinControl); override;
    procedure SetPen( APen : TFlexReportPen);
    procedure SetDivider( ADivider : TFlexReportPen);
    procedure SetBrush( ABrush : TFlexReportBrush);
    procedure SetFont( AFont : TFlexReportFont);
    procedure SetFrames( AFrames : TFlexReportFrames);
    procedure SetPartitions( APartitions : Integer);
    procedure SetLetters( AValue : String);

    procedure Paint; override;
  public
    { Public 錾 }
    constructor Create( AOwner : TComponent); override;
    destructor Destroy; override;

    procedure Print( AReport : TFlexCustomReport); override;
  published
    { Published 錾 }
    property Align;
    property Anchors;
    property Pen : TFlexReportPen read FPen write SetPen;
    property Divider : TFlexReportPen read FDivider write SetDivider;
    property Brush : TFlexReportBrush read FBrush write SetBrush;
    property Font : TFlexReportFont read FFont write SetFont;
    property Frames : TFlexReportFrames read FFrames write SetFrames;
    property Partitions : Integer read FPartitions write SetPartitions;
    property Letters : String read FLetters write SetLetters;
    property Transparent : Boolean read FTransparent write FTransparent default true;
  end;

  TFlexReportGrid = class(TFlexReportCustomItem)
  private
    { Private 錾 }
    FPen : TFlexReportPen;
    FBrush : TFlexReportBrush;
    FFont : TFlexReportFont;
    FFrames : TFlexReportFrames;
    FRowCount : Integer;
    FColCount : Integer;
    FDirection : TFlexReportScanDirection;
    FTransparent : Boolean;

    FFilloutCell : TFlexFilloutEvent;
  protected
    { Protected 錾 }
    procedure SetParent( AParent : TWinControl); override;
    procedure SetPen( APen : TFlexReportPen);
    procedure SetBrush( ABrush : TFlexReportBrush);
    procedure SetFont( AFont : TFlexReportFont);
    procedure SetFrames( AFrames : TFlexReportFrames);
    procedure SetRowCount( ACount : Integer);
    procedure SetColCount( ACount : Integer);

    procedure Paint; override;
  public
    { Public 錾 }
    constructor Create( AOwner : TComponent); override;
    destructor Destroy; override;

    procedure Print( AReport : TFlexCustomReport); override;
  published
    { Published 錾 }
    property Align;
    property Anchors;
    property Pen : TFlexReportPen read FPen write SetPen;
    property Brush : TFlexReportBrush read FBrush write SetBrush;
    property Font : TFlexReportFont read FFont write SetFont;
    property Frames : TFlexReportFrames read FFrames write SetFrames;
    property RowCount : Integer read FRowCount write SetRowCount;
    property ColCount : Integer read FColCount write SetColCount;
    property Direction : TFlexReportScanDirection read FDirection write FDirection;
    property Transparent : Boolean read FTransparent write FTransparent default true;

    property OnFilloutCell : TFlexFilloutEvent read FFilloutCell write FFilloutCell;
  end;

procedure Register;

procedure PrintIf( cond : Boolean);

implementation

{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure Register;
begin
	RegisterComponents( 'FlexReport', [
		TFlexReport,
		TFlexReportPaper,
		TFlexReportLayout,
		TFlexReportBand,
		TFlexReportArranger,
		TFlexReportPanel,
		TFlexReportInteger,
		TFlexReportDecimal,
		TFlexReportDateTime,
		TFlexReportPageNumber,
		TFlexReportScoreboard,
		TFlexReportGrid
	]);
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure PrintIf( cond : Boolean);
begin
	if not cond then Abort;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function InvokeHandler( Handler : TFlexPrintEvent; Sender: TObject; Report : TFlexCustomReport) : Boolean;
begin
	result := true;
	try
		if @Handler <> nil then Handler( Sender, Report);
	except
		on E : EAbort do result := false;
	else
		raise;
	end;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure SortLayouts( var Layouts : array of TFlexReportLayout; leftMost, rightMost : Integer);

	function CompareLayout( test, criterion : TFlexReportLayout) : Integer;
	begin
		result := 0;

		if test.Align <> criterion.Align then
		begin
			result := Ord( test.Align) - Ord( criterion.Align);
			exit;
		end;

		case test.Align of
		baTop : result := test.Top - criterion.Top;
		baBottom : result := criterion.Top - test.Top;
		end;
	end;
var
	leftEdge, rightEdge : Integer;
	criterion, swapTemp : TFlexReportLayout;
begin
	if leftMost >= rightMost then exit;
	criterion := Layouts[ ( leftMost + rightMost) div 2];
	leftEdge := leftMost;
	rightEdge := rightMost;

	while true do
	begin
		while ( leftEdge < rightEdge)
			and ( CompareLayout( Layouts[ leftEdge], criterion) < 0)
				do Inc( leftEdge);
		while ( rightEdge > leftEdge)
			and ( CompareLayout( Layouts[ rightEdge], criterion) > 0)
				do Dec( rightEdge);

		if leftEdge = rightEdge then break;

		swapTemp := Layouts[ leftEdge];
		Layouts[ leftEdge] := Layouts[ rightEdge];
		Layouts[ rightEdge] := swapTemp;
	end;

	SortLayouts( Layouts, leftMost, leftEdge);
	SortLayouts( Layouts, leftEdge + 1, rightMost);
end;

function LayoutsOf( Container : TWinControl) : TFlexReportLayouts;
var
	index : Integer;
begin
	for index := 0 to Container.ControlCount - 1 do
	begin
		if Container.Controls[ index] is TFlexReportLayout then
		begin
			SetLength( result, Length( result) + 1);
			result[ High( result)] := Container.Controls[ index] as TFlexReportLayout;
		end;
	end;

	SortLayouts( result, Low( result), High( result));
end;


{*******************************************************************************
  @subject: EFlexReport
  @update: 2004/05/08 (Sat) 00:00:00
  = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =
 *******************************************************************************}

{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

constructor EFlexReport.Create( AErrorCode : TFlexErrorCode);
begin
	case AErrorCode of
	FR_ERROR_TYPE_MISMATCH : inherited Create( 'ɂ͔zuł܂B');
	FR_ERROR_INVALID_COPY_COUNT : inherited Create( 'sK؂ȕʕw肳܂B');
	FR_ERROR_INVALID_SCALE_VALUE : inherited Create( 'sK؂ȃXP[w肳܂B');
	FR_ERROR_INVALID_SIZE_VALUE : inherited Create( 'sK؂ȗpTCYw肳܂B');
	FR_ERROR_INVALID_ITERATION : inherited Create( 'sK؂ȌJԂ񐔂w肳܂B');
	FR_ERROR_INVALID_ROW_COUNT : inherited Create( 'sK؂ȍsw肳܂B');
	FR_ERROR_INVALID_COL_COUNT : inherited Create( 'sK؂ȗ񐔂w肳܂B');
	FR_ERROR_UNKNOWN_ORIENTATION : inherited Create( '`̗pw肳܂B');
	FR_ERROR_UNKNOWN_PAPER_FORM : inherited Create( '`̗pTCYw肳܂B');
	FR_ERROR_UNKNOWN_BAND_ALIGN : inherited Create( '`̃ohzuw肳܂B');
	FR_ERROR_NOT_FREEFORM : inherited Create( 'RTCYł͂Ȃ̂ŗpTCY͕ς܂B');
	FR_ERROR_NOT_DESIGNING : inherited Create( 'pTCY͐݌vɂς܂B');
	FR_ERROR_NOW_PRINTING : inherited Create( 'Ȃ̂ŏł܂B');
	FR_ERROR_NOW_WAITING : inherited Create( 'ɂł܂B');
	FR_ERROR_NO_REPORT : inherited Create( '|[g܂B');
	FR_ERROR_NO_PAPER : inherited Create( 'y[p[܂B');
	else
		inherited Create( '');
		AErrorCode := FR_ERROR_UNKNOWN;
	end;

	FErrorCode := AErrorCode;
end;


{*******************************************************************************
  @subject: TFlexReportPen
  @update: 2004/05/08 (Sat) 00:00:00
  = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =
 *******************************************************************************}

{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

constructor TFlexReportPen.Create;
begin
	inherited Create;

	FStyle := psSolid;
	FColor := clBlack;
	FWidth := 1;
	FOnChange := nil;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReportPen.SetStyle( AStyle : TPenStyle);
begin
	FStyle := AStyle;
	if @FOnChange <> nil then FOnChange( Self);
end;

procedure TFlexReportPen.SetColor( AColor : TColor);
begin
	FColor := AColor;
	if @FOnChange <> nil then FOnChange( Self);
end;

procedure TFlexReportPen.SetWidth( AWidth : Integer);
begin
	FWidth := AWidth;
	if @FOnChange <> nil then FOnChange( Self);
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReportPen.Assign( Source : TPersistent);
begin
	if Source is TFlexReportPen then
	begin
		FStyle := ( Source as TFlexReportPen).Style;
		FColor := ( Source as TFlexReportPen).Color;
		FWidth := ( Source as TFlexReportPen).Width;
		if @FOnChange <> nil then FOnChange( Self);
		exit;
	end;

	inherited Assign( Source);
end;

procedure TFlexReportPen.AssignTo( Dest : TPersistent);
begin
	if Dest is TFlexReportPen then
	begin
		( Dest as TFlexReportPen).Style := Style;
		( Dest as TFlexReportPen).Color := Color;
		( Dest as TFlexReportPen).Width := Width;
		exit;
	end;

	inherited AssignTo( Dest);
end;


{*******************************************************************************
  @subject: TFlexReportBrush
  @update: 2004/05/08 (Sat) 00:00:00
  = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =
 *******************************************************************************}

{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

constructor TFlexReportBrush.Create;
begin
	inherited Create;

	FStyle := bsSolid;
	FColor := clWhite;
	FOnChange := nil;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReportBrush.SetStyle( AStyle : TBrushStyle);
begin
	FStyle := AStyle;
	if @FOnChange <> nil then FOnChange( Self);
end;

procedure TFlexReportBrush.SetColor( AColor : TColor);
begin
	FColor := AColor;
	if @FOnChange <> nil then FOnChange( Self);
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReportBrush.Assign( Source : TPersistent);
begin
	if Source is TFlexReportBrush then
	begin
		FStyle := ( Source as TFlexReportBrush).Style;
		FColor := ( Source as TFlexReportBrush).Color;
		if @FOnChange <> nil then FOnChange( Self);
		exit;
	end;

	inherited Assign( Source);
end;

procedure TFlexReportBrush.AssignTo( Dest : TPersistent);
begin
	if Dest is TFlexReportBrush then
	begin
		( Dest as TFlexReportBrush).Style := Style;
		( Dest as TFlexReportBrush).Color := Color;
		exit;
	end;

	inherited AssignTo( Dest);
end;


{*******************************************************************************
  @subject: TFlexReportFont
  @update: 2004/05/08 (Sat) 00:00:00
  = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =
 *******************************************************************************}

{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

constructor TFlexReportFont.Create;
begin
	inherited Create;

	FName := 'lr oSVbN';
	FCharset := DEFAULT_CHARSET;
	FPitch := fpDefault;
	FStyle := [];
	FColor := clBlack;
	FHeight := 105;
	FOnChange := nil;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReportFont.SetName( AName : TFontName);
begin
	FName := AName;
	if @FOnChange <> nil then FOnChange( Self);
end;

procedure TFlexReportFont.SetCharset( ACharset : TFontCharset);
begin
	FCharset := ACharset;
	if @FOnChange <> nil then FOnChange( Self);
end;

procedure TFlexReportFont.SetPitch( APitch : TFontPitch);
begin
	FPitch := APitch;
	if @FOnChange <> nil then FOnChange( Self);
end;

procedure TFlexReportFont.SetStyle( AStyle : TFontStyles);
begin
	FStyle := AStyle;
	if @FOnChange <> nil then FOnChange( Self);
end;

procedure TFlexReportFont.SetColor( AColor : TColor);
begin
	FColor := AColor;
	if @FOnChange <> nil then FOnChange( Self);
end;

procedure TFlexReportFont.SetHeight( AHeight : Integer);
begin
	FHeight := AHeight;
	if @FOnChange <> nil then FOnChange( Self);
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReportFont.Assign( Source : TPersistent);
begin
	if Source is TFlexReportFont then
	begin
		FName := ( Source as TFlexReportFont).Name;
		FCharset := ( Source as TFlexReportFont).Charset;
		FPitch := ( Source as TFlexReportFont).Pitch;
		FStyle := ( Source as TFlexReportFont).Style;
		FColor := ( Source as TFlexReportFont).Color;
		FHeight := ( Source as TFlexReportFont).Height;
		if @FOnChange <> nil then FOnChange( Self);
		exit;
	end;

	inherited Assign( Source);
end;

procedure TFlexReportFont.AssignTo( Dest : TPersistent);
begin
	if Dest is TFlexReportFont then
	begin
		( Dest as TFlexReportFont).Name := Name;
		( Dest as TFlexReportFont).Charset := Charset;
		( Dest as TFlexReportFont).Pitch := Pitch;
		( Dest as TFlexReportFont).Style := Style;
		( Dest as TFlexReportFont).Color := Color;
		( Dest as TFlexReportFont).Height := Height;
		exit;
	end;

	inherited AssignTo( Dest);
end;


{*******************************************************************************
  @subject: TFlexReportPaper
  @update: 2004/05/08 (Sat) 00:00:00
  = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =
 *******************************************************************************}

{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

constructor TFlexReportPaper.Create( AOwner : TComponent);
begin
	inherited Create( AOwner);

	FLeftMargin := 15;
	FTopMargin := 15;
	FRightMargin := 15;
	FBottomMargin := 15;
	FOrientation := poPortrait;
	FDesignScale := 2;
	PaperForm := pfA4;

	Visible := csDesigning in ComponentState;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReportPaper.Adjust;
var
	AvailableRect : TRect;
begin
	Width := PaperWidth * DesignScale;
	Height := PaperHeight * DesignScale;

	AlignControls( nil, AvailableRect);
	Invalidate;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

 procedure TFlexReportPaper.SetName( const NewName : TComponentName);
begin
	inherited SetName( NewName);
	Invalidate;
end;

procedure TFlexReportPaper.SetPaperForm( APaperForm : TFlexPaperForm);

	procedure SetPageExtent( AWidth, AHeight : Integer);
	begin
		if Orientation = poPortrait then
		begin
			FPaperWidth := AWidth;
			FPaperHeight := AHeight;
		end
		else
		begin
			FPaperWidth := AHeight;
			FPaperHeight := AWidth;
		end;

		FLeftMargin := Min( FLeftMargin, FPaperWidth);
		FTopMargin := Min( FTopMargin, FPaperHeight);
		FRightMargin := Min( FRightMargin, FPaperWidth - FLeftMargin);
		FBottomMargin := Min( FBottomMargin, FPaperHeight - FTopMargin);
	end;
begin
	case APaperForm of
	pfA6 : SetPageExtent( 105, 148);
	pfA5 : SetPageExtent( 148, 210);
	pfA4 : SetPageExtent( 210, 297);
	pfA3 : SetPageExtent( 297, 420);
	pfB6 : SetPageExtent( 128, 182);
	pfB5 : SetPageExtent( 182, 257);
	pfB4 : SetPageExtent( 257, 364);
	pfB3 : SetPageExtent( 364, 515);
	pfPostcard : SetPageExtent( 100, 148);
	pfFreeform : { do nothing }
	else
		raise EFlexReport.Create( FR_ERROR_UNKNOWN_PAPER_FORM);
	end;

	FPaperForm := APaperForm;

	Adjust;
end;

procedure TFlexReportPaper.SetOrientation( AOrientation : TPrinterOrientation);
var
	AWidth, AHeight : Integer;
begin
	if AOrientation = Orientation then exit;

	AWidth := PaperWidth;
	AHeight := PaperHeight;

	case AOrientation of
	poPortrait, poLandscape :
		begin
			FPaperWidth := AHeight;
			FPaperHeight := AWidth;

			FLeftMargin := Min( FLeftMargin, FPaperWidth);
			FTopMargin := Min( FTopMargin, FPaperHeight);
			FRightMargin := Min( FRightMargin, FPaperWidth - FLeftMargin);
			FBottomMargin := Min( FBottomMargin, FPaperHeight - FTopMargin);
		end;
	else
		raise EFlexReport.Create( FR_ERROR_UNKNOWN_ORIENTATION);
	end;

	FOrientation := AOrientation;

	Adjust;
end;

procedure TFlexReportPaper.SetPaperWidth( AWidth : Integer);
begin
	if PaperWidth = AWidth then exit;
	if PaperForm <> pfFreeform then raise EFlexReport.Create( FR_ERROR_NOT_FREEFORM);
	if AWidth <= 0 then raise EFlexReport.Create( FR_ERROR_INVALID_SIZE_VALUE);

	FPaperWidth := AWidth;

	Adjust;
end;

procedure TFlexReportPaper.SetPaperHeight( AHeight : Integer);
begin
	if PaperHeight = AHeight then exit;
	if PaperForm <> pfFreeform then raise EFlexReport.Create( FR_ERROR_NOT_FREEFORM);
	if AHeight <= 0 then raise EFlexReport.Create( FR_ERROR_INVALID_SIZE_VALUE);

	FPaperHeight := AHeight;

	Adjust;
end;

procedure TFlexReportPaper.SetLeftMargin( AMargin : Integer);
begin
	FLeftMargin := Min( AMargin, PaperWidth);
	FRightMargin := Min( FRightMargin, PaperWidth - FLeftMargin);

	Adjust;
end;

procedure TFlexReportPaper.SetTopMargin( AMargin : Integer);
begin
	FTopMargin := Min( AMargin, PaperHeight);
	FBottomMargin := Min( FBottomMargin, PaperHeight - FTopMargin);

	Adjust;
end;

procedure TFlexReportPaper.SetRightMargin( AMargin : Integer);
begin
	FRightMargin := Min( AMargin, PaperWidth - LeftMargin);

	Adjust;
end;

procedure TFlexReportPaper.SetBottomMargin( AMargin : Integer);
begin
	FBottomMargin := Min( AMargin, PaperHeight - TopMargin);

	Adjust;
end;

procedure TFlexReportPaper.SetDesignScale( AScale : Integer);
var
	OldScale : Integer;
begin
	if AScale <= 0 then raise EFlexReport.Create( FR_ERROR_INVALID_SCALE_VALUE);

	OldScale := FDesignScale;
	FDesignScale := AScale;

	ChangeScale( AScale, OldScale);
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReportPaper.ConstrainedResize( var MinWidth, MinHeight, MaxWidth, MaxHeight: Integer);
begin
	MinWidth := PaperWidth * DesignScale;
	MinHeight := PaperHeight * DesignScale;
	MaxWidth := PaperWidth * DesignScale;
	MaxHeight := PaperHeight * DesignScale;
end;

procedure TFlexReportPaper.AlignControls( AControl : TControl; var ARect : TRect);
begin
	ARect := Rect(
		LeftMargin * DesignScale,
		TopMargin * DesignScale,
		( PaperWidth - RightMargin) * DesignScale,
		( PaperHeight - BottomMargin) * DesignScale
	);
	inherited AlignControls( AControl, ARect);
end;

procedure TFlexReportPaper.Paint;
var
	X, Y : Integer;
begin
	Canvas.Brush.Color := clWhite;
	Canvas.Brush.Style := bsSolid;
	Canvas.FillRect( ClientRect);

	Canvas.Pen.Color := TColor( $DDDDDD);
	Canvas.Pen.Style := psDot;

	for X := 1 to PaperWidth div 5 do with Canvas do
	begin
		if X mod 2 = 0 then Pen.Style := psSolid else Pen.Style := psDot;
		MoveTo( X * 5 * DesignScale, 0);
		LineTo( X * 5 * DesignScale, Height * DesignScale);
	end;

	for Y := 1 to PaperHeight div 5 do with Canvas do
	begin
		if Y mod 2 = 0 then Pen.Style := psSolid else Pen.Style := psDot;
		MoveTo( 0, Y * 5 * DesignScale);
		LineTo( Width * DesignScale, Y * 5 * DesignScale);
	end;

	Canvas.Pen.Color := clNavy;
	Canvas.Brush.Style := bsClear;
	Canvas.Rectangle(
		LeftMargin * DesignScale,
		TopMargin * DesignScale,
		( PaperWidth - RightMargin) * DesignScale,
		( PaperHeight - BottomMargin) * DesignScale
	);

	Canvas.Font.Color := clGray;
	Canvas.Font.Height := 12;
	Canvas.TextOut( 2, 2, Name);
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReportPaper.Print( AReport : TFlexCustomReport);
var
	Layouts : TFlexReportLayouts;
	index : Integer;
begin
	Layouts := LayoutsOf( self);

	while true do
	begin
		AReport.SetCurrentBounds( Rect(
			LeftMargin * DesignScale,
			TopMargin * DesignScale,
			( PaperWidth - RightMargin) * DesignScale,
			TopMargin * DesignScale
		));

		if not InvokeHandler( BeforePrint, self, AReport) then break;

		try
			AReport.StartPage( self);
		except
			on E : EAbort do break;
		else
			raise;
		end;

		for index := Low( Layouts) to High( Layouts) do
		begin
			if Layouts[ index].Align = baTop then
				if Layouts[ index].Enabled then
					Layouts[ index].Print( AReport);
		end;

		AReport.SetCurrentBounds( Rect(
			LeftMargin * DesignScale,
			( PaperHeight - BottomMargin) * DesignScale,
			( PaperWidth - RightMargin) * DesignScale,
			( PaperHeight - BottomMargin) * DesignScale
		));

		for index := Low( Layouts) to High( Layouts) do
		begin
			if Layouts[ index].Align = baBottom then
				if Layouts[ index].Enabled then
					Layouts[ index].Print( AReport);
		end;

		try
			AReport.EndPage;
		except
			on E : EAbort do break;
		else
			raise;
		end;

		InvokeHandler( AfterPrint, self, AReport);

		if @BeforePrint = nil then break;
	end;

	if CascadeTo <> nil then CascadeTo.Print( AReport);
end;


{*******************************************************************************
  @subject: TFlexCustomReport
  @update: 2004/05/08 (Sat) 00:00:00
  = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =
 *******************************************************************************}

{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

constructor TFlexCustomReport.Create( AOwner : TComponent);
begin
	inherited Create( AOwner);

	FTitle := '()';
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexCustomReport.SetPen( APen : TFlexReportPen);
begin
	FPen.Assign( APen);
end;

procedure TFlexCustomReport.SetBrush( ABrush : TFlexReportBrush);
begin
	FBrush.Assign( ABrush);
end;

procedure TFlexCustomReport.SetFont( AFont : TFlexReportFont);
begin
	FFont.Assign( AFont);
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexCustomReport.SetTitle( ATitle : String);
begin
	if NowPrinting then raise EFlexReport.Create( FR_ERROR_NOW_PRINTING);

	FTitle := ATitle;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexCustomReport.SetCurrentBounds( ABounds : TRect);
begin
	if not NowPrinting then raise EFlexReport.Create( FR_ERROR_NOW_WAITING);

	FCurrentBounds.Left := Max( CurrentPaper.LeftMargin * CurrentPaper.DesignScale, Min( ABounds.Left, ( CurrentPaper.PaperWidth - CurrentPaper.RightMargin) * CurrentPaper.DesignScale));
	FCurrentBounds.Top := Max( CurrentPaper.TopMargin * CurrentPaper.DesignScale, Min( ABounds.Top, ( CurrentPaper.PaperHeight - CurrentPaper.BottomMargin) * CurrentPaper.DesignScale));
	FCurrentBounds.Right := Max( CurrentPaper.LeftMargin * CurrentPaper.DesignScale, Min( ABounds.Right, ( CurrentPaper.PaperWidth - CurrentPaper.RightMargin) * CurrentPaper.DesignScale));
	FCurrentBounds.Bottom := Max( CurrentPaper.TopMargin * CurrentPaper.DesignScale, Min( ABounds.Bottom, ( CurrentPaper.PaperHeight - CurrentPaper.BottomMargin) * CurrentPaper.DesignScale));
	FAvailableWidth := ( CurrentPaper.PaperWidth - CurrentPaper.LeftMargin - CurrentPaper.RightMargin) * CurrentPaper.DesignScale;
	FAvailableHeight := ( CurrentPaper.PaperHeight - CurrentPaper.BottomMargin) * CurrentPaper.DesignScale - FCurrentBounds.Bottom;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexCustomReport.StartDocument( APaper : TFlexReportPaper);
begin
	FCurrentPaper := APaper;

	FCurrentBounds.Left := 0;
	FCurrentBounds.Top := 0;
	FCurrentBounds.Right := 0;
	FCurrentBounds.Bottom := 0;
	FAvailableWidth := APaper.PaperWidth - APaper.LeftMargin - APaper.RightMargin;
	FAvailableHeight := APaper.PaperHeight - APaper.TopMargin - APaper.BottomMargin;

	FPageNumber := 1;
end;

procedure TFlexCustomReport.EndDocument;
begin
	FCurrentPaper := nil;

	FCurrentBounds.Left := 0;
	FCurrentBounds.Top := 0;
	FCurrentBounds.Right := 0;
	FCurrentBounds.Bottom := 0;
	FAvailableWidth := 0;
	FAvailableHeight := 0;

	FPageNumber := 0;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexCustomReport.StartPage( APaper : TFlexReportPaper);
begin
	FCurrentPaper := APaper;

	if @FBeforePage <> nil then FBeforePage( self);
end;

procedure TFlexCustomReport.EndPage;
begin
	if @FAfterPage <> nil then FAfterPage( self);

	Inc( FPageNumber);
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexCustomReport.Print( APaper : TFlexReportPaper);

	procedure ClearPrintedOfItem( Item : TFlexReportCustomItem);
	var
		index : Integer;
	begin
		for index := 0 to Item.ControlCount - 1 do
		begin
			if Item.Controls[ index] is TFlexReportCustomItem then
				ClearPrintedOfItem( Item.Controls[ index] as TFlexReportCustomItem);
		end;

		Item.FPrinted := false;
	end;

	procedure ClearPrintedOfPaper( Paper : TFlexReportPaper);
	var
		index : Integer;
	begin
		for index := 0 to Paper.ControlCount - 1 do
		begin
			if Paper.Controls[ index] is TFlexReportCustomItem then
				ClearPrintedOfItem( Paper.Controls[ index] as TFlexReportCustomItem);
		end;

		if Paper.CascadeTo <> nil then ClearPrintedOfPaper( Paper.CascadeTo);
	end;
begin
	if APaper = nil then raise EFlexReport.Create( FR_ERROR_NO_PAPER);

	ClearPrintedOfPaper( APaper);

	if not InvokeHandler( BeforePrint, self, self) then exit;

	StartDocument( APaper);

	FNowPrinting := true;
	APaper.Print( self);
	FNowPrinting := false;

	EndDocument;

	InvokeHandler( AfterPrint, self, self);
end;


{*******************************************************************************
  @subject: TFlexReport
  @update: 2004/05/08 (Sat) 00:00:00
  = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =
 *******************************************************************************}

{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

constructor TFlexReport.Create( AOwner : TComponent);
begin
	inherited Create( AOwner);

	FPen := TFlexReportPen.Create;
	FBrush := TFlexReportBrush.Create;
	FFont := TFlexReportFont.Create;

	SetLength( FPageEMFs, 0);
end;

destructor TFlexReport.Destroy;
var
	index : Integer;
begin
	FPen.Free;
	FBrush.Free;
	FFont.Free;

	for index := Low( FPageEMFs) to High( FPageEMFs) do
	begin
		FPageEMFs[ index].Free;
	end;
	SetLength( FPageEMFs, 0);

	inherited Destroy;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function TFlexReport.GetPageCount() : Integer;
begin
	result := Length( FPageEMFs);
end;

function TFlexReport.GetPageMetafiles( index : Integer) : TMetafile;
begin
	result := FPageEMFs[ index];
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReport.StartDocument( APaper : TFlexReportPaper);
var
	index : Integer;
begin
	inherited StartDocument( APaper);

	for index := Low( FPageEMFs) to High( FPageEMFs) do
	begin
		FPageEMFs[ index].Free;
	end;
	SetLength( FPageEMFs, 0);
end;

procedure TFlexReport.EndDocument;
begin
	inherited EndDocument;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReport.StartPage( APaper : TFlexReportPaper);
begin
	inherited StartPage( APaper);

	SetLength( FPageEMFs, Length( FPageEMFs) + 1);
	FPageEMFs[ High( FPageEMFs)] := TMetafile.Create;
	FPageEMFs[ High( FPageEMFs)].MMWidth := APaper.PaperWidth * 100;
	FPageEMFs[ High( FPageEMFs)].MMHeight := APaper.PaperHeight * 100;

	FCanvas := TMetafileCanvas.Create( FPageEMFs[ High( FPageEMFs)], 0);
	SetMapMode( FCanvas.Handle, MM_HIMETRIC);
end;

procedure TFlexReport.EndPage;
begin
	inherited EndPage;

	FCanvas.Free;
	FCanvas := nil;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReport.MatiereSetup;
var
	brushInfo : LOGBRUSH;
begin
	if FCanvas = nil then EFlexReport.Create( FR_ERROR_NOW_WAITING);

	brushInfo.lbStyle := BS_SOLID;
	brushInfo.lbColor := Pen.Color;
	brushInfo.lbHatch := 0;
	FCanvas.Pen.Handle := ExtCreatePen( Ord( Pen.Style) or PS_GEOMETRIC, Pen.Width * 12, brushInfo, 0, nil);

	FCanvas.Brush.Color := Brush.Color;
	FCanvas.Brush.Style := Brush.Style;

	FCanvas.Font.Color := Font.Color;
	FCanvas.Font.Charset := Font.Charset;
	FCanvas.Font.Name := Font.Name;
	FCanvas.Font.Pitch := Font.Pitch;
	FCanvas.Font.Style := Font.Style;
	FCanvas.Font.Height := - MulDiv( Font.Height, 254, 72);
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function TFlexReport.DesignToPaperX( X : Integer) : Integer;
begin
	if CurrentPaper = nil then EFlexReport.Create( FR_ERROR_NOW_WAITING);
	result := MulDiv( CurrentBounds.Left + X, 100, CurrentPaper.DesignScale);
end;

function TFlexReport.PaperToDesignX( X : Integer) : Integer;
begin
	if CurrentPaper = nil then EFlexReport.Create( FR_ERROR_NOW_WAITING);
	result := MulDiv( X - ( CurrentBounds.Left * 100 div CurrentPaper.DesignScale), CurrentPaper.DesignScale, 100);
end;

function TFlexReport.DesignToPaperY( Y : Integer) : Integer;
begin
	if CurrentPaper = nil then EFlexReport.Create( FR_ERROR_NOW_WAITING);
	result := - MulDiv( CurrentBounds.Top + Y, 100, CurrentPaper.DesignScale);
end;

function TFlexReport.PaperToDesignY( Y : Integer) : Integer;
begin
	if CurrentPaper = nil then EFlexReport.Create( FR_ERROR_NOW_WAITING);
	result := - MulDiv( Y + ( CurrentBounds.Top * 100 div CurrentPaper.DesignScale), CurrentPaper.DesignScale, 100);
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReport.BeginClip( X1, Y1, X2, Y2 : Integer);
begin
	SaveDC( FCanvas.Handle);
	IntersectClipRect( FCanvas.Handle,
		DesignToPaperX( X1), DesignToPaperY( Y1),
		DesignToPaperX( X2), DesignToPaperY( Y2)
	);
end;

procedure TFlexReport.EndClip();
begin
	RestoreDC( FCanvas.Handle, -1);
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function TFlexReport.TextExtent( const Text : String) : TSize;
begin
	MatiereSetup;
	result := FCanvas.TextExtent( Text);
	result.cx := PaperToDesignX( ( CurrentBounds.Left * 100 div CurrentPaper.DesignScale) + result.cx);
	result.cy := PaperToDesignY( - ( ( CurrentBounds.Top * 100 div CurrentPaper.DesignScale) + result.cy));
end;

function TFlexReport.TextHeight( const Text : String) : Integer;
begin
	MatiereSetup;
	result := PaperToDesignY( - ( ( CurrentBounds.Top * 100 div CurrentPaper.DesignScale) + FCanvas.TextHeight( Text)));
end;

function TFlexReport.TextWidth( const Text : String) : Integer;
begin
	MatiereSetup;
	result := PaperToDesignX( ( CurrentBounds.Left * 100 div CurrentPaper.DesignScale) + FCanvas.TextWidth( Text));
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReport.Line( X1, Y1, X2, Y2 : Integer);
begin
	MatiereSetup;
	FCanvas.MoveTo( DesignToPaperX( X1), DesignToPaperY( Y1));
	FCanvas.LineTo( DesignToPaperX( X2), DesignToPaperY( Y2));
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReport.Arc( X1, Y1, X2, Y2, X3, Y3, X4, Y4 : Integer);
begin
	MatiereSetup;
	FCanvas.Arc(
		DesignToPaperX( X1), DesignToPaperY( Y1),
		DesignToPaperX( X2), DesignToPaperY( Y2),
		DesignToPaperX( X3), DesignToPaperY( Y3),
		DesignToPaperX( X4), DesignToPaperY( Y4)
	);
end;

procedure TFlexReport.Pie( X1, Y1, X2, Y2, X3, Y3, X4, Y4 : Integer);
begin
	MatiereSetup;
	FCanvas.Pie(
		DesignToPaperX( X1), DesignToPaperY( Y1),
		DesignToPaperX( X2), DesignToPaperY( Y2),
		DesignToPaperX( X3), DesignToPaperY( Y3),
		DesignToPaperX( X4), DesignToPaperY( Y4)
	);
end;

procedure TFlexReport.Chord( X1, Y1, X2, Y2, X3, Y3, X4, Y4 : Integer);
begin
	MatiereSetup;
	FCanvas.Chord(
		DesignToPaperX( X1), DesignToPaperY( Y1),
		DesignToPaperX( X2), DesignToPaperY( Y2),
		DesignToPaperX( X3), DesignToPaperY( Y3),
		DesignToPaperX( X4), DesignToPaperY( Y4)
	);
end;

procedure TFlexReport.Ellipse( X1, Y1, X2, Y2 : Integer);
begin
	MatiereSetup;
	FCanvas.Ellipse(
		DesignToPaperX( X1), DesignToPaperY( Y1),
		DesignToPaperX( X2), DesignToPaperY( Y2)
	);
end;

procedure TFlexReport.Ellipse( const Rect : TRect);
begin
	Ellipse( Rect.Left, Rect.Top, Rect.Right, Rect.Bottom);
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReport.Rectangle( X1, Y1, X2, Y2 : Integer);
begin
	MatiereSetup;
	FCanvas.Rectangle(
		DesignToPaperX( X1), DesignToPaperY( Y1),
		DesignToPaperX( X2), DesignToPaperY( Y2)
	);
end;

procedure TFlexReport.Rectangle( const Rect : TRect);
begin
	Rectangle( Rect.Left, Rect.Top, Rect.Right, Rect.Bottom);
end;

procedure TFlexReport.RoundRect( X1, Y1, X2, Y2, CornerWidth, CornerHeight : Integer);
begin
	MatiereSetup;
	FCanvas.RoundRect(
		DesignToPaperX( X1), DesignToPaperY( Y1),
		DesignToPaperX( X2), DesignToPaperY( Y2),
		DesignToPaperX( CornerWidth), DesignToPaperY( CornerHeight)
	);
end;

procedure TFlexReport.RoundRect( const Rect : TRect; CornerWidth, CornerHeight : Integer);
begin
	RoundRect(
		Rect.Left, Rect.Top,
		Rect.Right, Rect.Bottom,
		CornerWidth, CornerHeight
	);
end;

procedure TFlexReport.FillRect( X1, Y1, X2, Y2 : Integer);
begin
	MatiereSetup;
	FCanvas.FillRect( Rect(
		DesignToPaperX( X1), DesignToPaperY( Y1),
		DesignToPaperX( X2), DesignToPaperY( Y2)
	));
end;

procedure TFlexReport.FillRect( const Rect : TRect);
begin
	FillRect( Rect.Left, Rect.Top, Rect.Right, Rect.Bottom);
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReport.Polyline( Points : array of TPoint);
var
	index : Integer;
	ScaledPoints : array of TPoint;
begin
	MatiereSetup;

	SetLength( ScaledPoints, Length( Points));
	for index := Low( Points) to High( Points) do
	begin
		ScaledPoints[ index].X := DesignToPaperX( Points[ index].X);
		ScaledPoints[ index].Y := DesignToPaperY( Points[ index].Y);
	end;

	FCanvas.Polyline( ScaledPoints);
end;

procedure TFlexReport.PolyBezier( const Points : array of TPoint);
var
	index : Integer;
	ScaledPoints : array of TPoint;
begin
	MatiereSetup;

	SetLength( ScaledPoints, Length( Points));
	for index := Low( Points) to High( Points) do
	begin
		ScaledPoints[ index].X := DesignToPaperX( Points[ index].X);
		ScaledPoints[ index].Y := DesignToPaperY( Points[ index].Y);
	end;

	FCanvas.PolyBezier( ScaledPoints);
end;

procedure TFlexReport.Polygon( Points : array of TPoint);
var
	index : Integer;
	ScaledPoints : array of TPoint;
begin
	MatiereSetup;

	SetLength( ScaledPoints, Length( Points));
	for index := Low( Points) to High( Points) do
	begin
		ScaledPoints[ index].X := DesignToPaperX( Points[ index].X);
		ScaledPoints[ index].Y := DesignToPaperY( Points[ index].Y);
	end;

	FCanvas.Polygon( ScaledPoints);
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReport.TextOut( X, Y : Integer; const Text : String; _Transparent : Boolean);
begin
	MatiereSetup;

	if _Transparent then
		SetBkMode( FCanvas.Handle, TRANSPARENT)
	else
		SetBkMode( FCanvas.Handle, OPAQUE);

	SetBkColor( FCanvas.Handle, Brush.Color);

	FCanvas.TextOut( DesignToPaperX( X), DesignToPaperY( Y), Text);
end;

procedure TFlexReport.TextRect( Rect : TRect; X, Y : Integer; const Text : String; _Transparent : Boolean);
begin
	MatiereSetup;

	if _Transparent then
		SetBkMode( FCanvas.Handle, TRANSPARENT)
	else
		SetBkMode( FCanvas.Handle, OPAQUE);

	SetBkColor( FCanvas.Handle, Brush.Color);

	Rect.Left := DesignToPaperX( Rect.Left);
	Rect.Top := DesignToPaperY( Rect.Top);
	Rect.Right := DesignToPaperX( Rect.Right);
	Rect.Bottom := DesignToPaperY( Rect.Bottom);

	FCanvas.TextRect( Rect, DesignToPaperX( X), DesignToPaperY( Y), Text);
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReport.Draw( X1, Y1, X2, Y2 : Integer; Graphic : TGraphic);
begin
	MatiereSetup;
	FCanvas.StretchDraw(
		Rect( DesignToPaperX( X1), DesignToPaperY( Y1), DesignToPaperX( X2), DesignToPaperY( Y2)),
		Graphic
	);
end;

procedure TFlexReport.Draw( const Rect : TRect; Graphic : TGraphic);
begin
	Draw( Rect.Left, Rect.Top, Rect.Right, Rect.Bottom, Graphic);
end;

procedure TFlexReport.Draw( X, Y : Integer; Graphic : TGraphic);
begin
	MatiereSetup;
	FCanvas.Draw( DesignToPaperX( X), DesignToPaperY( Y), Graphic);
end;


{*******************************************************************************
  @subject: TFlexReportCustomItem
  @update: 2004/05/08 (Sat) 00:00:00
  = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =
 *******************************************************************************}

{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function TFlexReportCustomItem.GetAbsoluteLeft : Integer;
begin
	if Parent is TFlexReportBand then
		result := Left + FArrangementX
	else
		result := ( Parent as TFlexReportCustomItem).AbsoluteLeft + Left + FArrangementX;
end;

function TFlexReportCustomItem.GetAbsoluteTop : Integer;
begin
	if Parent is TFlexReportBand then
		result := Top + FArrangementY
	else
		result := ( Parent as TFlexReportCustomItem).AbsoluteTop + Top + FArrangementY;
end;

function TFlexReportCustomItem.GetAbsoluteRight : Integer;
begin
	result := AbsoluteLeft + Width;
end;

function TFlexReportCustomItem.GetAbsoluteBottom : Integer;
begin
	result := AbsoluteTop + Height;
end;

function TFlexReportCustomItem.GetAbsoluteRect : TRect;
begin
	result := Rect( AbsoluteLeft, AbsoluteTop, AbsoluteRight, AbsoluteBottom);
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function TFlexReportCustomItem.ContainerPaper : TFlexReportPaper;
var
	scanner : TWinControl;
begin
	scanner := self;
	repeat
		scanner := scanner.Parent;
	until ( scanner is TFlexReportPaper) or ( scanner = nil);
	result := ( scanner as TFlexReportPaper);
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReportCustomItem.Material_Changed( Sender : TObject);
begin
	Invalidate;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReportCustomItem.Print( X, Y : Integer; AReport : TFlexCustomReport);
begin
	FArrangementX := X;
	FArrangementY := Y;

	Print( AReport);

	FArrangementX := 0;
	FArrangementY := 0;
end;


{*******************************************************************************
  @subject: TFlexReportLayout
  @update: 2004/05/08 (Sat) 00:00:00
  = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =
 *******************************************************************************}

{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

constructor TFlexReportLayout.Create( AOwner : TComponent);
begin
	inherited Create( AOwner);

	BevelInner := bvNone;
	BevelOuter := bvNone;

	Align := baTop;
	FIteration := false;
end;

destructor TFlexReportLayout.Destroy;
begin
	inherited Destroy;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReportLayout.SetName( const NewName : TComponentName);
begin
	inherited SetName( NewName);
	Invalidate;
end;

procedure TFlexReportLayout.SetParent( AParent: TWinControl);
begin
	if ( AParent <> nil)
	and ( not ( AParent is TFlexReportPaper))
	and ( not ( AParent is TFlexReportLayout))
	then
		raise EFlexReport.Create( FR_ERROR_TYPE_MISMATCH);

	inherited SetParent( AParent);
end;

procedure TFlexReportLayout.SetAlign( AAlign : TFlexBandAlign);
begin
	case AAlign of
	baTop : inherited Align := alTop;
	baBottom : inherited Align := alBottom;
	else raise EFlexReport.Create( FR_ERROR_UNKNOWN_BAND_ALIGN);
	end;

	FAlign := AAlign;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReportLayout.Paint;
begin
	Canvas.Brush.Color := clWhite;
	Canvas.Brush.Style := bsSolid;
	Canvas.FillRect( ClientRect);

	Canvas.Pen.Color := clSilver;
	Canvas.Pen.Style := psSolid;
	Canvas.Pen.Width := 1;

	if Align = baTop then
	begin
		Canvas.MoveTo( 0, Height - 1);
		Canvas.LineTo( Width, Height - 1);
	end
	else
	begin
		Canvas.MoveTo( 0, 0);
		Canvas.LineTo( Width, 0);
	end;

	Canvas.Font.Color := clGray;
	Canvas.Font.Height := 12;
	Canvas.TextOut( 2, 2, Name);
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReportLayout.Print( AReport : TFlexCustomReport);
var
	index : Integer;
	Layouts : TFlexReportLayouts;
begin
	Layouts := LayoutsOf( self);

	FIterator := 0;
	while ( FIterator = 0) or Iteration do
	begin
		if not InvokeHandler( BeforePrint, self, AReport) then break;

		for index := Low( Layouts) to High( Layouts) do
		begin
			if Layouts[ index].Enabled then Layouts[ index].Print( AReport);
		end;

		InvokeHandler( AfterPrint, self, AReport);

		Inc( FIterator);
		FPrinted := true;
	end;
end;


{*******************************************************************************
  @subject: TFlexReportBand
  @update: 2004/05/08 (Sat) 00:00:00
  = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =
 *******************************************************************************}

{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

constructor TFlexReportBand.Create( AOwner : TComponent);
begin
	inherited Create( AOwner);

	FBrush := TFlexReportBrush.Create;
	FBrush.OnChange := Material_Changed;

	FPen := TFlexReportPen.Create;
	FPen.OnChange := Material_Changed;

	FTransparent := true;
end;

destructor TFlexReportBand.Destroy;
begin
	FPen.Free;

	inherited Destroy;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReportBand.SetFrames( AFrames : TFlexReportFrames);
begin
	FFrames := AFrames;
	Invalidate;
end;

procedure TFlexReportBand.SetBrush( ABrush : TFlexReportBrush);
begin
	FBrush.Assign( ABrush);
	Invalidate;
end;

procedure TFlexReportBand.SetPen( APen : TFlexReportPen);
begin
	FPen.Assign( APen);
	Invalidate;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReportBand.Paint;
begin
	Canvas.Brush.Color := Brush.Color;
	Canvas.Brush.Style := Brush.Style;
	Canvas.FillRect( ClientRect);

	Canvas.Font.Color := clGray;
	Canvas.Font.Height := 12;
	Canvas.TextOut( 2, 2, Name);

	Canvas.Pen.Color := clSilver;
	Canvas.Pen.Style := psSolid;
	Canvas.Pen.Width := 1;

	if Align = baTop then
	begin
		Canvas.MoveTo( 0, Height - 1);
		Canvas.LineTo( Width, Height - 1);
	end
	else
	begin
		Canvas.MoveTo( 0, 0);
		Canvas.LineTo( Width, 0);
	end;

	Canvas.Pen.Color := Pen.Color;
	Canvas.Pen.Style := Pen.Style;
	Canvas.Pen.Width := Max( 1, Pen.Width * ContainerPaper.DesignScale div 8);

	if frLeft in Frames then
	begin
		Canvas.MoveTo( 0, 0);
		Canvas.LineTo( 0, Height);
	end;

	if frTop in Frames then
	begin
		Canvas.MoveTo( 0, 0);
		Canvas.LineTo( Width, 0);
	end;

	if frRight in Frames then
	begin
		Canvas.MoveTo( Width - 1, 0);
		Canvas.LineTo( Width - 1, Height);
	end;

	if frBottom in Frames then
	begin
		Canvas.MoveTo( 0, Height - 1);
		Canvas.LineTo( Width, Height - 1);
	end;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReportBand.Print( AReport : TFlexCustomReport);
var
	index : Integer;
begin
	FIterator := 0;
	while ( FIterator = 0) or Iteration do
	begin
		if not InvokeHandler( BeforePrint, self, AReport) then break;

		if Align = baTop then
			AReport.SetCurrentBounds( Rect(
				AReport.CurrentBounds.Left,
				AReport.CurrentBounds.Bottom,
				AReport.CurrentBounds.Right,
				AReport.CurrentBounds.Bottom + Height
			))
		else
			AReport.SetCurrentBounds( Rect(
				AReport.CurrentBounds.Left,
				AReport.CurrentBounds.Top - Height,
				AReport.CurrentBounds.Right,
				AReport.CurrentBounds.Top
			));

		if not Transparent then
		begin
			AReport.Brush := Brush;
			AReport.FillRect( ClientRect);
		end;

		for index := 0 to ControlCount - 1 do
		begin
			if Controls[ index] is TFlexReportCustomItem then
			begin
				if Controls[ index].Enabled then
					( Controls[ index] as TFlexReportCustomItem).Print( AReport);
			end;
		end;

		AReport.Pen := Pen;

		if frLeft in Frames then AReport.Line( 0, 0, 0, Height);
		if frTop in Frames then AReport.Line( 0, 0, Width, 0);
		if frRight in Frames then AReport.Line( Width, 0, Width, Height);
		if frBottom in Frames then AReport.Line( 0, Height, Width, Height);

		InvokeHandler( AfterPrint, self, AReport);

		Inc( FIterator);
		FPrinted := true;
	end;
end;


{*******************************************************************************
  @subject: TFlexReportArranger
  @update: 2004/05/08 (Sat) 00:00:00
  = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =
 *******************************************************************************}

{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

constructor TFlexReportArranger.Create( AOwner : TComponent);
begin
	inherited Create( AOwner);

	BevelInner := bvNone;
	BevelOuter := bvNone;

	FPen := TFlexReportPen.Create;
	FBrush := TFlexReportBrush.Create;
	FFont := TFlexReportFont.Create;
	FRowCount := 2;
	FColCount := 2;
	FDirection := gdColToRow;
	FTransparent := true;

	FPen.OnChange := Material_Changed;
	FBrush.OnChange := Material_Changed;
	FFont.OnChange := Material_Changed;
end;

destructor TFlexReportArranger.Destroy;
begin
	FPen.Free;
	FBrush.Free;
	FFont.Free;

	inherited Destroy;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReportArranger.SetParent( AParent: TWinControl);
begin
	if ( AParent <> nil)
	and ( not ( AParent is TFlexReportBand))
	and ( not ( AParent is TFlexReportCustomPanel))
	then
		raise EFlexReport.Create( FR_ERROR_TYPE_MISMATCH);

	inherited SetParent( AParent);
end;

procedure TFlexReportArranger.SetPen( APen : TFlexReportPen);
begin
	FPen.Assign( APen);
	Invalidate;
end;

procedure TFlexReportArranger.SetBrush( ABrush : TFlexReportBrush);
begin
	FBrush.Assign( ABrush);
	Invalidate;
end;

procedure TFlexReportArranger.SetFont( AFont : TFlexReportFont);
begin
	FFont.Assign( AFont);
	Invalidate;
end;

procedure TFlexReportArranger.SetFrames( AFrames : TFlexReportFrames);
begin
	FFrames := AFrames;
	Invalidate;
end;

procedure TFlexReportArranger.SetRowCount( ACount : Integer);
var
	Rect : TRect;
begin
	if ACount <= 0 then raise EFlexReport.Create( FR_ERROR_INVALID_ROW_COUNT);
	FRowCount := ACount;
	Invalidate;
	AlignControls( nil, Rect);
end;

procedure TFlexReportArranger.SetColCount( ACount : Integer);
var
	Rect : TRect;
begin
	if ACount <= 0 then raise EFlexReport.Create( FR_ERROR_INVALID_COL_COUNT);
	FColCount := ACount;
	Invalidate;
	AlignControls( nil, Rect);
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReportArranger.AlignControls( AControl : TControl; var ARect : TRect);
begin
	ARect := Rect(
		0,
		0,
		Width div ColCount,
		Height div RowCount
	);
	inherited AlignControls( AControl, ARect);
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReportArranger.Paint;
var
	ARow, ACol : Integer;
	Arrangement : TRect;
begin
	Canvas.Pen.Color := clSilver;
	Canvas.Pen.Style := psSolid;
	Canvas.Pen.Width := 1;
	Canvas.Brush.Color := Brush.Color;
	Canvas.Brush.Style := Brush.Style;
	Canvas.Rectangle( ClientRect);
	// default drawing

	Canvas.Pen.Color := clGray;
	Canvas.Pen.Style := psDot;
	Canvas.Pen.Width := 1;

	for ARow := 0 to RowCount - 1 do
	begin
		for ACol := 0 to ColCount - 1 do
		begin
			Arrangement.Left := ACol * Width div ColCount;
			Arrangement.Top := ARow * Height div RowCount;
			Arrangement.Right := ( ACol + 1) * Width div ColCount;
			Arrangement.Bottom := ( ARow + 1) * Height div RowCount;

			Canvas.MoveTo( Arrangement.Right, Arrangement.Top);
			Canvas.LineTo( Arrangement.Right, Arrangement.Bottom);

			Canvas.MoveTo( Arrangement.Left, Arrangement.Bottom);
			Canvas.LineTo( Arrangement.Right, Arrangement.Bottom);
		end;
	end;
	// painting

	Canvas.Pen.Color := Pen.Color;
	Canvas.Pen.Style := Pen.Style;
	Canvas.Pen.Width := Max( 1, Pen.Width * ContainerPaper.DesignScale div 8);

	if frLeft in Frames then
	begin
		Canvas.MoveTo( 0, 0);
		Canvas.LineTo( 0, Height div RowCount);
	end;

	if frTop in Frames then
	begin
		Canvas.MoveTo( 0, 0);
		Canvas.LineTo( Width div ColCount, 0);
	end;

	if frRight in Frames then
	begin
		Canvas.MoveTo( ( Width div ColCount) - 1, 0);
		Canvas.LineTo( ( Width div ColCount) - 1, Height div RowCount);
	end;

	if frBottom in Frames then
	begin
		Canvas.MoveTo( 0, ( Height div RowCount) - 1);
		Canvas.LineTo( Width div ColCount, ( Height div RowCount) - 1);
	end;
	// frame drawing
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReportArranger.Print( AReport : TFlexCustomReport);

	function InvokeArrangeHandler( Handler : TFlexArrangeEvent; Sender: TObject; ARow, ACol : Integer; var Arrangement : TFlexReportArrangement) : Boolean;
	begin
		result := true;
		try
			if @Handler <> nil then Handler( Sender, ARow, ACol, Arrangement);
		except
			on E : EAbort do result := false;
		else
			raise;
		end;
	end;

	procedure PrintArrangement( var Arrangement : TFlexReportArrangement);
	var
		index : Integer;
	begin
		AReport.Pen := Arrangement.Pen;
		AReport.Brush := Arrangement.Brush;
		AReport.Font := Arrangement.Font;
		// matiere setup

		if not Transparent then AReport.FillRect( Arrangement.Left, Arrangement.Top, Arrangement.Right, Arrangement.Bottom);

		for index := 0 to ControlCount - 1 do
		begin
			if Controls[ index] is TFlexReportCustomItem then
				if Controls[ index].Enabled then
					( Controls[ index] as TFlexReportCustomItem).Print(
						Arrangement.Left - AbsoluteLeft,
						Arrangement.Top - AbsoluteTop,
						AReport
					);
		end;
		// child drawing

		AReport.Pen := Arrangement.Pen;

		if frLeft in Arrangement.Frames then
		begin
			AReport.Line( Arrangement.Left, Arrangement.Top, Arrangement.Left, Arrangement.Bottom);
		end;

		if frTop in Arrangement.Frames then
		begin
			AReport.Line( Arrangement.Left, Arrangement.Top, Arrangement.Right, Arrangement.Top);
		end;

		if frRight in Arrangement.Frames then
		begin
			AReport.Line( Arrangement.Right, Arrangement.Top, Arrangement.Right, Arrangement.Bottom);
		end;

		if frBottom in Arrangement.Frames then
		begin
			AReport.Line( Arrangement.Left, Arrangement.Bottom, Arrangement.Right, Arrangement.Bottom);
		end;
		// frame drawing
	end;
var
	ARow, ACol : Integer;
	Arrangement : TFlexReportArrangement;
begin
	if not InvokeHandler( BeforePrint, self, AReport) then exit;

	Arrangement.Pen := TFlexReportPen.Create;
	Arrangement.Pen.Assign( Pen);
	Arrangement.Brush := TFlexReportBrush.Create;
	Arrangement.Brush.Assign( Brush);
	Arrangement.Font := TFlexReportFont.Create;
	Arrangement.Font.Assign( Font);
	Arrangement.Frames := Frames;
	// arrangement descriptor setup

	if Direction = gdColToRow then
	begin
		for ARow := 0 to RowCount - 1 do
		begin
			for ACol := 0 to ColCount - 1 do
			begin
				Arrangement.Left := AbsoluteLeft + ACol * Width div ColCount;
				Arrangement.Top := AbsoluteTop + ARow * Height div RowCount;
				Arrangement.Right := AbsoluteLeft + ( ACol + 1) * Width div ColCount;
				Arrangement.Bottom := AbsoluteTop + ( ARow + 1) * Height div RowCount;
				Arrangement.Width := Arrangement.Right - Arrangement.Left;
				Arrangement.Height := Arrangement.Bottom - Arrangement.Top;
				InvokeArrangeHandler( FBeforeArrange, self, ARow, ACol, Arrangement);
				PrintArrangement( Arrangement);
				InvokeArrangeHandler( FAfterArrange, self, ARow, ACol, Arrangement);
			end;
		end;
	end
	else
	begin
		for ACol := 0 to ColCount - 1 do
		begin
			for ARow := 0 to RowCount - 1 do
			begin
				Arrangement.Left := AbsoluteLeft + ACol * Width div ColCount;
				Arrangement.Top := AbsoluteTop + ARow * Height div RowCount;
				Arrangement.Right := AbsoluteLeft + ( ACol + 1) * Width div ColCount;
				Arrangement.Bottom := AbsoluteTop + ( ARow + 1) * Height div RowCount;
				Arrangement.Width := Arrangement.Right - Arrangement.Left;
				Arrangement.Height := Arrangement.Bottom - Arrangement.Top;
				InvokeArrangeHandler( FBeforeArrange, self, ARow, ACol, Arrangement);
				PrintArrangement( Arrangement);
				InvokeArrangeHandler( FAfterArrange, self, ARow, ACol, Arrangement);
			end;
		end;
	end;
	// painting

	Arrangement.Pen.Free;
	Arrangement.Brush.Free;
	Arrangement.Font.Free;
	// arrangement descriptor cleanup

	InvokeHandler( AfterPrint, self, AReport);

	FPrinted := true;
end;


{*******************************************************************************
  @subject: TFlexReportPanel
  @update: 2004/05/08 (Sat) 00:00:00
  = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =
 *******************************************************************************}

{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

constructor TFlexReportPanel.Create( AOwner : TComponent);
begin
	inherited Create( AOwner);

	BevelInner := bvNone;
	BevelOuter := bvNone;

	FPen := TFlexReportPen.Create;
	FBrush := TFlexReportBrush.Create;
	FFont := TFlexReportFont.Create;
	FHorzAlignment := haLeft;
	FVertAlignment := vaCenter;
	FHorzSpacing := 1;
	FVertSpacing := 1;
	FLineSpacing := 0;
	FLines := TStringList.Create;
	FTransparent := true;

	FPen.OnChange := Material_Changed;
	FBrush.OnChange := Material_Changed;
	FFont.OnChange := Material_Changed;
end;

destructor TFlexReportPanel.Destroy;
begin
	FPen.Free;
	FBrush.Free;
	FFont.Free;
	FLines.Free;

	inherited Destroy;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReportPanel.SetParent( AParent: TWinControl);
begin
	if ( AParent <> nil)
	and ( not ( AParent is TFlexReportBand))
	and ( not ( AParent is TFlexReportCustomPanel))
	then
		raise EFlexReport.Create( FR_ERROR_TYPE_MISMATCH);

	inherited SetParent( AParent);
end;

procedure TFlexReportPanel.SetPen( APen : TFlexReportPen);
begin
	FPen.Assign( APen);
	Invalidate;
end;

procedure TFlexReportPanel.SetBrush( ABrush : TFlexReportBrush);
begin
	FBrush.Assign( ABrush);
	Invalidate;
end;

procedure TFlexReportPanel.SetFont( AFont : TFlexReportFont);
begin
	FFont.Assign( AFont);
	Invalidate;
end;

procedure TFlexReportPanel.SetFrames( AFrames : TFlexReportFrames);
begin
	FFrames := AFrames;
	Invalidate;
end;

procedure TFlexReportPanel.SetHorzAlignment( AnAlignment : TFlexReportHorzAlignment);
begin
	FHorzAlignment := AnAlignment;
	Invalidate;
end;

procedure TFlexReportPanel.SetVertAlignment( AnAlignment : TFlexReportVertAlignment);
begin
	FVertAlignment := AnAlignment;
	Invalidate;
end;

procedure TFlexReportPanel.SetHorzSpacing( ASpacing : Integer);
begin
	FHorzSpacing := ASpacing;
	Invalidate;
end;

procedure TFlexReportPanel.SetVertSpacing( ASpacing : Integer);
begin
	FVertSpacing := ASpacing;
	Invalidate;
end;

procedure TFlexReportPanel.SetLineSpacing( ASpacing : Integer);
begin
	FLineSpacing := ASpacing;
	Invalidate;
end;

procedure TFlexReportPanel.SetLines( ALines : TStrings);
begin
	FLines.Assign( ALines);
	Invalidate;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function TFlexReportPanel.ParseString( const Source : String) : String;
begin
	result := Source;
	if not ( csDesigning in ComponentState) then
		if @FParseString <> nil then FParseString( self, result);
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReportPanel.Paint;
var
	X, Y : Integer;
	BodyHeight, LineHeight, LineIndex : Integer;
	ClipRect : TRect;
begin
	Canvas.Pen.Color := clSilver;
	Canvas.Pen.Style := psSolid;
	Canvas.Pen.Width := 1;
	Canvas.Brush.Color := Brush.Color;
	Canvas.Brush.Style := Brush.Style;
	Canvas.Rectangle( ClientRect);
	// default drawing

	Canvas.Pen.Color := Pen.Color;
	Canvas.Pen.Style := Pen.Style;
	Canvas.Pen.Width := Max( 1, Pen.Width * ContainerPaper.DesignScale div 8);

	Canvas.Brush.Color := Brush.Color;
	Canvas.Brush.Style := Brush.Style;

	Canvas.Font.Color := Font.Color;
	Canvas.Font.Charset := Font.Charset;
	Canvas.Font.Name := Font.Name;
	Canvas.Font.Pitch := Font.Pitch;
	Canvas.Font.Style := Font.Style;
	Canvas.Font.Height := - Ceil( Font.Height * 275 * ContainerPaper.DesignScale div 7200);
	// matiere setup

	BodyHeight := Canvas.TextHeight( 'W');
	LineHeight := BodyHeight + LineSpacing * ContainerPaper.DesignScale;

	case VertAlignment of
	vaTop : Y := VertSpacing * ContainerPaper.DesignScale;
	vaBottom : Y := Height - BodyHeight - LineHeight * Max( 0, Lines.Count - 1) - VertSpacing * ContainerPaper.DesignScale;
	vaCenter : Y := ( Height - BodyHeight - LineHeight * Max( 0, Lines.Count - 1)) div 2;
	end;
	// origin calculation

	ClipRect := Rect(
		HorzSpacing * ContainerPaper.DesignScale,
		VertSpacing * ContainerPaper.DesignScale,
		Width - HorzSpacing * ContainerPaper.DesignScale,
		Height - VertSpacing * ContainerPaper.DesignScale
	);

	Canvas.Brush.Style := bsClear;
	for LineIndex := 0 to Lines.Count - 1 do
	begin
		case HorzAlignment of
		haLeft : X := HorzSpacing * ContainerPaper.DesignScale;
		haRight : X := Width - Canvas.TextWidth( ParseString( Lines[ LineIndex])) - HorzSpacing * ContainerPaper.DesignScale;
		haCenter : X := ( Width - Canvas.TextWidth( ParseString( Lines[ LineIndex]))) div 2;
		end;

		Canvas.TextRect( ClipRect, X, Y, ParseString( Lines[ LineIndex]));
		Y := Y + LineHeight;
	end;
	// text drawing

	if frLeft in Frames then
	begin
		Canvas.MoveTo( 0, 0);
		Canvas.LineTo( 0, Height);
	end;

	if frTop in Frames then
	begin
		Canvas.MoveTo( 0, 0);
		Canvas.LineTo( Width, 0);
	end;

	if frRight in Frames then
	begin
		Canvas.MoveTo( Width - 1, 0);
		Canvas.LineTo( Width - 1, Height);
	end;

	if frBottom in Frames then
	begin
		Canvas.MoveTo( 0, Height - 1);
		Canvas.LineTo( Width, Height - 1);
	end;
	// frame drawing
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReportPanel.Print( AReport : TFlexCustomReport);
var
	X, Y : Integer;
	BodyHeight, LineHeight, LineIndex, index : Integer;
	ClipRect : TRect;
begin
	if not InvokeHandler( BeforePrint, self, AReport) then exit;

	AReport.Pen := Pen;
	AReport.Brush := Brush;
	AReport.Font := Font;
	// matiere setup

	if not Transparent then AReport.FillRect( AbsoluteRect);

	BodyHeight := AReport.TextHeight( 'W');
	LineHeight := BodyHeight + LineSpacing * ContainerPaper.DesignScale;

	Y := AbsoluteTop;
	case VertAlignment of
	vaTop : Y := Y + VertSpacing * ContainerPaper.DesignScale;
	vaBottom : Y := Y + Height - BodyHeight - LineHeight * Max( 0, Lines.Count - 1) - VertSpacing * ContainerPaper.DesignScale;
	vaCenter : Y := Y + ( Height - BodyHeight - LineHeight * Max( 0, Lines.Count - 1)) div 2;
	end;
	// origin calculation

	ClipRect := Rect(
		AbsoluteLeft + HorzSpacing * ContainerPaper.DesignScale,
		AbsoluteTop + VertSpacing * ContainerPaper.DesignScale,
		AbsoluteLeft + Width - HorzSpacing * ContainerPaper.DesignScale,
		AbsoluteTop + Height - VertSpacing * ContainerPaper.DesignScale
	);

	AReport.Brush.Style := bsClear;
	for LineIndex := 0 to Lines.Count - 1 do
	begin
		X := AbsoluteLeft;
		case HorzAlignment of
		haLeft : X := X + HorzSpacing * ContainerPaper.DesignScale;
		haRight : X := X + Width - AReport.TextWidth( ParseString( Lines[ LineIndex])) - HorzSpacing * ContainerPaper.DesignScale;
		haCenter : X := X + ( Width - AReport.TextWidth( ParseString( Lines[ LineIndex]))) div 2;
		end;

		AReport.TextRect( ClipRect, X, Y, ParseString( Lines[ LineIndex]), true);
		Y := Y + LineHeight;
	end;
	// text drawing

	for index := 0 to ControlCount - 1 do
	begin
		if Controls[ index] is TFlexReportCustomItem then
			if Controls[ index].Enabled then
				( Controls[ index] as TFlexReportCustomItem).Print( AReport);
	end;
	// child drawing

	AReport.Pen := Pen;

	if frLeft in Frames then AReport.Line( AbsoluteLeft, AbsoluteTop, AbsoluteLeft, AbsoluteBottom);
	if frTop in Frames then AReport.Line( AbsoluteLeft, AbsoluteTop, AbsoluteRight, AbsoluteTop);
	if frRight in Frames then AReport.Line( AbsoluteRight, AbsoluteTop, AbsoluteRight, AbsoluteBottom);
	if frBottom in Frames then AReport.Line( AbsoluteLeft, AbsoluteBottom, AbsoluteRight, AbsoluteBottom);
	// frame drawing

	InvokeHandler( AfterPrint, self, AReport);

	FPrinted := true;
end;


{*******************************************************************************
  @subject: TFlexReportInteger
  @update: 2004/05/08 (Sat) 00:00:00
  = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =
 *******************************************************************************}

{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReportInteger.SetValue( AValue : Integer);
begin
	FValue := AValue;
	Invalidate;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function TFlexReportInteger.ParseString( const Source : String) : String;
begin
	result := inherited ParseString( Format( Source, [ Value]));
end;


{*******************************************************************************
  @subject: TFlexReportDecimal
  @update: 2004/05/08 (Sat) 00:00:00
  = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =
 *******************************************************************************}

{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReportDecimal.SetValue( AValue : Double);
begin
	FValue := AValue;
	Invalidate;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function TFlexReportDecimal.ParseString( const Source : String) : String;
begin
	result := inherited ParseString( Format( Source, [ Value]));
end;


{*******************************************************************************
  @subject: TFlexReportPageNumber
  @update: 2004/05/08 (Sat) 00:00:00
  = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =
 *******************************************************************************}

{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function TFlexReportPageNumber.ParseString( const Source : String) : String;
begin
	if FCurrentReport <> nil then
		result := inherited ParseString( Format( Source, [ FCurrentReport.PageNumber]))
	else
		result := inherited ParseString( Format( Source, [ 999]));
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReportPageNumber.Print( AReport : TFlexCustomReport);
begin
	FCurrentReport := AReport;
	inherited Print( AReport);
	FCurrentReport := nil;
end;


{*******************************************************************************
  @subject: TFlexReportDateTime
  @update: 2004/05/08 (Sat) 00:00:00
  = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =
 *******************************************************************************}

{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReportDateTime.SetAutoToday( AValue : Boolean);
begin
	FAutoToday := AValue;
	Invalidate;
end;

procedure TFlexReportDateTime.SetValue( AValue : TDateTime);
begin
	FValue := AValue;
	Invalidate;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function TFlexReportDateTime.ParseString( const Source : String) : String;
begin
	if AutoToday then
		result := inherited ParseString( FormatDateTime( Source, Today))
	else
		result := inherited ParseString( FormatDateTime( Source, Value));
end;


{*******************************************************************************
  @subject: TFlexReportScoreboard
  @update: 2004/05/08 (Sat) 00:00:00
  = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =
 *******************************************************************************}

{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

constructor TFlexReportScoreboard.Create( AOwner : TComponent);
begin
	inherited Create( AOwner);

	BevelInner := bvNone;
	BevelOuter := bvNone;

	FPen := TFlexReportPen.Create;
	FDivider := TFlexReportPen.Create;
	FBrush := TFlexReportBrush.Create;
	FFont := TFlexReportFont.Create;
	FPartitions := 1;
	FTransparent := true;

	FPen.OnChange := Material_Changed;
	FBrush.OnChange := Material_Changed;
	FFont.OnChange := Material_Changed;
end;

destructor TFlexReportScoreboard.Destroy;
begin
	FPen.Free;
	FDivider.Free;
	FBrush.Free;
	FFont.Free;

	inherited Destroy;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReportScoreboard.SetParent( AParent: TWinControl);
begin
	if ( AParent <> nil)
	and ( not ( AParent is TFlexReportBand))
	and ( not ( AParent is TFlexReportCustomPanel))
	then
		raise EFlexReport.Create( FR_ERROR_TYPE_MISMATCH);

	inherited SetParent( AParent);
end;

procedure TFlexReportScoreboard.SetPen( APen : TFlexReportPen);
begin
	FPen.Assign( APen);
	Invalidate;
end;

procedure TFlexReportScoreboard.SetDivider( ADivider : TFlexReportPen);
begin
	FDivider.Assign( ADivider);
	Invalidate;
end;

procedure TFlexReportScoreboard.SetBrush( ABrush : TFlexReportBrush);
begin
	FBrush.Assign( ABrush);
	Invalidate;
end;

procedure TFlexReportScoreboard.SetFont( AFont : TFlexReportFont);
begin
	FFont.Assign( AFont);
	Invalidate;
end;

procedure TFlexReportScoreboard.SetFrames( AFrames : TFlexReportFrames);
begin
	FFrames := AFrames;
	Invalidate;
end;

procedure TFlexReportScoreboard.SetPartitions( APartitions : Integer);
begin
	if APartitions <= 0 then raise EFlexReport.Create( FR_ERROR_INVALID_COL_COUNT);
	FPartitions := APartitions;
	Invalidate;
end;

procedure TFlexReportScoreboard.SetLetters( AValue : String);
begin
	FLetters := AValue;
	Invalidate;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReportScoreboard.Paint;

	function Letter( index : Integer) : String;
	begin
		if ( index < 0) or ( index >= Length( Letters)) then
			result := ' '
		else
			result := Letters[ index + 1];
	end;
var
	index : Integer;
	X, Y : Integer;
	CellRect : TRect;
begin
	Canvas.Pen.Color := clSilver;
	Canvas.Pen.Style := psSolid;
	Canvas.Pen.Width := 1;
	Canvas.Brush.Color := Brush.Color;
	Canvas.Brush.Style := Brush.Style;
	Canvas.Rectangle( ClientRect);
	// default drawing

	Canvas.Pen.Color := Divider.Color;
	Canvas.Pen.Style := Divider.Style;
	Canvas.Pen.Width := Max( 1, Divider.Width * ContainerPaper.DesignScale div 8);

	Canvas.Font.Color := Font.Color;
	Canvas.Font.Charset := Font.Charset;
	Canvas.Font.Name := Font.Name;
	Canvas.Font.Pitch := Font.Pitch;
	Canvas.Font.Style := Font.Style;
	Canvas.Font.Height := - Ceil( Font.Height * 275 * ContainerPaper.DesignScale div 7200);
	// matiere setup

	Y := ( Height - Canvas.TextHeight( Letters)) div 2;

	Canvas.Brush.Style := bsClear;

	for index := 0 to Partitions - 1 do
	begin
		CellRect.Left := index * Width div Partitions;
		CellRect.Top := 0;
		CellRect.Right := ( index + 1) * Width div Partitions;
		CellRect.Bottom := Height;
		X := CellRect.Left + ( Width div Partitions - Canvas.TextWidth( Letter( index))) div 2;

		Canvas.TextRect( CellRect, X, Y, Letter( index));

		if index > 0 then
		begin
			Canvas.MoveTo( index * Width div Partitions, 0);
			Canvas.LineTo( index * Width div Partitions, Height);
		end;
	end;
	// painting

	Canvas.Pen.Color := Pen.Color;
	Canvas.Pen.Style := Pen.Style;
	Canvas.Pen.Width := Max( 1, Pen.Width * ContainerPaper.DesignScale div 8);

	if frLeft in Frames then
	begin
		Canvas.MoveTo( 0, 0);
		Canvas.LineTo( 0, Height);
	end;

	if frTop in Frames then
	begin
		Canvas.MoveTo( 0, 0);
		Canvas.LineTo( Width, 0);
	end;

	if frRight in Frames then
	begin
		Canvas.MoveTo( Width - 1, 0);
		Canvas.LineTo( Width - 1, Height);
	end;

	if frBottom in Frames then
	begin
		Canvas.MoveTo( 0, Height - 1);
		Canvas.LineTo( Width, Height - 1);
	end;
	// frame drawing
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReportScoreboard.Print( AReport : TFlexCustomReport);

	function Letter( index : Integer) : String;
	begin
		if ( index < 0) or ( index >= Length( Letters)) then
			result := ' '
		else
			result := Letters[ index + 1];
	end;
var
	index : Integer;
	X, Y : Integer;
	CellRect : TRect;
begin
	if not InvokeHandler( BeforePrint, self, AReport) then exit;

	AReport.Pen := Divider;
	AReport.Brush := Brush;
	AReport.Font := Font;
	// matiere setup

	if not Transparent then AReport.FillRect( AbsoluteRect);

	Y := AbsoluteTop + ( Height - AReport.TextHeight( Letters)) div 2;

	AReport.Brush.Style := bsClear;
	for index := 0 to Partitions - 1 do
	begin
		CellRect.Left := AbsoluteLeft + index * Width div Partitions;
		CellRect.Top := AbsoluteTop;
		CellRect.Right := AbsoluteLeft + ( index + 1) * Width div Partitions;
		CellRect.Bottom := AbsoluteBottom;
		X := CellRect.Left + ( Width div Partitions - AReport.TextWidth( Letter( index))) div 2;

		AReport.TextRect( CellRect, X, Y, Letter( index), true);

		if index > 0 then
		begin
			AReport.Line( CellRect.Left, CellRect.Top, CellRect.Left, CellRect.Bottom);
		end;
	end;
	// painting

	AReport.Pen := Pen;

	if frLeft in Frames then AReport.Line( AbsoluteLeft, AbsoluteTop, AbsoluteLeft, AbsoluteBottom);
	if frTop in Frames then AReport.Line( AbsoluteLeft, AbsoluteTop, AbsoluteRight, AbsoluteTop);
	if frRight in Frames then AReport.Line( AbsoluteRight, AbsoluteTop, AbsoluteRight, AbsoluteBottom);
	if frBottom in Frames then AReport.Line( AbsoluteLeft, AbsoluteBottom, AbsoluteRight, AbsoluteBottom);
	// frame drawing

	InvokeHandler( AfterPrint, self, AReport);

	FPrinted := true;
end;


{*******************************************************************************
  @subject: TFlexReportGrid
  @update: 2004/05/08 (Sat) 00:00:00
  = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =
 *******************************************************************************}

{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

constructor TFlexReportGrid.Create( AOwner : TComponent);
begin
	inherited Create( AOwner);

	BevelInner := bvNone;
	BevelOuter := bvNone;

	FPen := TFlexReportPen.Create;
	FBrush := TFlexReportBrush.Create;
	FFont := TFlexReportFont.Create;
	FRowCount := 1;
	FColCount := 1;
	FDirection := gdColToRow;
	FTransparent := true;

	FPen.OnChange := Material_Changed;
	FBrush.OnChange := Material_Changed;
	FFont.OnChange := Material_Changed;
end;

destructor TFlexReportGrid.Destroy;
begin
	FPen.Free;
	FBrush.Free;
	FFont.Free;

	inherited Destroy;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReportGrid.SetParent( AParent: TWinControl);
begin
	if ( AParent <> nil)
	and ( not ( AParent is TFlexReportBand))
	and ( not ( AParent is TFlexReportCustomPanel))
	then
		raise EFlexReport.Create( FR_ERROR_TYPE_MISMATCH);

	inherited SetParent( AParent);
end;

procedure TFlexReportGrid.SetPen( APen : TFlexReportPen);
begin
	FPen.Assign( APen);
	Invalidate;
end;

procedure TFlexReportGrid.SetBrush( ABrush : TFlexReportBrush);
begin
	FBrush.Assign( ABrush);
	Invalidate;
end;

procedure TFlexReportGrid.SetFont( AFont : TFlexReportFont);
begin
	FFont.Assign( AFont);
	Invalidate;
end;

procedure TFlexReportGrid.SetFrames( AFrames : TFlexReportFrames);
begin
	FFrames := AFrames;
	Invalidate;
end;

procedure TFlexReportGrid.SetRowCount( ACount : Integer);
begin
	if ACount <= 0 then raise EFlexReport.Create( FR_ERROR_INVALID_ROW_COUNT);
	FRowCount := ACount;
	Invalidate;
end;

procedure TFlexReportGrid.SetColCount( ACount : Integer);
begin
	if ACount <= 0 then raise EFlexReport.Create( FR_ERROR_INVALID_COL_COUNT);
	FColCount := ACount;
	Invalidate;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReportGrid.Paint;

	procedure PaintCell( var Cell : TFlexReportCell);
	begin
		Canvas.Pen.Color := Cell.Pen.Color;
		Canvas.Pen.Style := Cell.Pen.Style;
		Canvas.Pen.Width := Max( 1, Cell.Pen.Width * ContainerPaper.DesignScale div 8);

		Canvas.Brush.Color := Cell.Brush.Color;
		Canvas.Brush.Style := Cell.Brush.Style;
		// matiere setup

		if not Transparent then Canvas.FillRect( Rect( Cell.Left, Cell.Top, Cell.Right, Cell.Bottom));

		if frLeft in Cell.Frames then
		begin
			Canvas.MoveTo( Cell.Left, Cell.Top);
			Canvas.LineTo( Cell.Left, Cell.Bottom);
		end;

		if frTop in Cell.Frames then
		begin
			Canvas.MoveTo( Cell.Left, Cell.Top);
			Canvas.LineTo( Cell.Right, Cell.Top);
		end;

		if frRight in Cell.Frames then
		begin
			Canvas.MoveTo( Cell.Right, Cell.Top);
			Canvas.LineTo( Cell.Right, Cell.Bottom);
		end;

		if frBottom in Cell.Frames then
		begin
			Canvas.MoveTo( Cell.Left, Cell.Bottom);
			Canvas.LineTo( Cell.Right, Cell.Bottom);
		end;
		// frame drawing
	end;
var
	ARow, ACol : Integer;
	Cell : TFlexReportCell;
begin
	Canvas.Pen.Color := clSilver;
	Canvas.Pen.Style := psSolid;
	Canvas.Pen.Width := 1;
	Canvas.Brush.Color := Brush.Color;
	Canvas.Brush.Style := Brush.Style;
	Canvas.Rectangle( ClientRect);
	// default drawing

	Cell.Pen := TFlexReportPen.Create;
	Cell.Brush := TFlexReportBrush.Create;
	Cell.Font := TFlexReportFont.Create;
	Cell.Frames := [ frRight, frBottom];
	Cell.HorzAlignment := haCenter;
	Cell.VertAlignment := vaCenter;
	Cell.HorzSpacing := 2;
	Cell.VertSpacing := 2;
	Cell.LineSpacing := 1;
	// cell descriptor setup

	for ARow := 0 to RowCount - 1 do
	begin
		for ACol := 0 to ColCount - 1 do
		begin
			Cell.Left := ACol * Width div ColCount;
			Cell.Top := ARow * Height div RowCount;
			Cell.Right := ( ACol + 1) * Width div ColCount;
			Cell.Bottom := ( ARow + 1) * Height div RowCount;
			Cell.Width := Cell.Right - Cell.Left;
			Cell.Height := Cell.Bottom - Cell.Top;
			PaintCell( Cell);
		end;
	end;
	// painting

	Cell.Pen.Free;
	Cell.Brush.Free;
	Cell.Font.Free;
	// cell descriptor cleanup

	Canvas.Pen.Color := Pen.Color;
	Canvas.Pen.Style := Pen.Style;
	Canvas.Pen.Width := Max( 1, Pen.Width * ContainerPaper.DesignScale div 8);

	if frLeft in Frames then
	begin
		Canvas.MoveTo( 0, 0);
		Canvas.LineTo( 0, Height);
	end;

	if frTop in Frames then
	begin
		Canvas.MoveTo( 0, 0);
		Canvas.LineTo( Width, 0);
	end;

	if frRight in Frames then
	begin
		Canvas.MoveTo( Width - 1, 0);
		Canvas.LineTo( Width - 1, Height);
	end;

	if frBottom in Frames then
	begin
		Canvas.MoveTo( 0, Height - 1);
		Canvas.LineTo( Width, Height - 1);
	end;
	// frame drawing
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TFlexReportGrid.Print( AReport : TFlexCustomReport);

	procedure PrintCell( var Cell : TFlexReportCell);
	var
		X, Y : Integer;
		BodyHeight, LineHeight, LineIndex : Integer;
		ClipRect : TRect;
	begin
		AReport.Pen := Cell.Pen;
		AReport.Brush := Cell.Brush;
		AReport.Font := Cell.Font;
		// matiere setup

		if not Transparent then AReport.FillRect( Cell.Left, Cell.Top, Cell.Right, Cell.Bottom);

		BodyHeight := AReport.TextHeight( 'W');
		LineHeight := BodyHeight + Cell.LineSpacing * ContainerPaper.DesignScale;

		Y := Cell.Top;
		case Cell.VertAlignment of
		vaTop : Y := Y + Cell.VertSpacing * ContainerPaper.DesignScale;
		vaBottom : Y := Y + Cell.Height - BodyHeight - LineHeight * Max( 0, Cell.Lines.Count - 1) - Cell.VertSpacing * ContainerPaper.DesignScale;
		vaCenter : Y := Y + ( Cell.Height - BodyHeight - LineHeight * Max( 0, Cell.Lines.Count - 1)) div 2;
		end;
		// origin calculation

		ClipRect := Rect(
			Cell.Left + Cell.HorzSpacing * ContainerPaper.DesignScale,
			Cell.Top + Cell.VertSpacing * ContainerPaper.DesignScale,
			Cell.Left + Cell.Width - Cell.HorzSpacing * ContainerPaper.DesignScale,
			Cell.Top + Cell.Height - Cell.VertSpacing * ContainerPaper.DesignScale
		);

		AReport.Brush.Style := bsClear;
		for LineIndex := 0 to Cell.Lines.Count - 1 do
		begin
			X := Cell.Left;
			case Cell.HorzAlignment of
			haLeft : X := X + Cell.HorzSpacing * ContainerPaper.DesignScale;
			haRight : X := X + Cell.Width - AReport.TextWidth( Cell.Lines[ LineIndex]) - Cell.HorzSpacing * ContainerPaper.DesignScale;
			haCenter : X := X + ( Cell.Width - AReport.TextWidth( Cell.Lines[ LineIndex])) div 2;
			end;

			AReport.TextRect( ClipRect, X, Y, Cell.Lines[ LineIndex], true);
			Y := Y + LineHeight;
		end;
		// text drawing

		if frLeft in Cell.Frames then
		begin
			AReport.Line( Cell.Left, Cell.Top, Cell.Left, Cell.Bottom);
		end;

		if frTop in Cell.Frames then
		begin
			AReport.Line( Cell.Left, Cell.Top, Cell.Right, Cell.Top);
		end;

		if frRight in Cell.Frames then
		begin
			AReport.Line( Cell.Right, Cell.Top, Cell.Right, Cell.Bottom);
		end;

		if frBottom in Cell.Frames then
		begin
			AReport.Line( Cell.Left, Cell.Bottom, Cell.Right, Cell.Bottom);
		end;
		// frame drawing
	end;
var
	ARow, ACol : Integer;
	Cell : TFlexReportCell;
begin
	if not InvokeHandler( BeforePrint, self, AReport) then exit;

	Cell.Pen := TFlexReportPen.Create;
	Cell.Pen.Assign( Pen);
	Cell.Brush := TFlexReportBrush.Create;
	Cell.Brush.Assign( Brush);
	Cell.Font := TFlexReportFont.Create;
	Cell.Font.Assign( Font);
	Cell.Frames := [ frRight, frBottom];
	Cell.HorzAlignment := haCenter;
	Cell.VertAlignment := vaCenter;
	Cell.HorzSpacing := 1;
	Cell.VertSpacing := 1;
	Cell.LineSpacing := 1;
	Cell.Lines := TStringList.Create;
	// cell descriptor setup

	if Direction = gdColToRow then
	begin
		for ARow := 0 to RowCount - 1 do
		begin
			for ACol := 0 to ColCount - 1 do
			begin
				Cell.Left := AbsoluteLeft + ACol * Width div ColCount;
				Cell.Top := AbsoluteTop + ARow * Height div RowCount;
				Cell.Right := AbsoluteLeft + ( ACol + 1) * Width div ColCount;
				Cell.Bottom := AbsoluteTop + ( ARow + 1) * Height div RowCount;
				Cell.Width := Cell.Right - Cell.Left;
				Cell.Height := Cell.Bottom - Cell.Top;
				Cell.Lines.Clear;
				if @FFilloutCell <> nil then FFilloutCell( self, ARow, ACol, Cell);
				PrintCell( Cell);
			end;
		end;
	end
	else
	begin
		for ACol := 0 to ColCount - 1 do
		begin
			for ARow := 0 to RowCount - 1 do
			begin
				Cell.Left := AbsoluteLeft + ACol * Width div ColCount;
				Cell.Top := AbsoluteTop + ARow * Height div RowCount;
				Cell.Right := AbsoluteLeft + ( ACol + 1) * Width div ColCount;
				Cell.Bottom := AbsoluteTop + ( ARow + 1) * Height div RowCount;
				Cell.Width := Cell.Right - Cell.Left;
				Cell.Height := Cell.Bottom - Cell.Top;
				Cell.Lines.Clear;
				if @FFilloutCell <> nil then FFilloutCell( self, ARow, ACol, Cell);
				PrintCell( Cell);
			end;
		end;
	end;
	// painting

	Cell.Pen.Free;
	Cell.Brush.Free;
	Cell.Font.Free;
	Cell.Lines.Free;
	// cell descriptor cleanup

	AReport.Pen := Pen;

	if frLeft in Frames then AReport.Line( AbsoluteLeft, AbsoluteTop, AbsoluteLeft, AbsoluteBottom);
	if frTop in Frames then AReport.Line( AbsoluteLeft, AbsoluteTop, AbsoluteRight, AbsoluteTop);
	if frRight in Frames then AReport.Line( AbsoluteRight, AbsoluteTop, AbsoluteRight, AbsoluteBottom);
	if frBottom in Frames then AReport.Line( AbsoluteLeft, AbsoluteBottom, AbsoluteRight, AbsoluteBottom);
	// frame drawing

	InvokeHandler( AfterPrint, self, AReport);

	FPrinted := true;
end;

end.
