unit Frame_ClientPrivateGraph;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms,
  Dialogs, MainDatastore, DB, DBClient, Provider, IBCustomDataSet, IBQuery,
  StdCtrls, Grids, DBCtrls, IBSQL, IBDatabase, DBLocal, DBLocalI, Math, ImgList,
  DateUtils, StrUtils, ActnList, Misc_Constants, Misc_Utilities, Base_Frame;

type
  TClientPrivateGraphFrame = class(TFrameTemplate)
    Profile: TIBQuery;
    Schedule: TIBClientDataSet;
    Selection: TIBClientDataSet;
    DataSourceOf_Selection: TDataSource;

    iglIcon: TImageList;

    txtProfileInfo: TEdit;
    btnSelectProfile: TButton;
    btnChooseProfile: TButton;

    cboSelection: TDBLookupComboBox;
    btnSelectSchedule: TButton;
    btnDeleteSchedule: TButton;

    gridGraph: TStringGrid;

    Shortcuts: TActionList;
    Shortcut_SelectProfile: TAction;

    procedure btnSelectProfile_Click(Sender: TObject);
    procedure btnChooseProfile_Click(Sender: TObject);

    procedure cboSelection_Click(Sender: TObject);
    procedure btnSelectSchedule_Click(Sender: TObject);
    procedure btnDeleteSchedule_Click(Sender: TObject);

    procedure gridGraph_DrawCell(Sender: TObject; ACol, ARow: Integer; Rect: TRect; State: TGridDrawState);

    procedure gridGraph_SelectCell(Sender: TObject; ACol, ARow: Integer; var CanSelect: Boolean);
    procedure gridGraph_MouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure gridGraph_DblClick(Sender: TObject);
    procedure gridGraph_KeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);

    procedure gridGraph_DragOver(Sender, Source: TObject; X, Y: Integer; State: TDragState; var Accept: Boolean);
    procedure gridGraph_DragDrop(Sender, Source: TObject; X, Y: Integer);
  private
    { Private 錾 }
    FClientKey : Integer;
    FScheduleFrom : TDate;
    FScheduleTo : TDate;
    FObsolete : Boolean;

    FCursorCol : Integer;

    FCalendars : array of TCalendarPacket;
    FSchedules : array of TSchedulePacket;

    sc_keyInstance, sc_typProgress : Integer;
    sc_datSchedule, sc_typSchedule, sc_strService, sc_intMinutesFrom, sc_intMinutesTo : Integer;
    sc_refSubject, sc_refSubjectSubject, sc_typSubject, sc_intSubjectAt, sc_intSubjectFor, sc_refSubjectServer : Integer;
    sc_typAddressFrom, sc_refAddressFrom, sc_strAddressFrom, sc_refResidentFrom : Integer;
    sc_typAddressTo, sc_refAddressTo, sc_strAddressTo, sc_refResidentTo : Integer;
    sc_refClient, sc_strClient : Integer;
    sc_refServer, sc_strServer : Integer;
    sc_strOpponent : Integer;

    property CursorCol : Integer read FCursorCol;
  protected
    { Protected 錾 }
    function MinutesToSurface( Minutes : Integer) : Integer; overload;
    function MinutesToSurface( Minutes, Row : Integer) : Integer; overload;
    function RowFromY( Y : Integer) : Integer;
    function ColFromX( X : Integer) : Integer;
    function DateFromCol( ACol : Integer) : TDate;
    function ColFromDate( datAt : TDate; cascaded : Boolean) : Integer;
    function PacketFromPos( X, Y : Integer) : TSchedulePacket;

    procedure ResetSelection( Col, keySchedule : Integer);
  public
    { Public 錾 }
    constructor Create(AOwner: TComponent); override;

    property ClientKey : Integer read FClientKey;
    property ScheduleFrom : TDate read FScheduleFrom;
    property ScheduleTo : TDate read FScheduleTo;
    property Obsolete : Boolean read FObsolete write FObsolete;

    function ScheduleAt() : TDate;

    procedure Reset( keyProfile : Integer; datFrom, datTo : TDate);
  end;

var
  ClientPrivateGraphFrame: TClientPrivateGraphFrame;

implementation

{$R *.dfm}

uses Piece_SchedulePalette, Chooser_Client, Set_Client, Set_Care, Set_Menage, Set_Migration, Set_Transfer;

{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

constructor TClientPrivateGraphFrame.Create(AOwner: TComponent);
begin
	inherited Create( AOwner);

	FClientKey := -1;
	FScheduleFrom := StartOfTheMonth( Today);
	FScheduleTo := EndOfTheMonth( FScheduleFrom);
	FObsolete := false;

	FCursorCol := -1;

	gridGraph.RowHeights[ 0] := 18;
	gridGraph.TopRow := 7;

	sc_keyInstance := -1; // see procedure Reset( keyProfile : Integer; datFrom, datTo : TDate);

	Selection.Open;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function TClientPrivateGraphFrame.ScheduleAt() : TDate;
begin
	if Selection.IsEmpty then
		result := ScheduleFrom
	else
		result := DateFromCol( Selection.FieldByName( 'keyCol').AsInteger);
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function TClientPrivateGraphFrame.MinutesToSurface( Minutes : Integer) : Integer;
begin
	result := Minutes * ( gridGraph.DefaultRowHeight + 1) div 60;
end;

function TClientPrivateGraphFrame.MinutesToSurface( Minutes, Row : Integer) : Integer;
begin
	result := MinutesToSurface( Minutes) - Row * ( gridGraph.DefaultRowHeight + 1);
end;

function TClientPrivateGraphFrame.RowFromY( Y : Integer) : Integer;
begin
	with gridGraph do result := Min( RowCount - 1, Max( 0, TopRow + ( Y - RowHeights[ 0]) div ( DefaultRowHeight + 1)));
end;

function TClientPrivateGraphFrame.ColFromX( X : Integer) : Integer;
begin
	with gridGraph do result := Min( ColCount - 1, Max( 0, LeftCol + X div ( DefaultColWidth + 1)));
end;

function TClientPrivateGraphFrame.DateFromCol( ACol : Integer) : TDate;
begin
	result := FCalendars[ Max( 0, Min( ACol, gridGraph.ColCount - 1))].calendar;
end;

function TClientPrivateGraphFrame.ColFromDate( datAt : TDate; cascaded : Boolean) : Integer;
var
	index : Integer;
begin
	result := -1;
	for index := Low( FCalendars) to High( FCalendars) do
	begin
		if ( FCalendars[ index].calendar = datAt) and ( FCalendars[ index].cascaded = cascaded) then
		begin
			result := index;
			exit;
		end;
	end;
end;

function TClientPrivateGraphFrame.PacketFromPos( X, Y : Integer) : TSchedulePacket;
var
	clickedX, clickedY, currSchedule : Integer;
begin
	result.instanceKey := -1;
	result.scheduleType := -1;
	result.progress := -1;
	result.presence := -1;
	result.minutesFrom := -1;
	result.minutesTo := -1;
	result.subjectAt := -1;
	result.serverReady := false;
	result.subjectKey := -1;
	result.nextIndex := -1;

	with gridGraph do
	begin
		if ( X mod ( DefaultColWidth + 1)) < 24 then
			clickedX := 0
		else if ( X mod ( DefaultColWidth + 1)) > 38 then
			clickedX := 2
		else
			clickedX := 1;
		clickedY := ( Y - RowHeights[ 0]) + 60 * TopRow;
	end;

	currSchedule := FCalendars[ ColFromX( X)].scheduleIndex;
	while currSchedule >= 0 do
	begin
		with FSchedules[ currSchedule] do
		begin
			case scheduleType of
			CARE_SCHEDULE, MENAGE_SCHEDULE :
				if ( clickedX = 1) and ( MinutesToSurface( minutesFrom) <= clickedY) and ( clickedY <= MinutesToSurface( minutesTo)) then
					result := FSchedules[ currSchedule];

			MIGRATION_SCHEDULE :
				case presence of
				BEFORE_PRESENCE :
					if ( clickedX = 2) and ( MinutesToSurface( minutesFrom) <= clickedY) and ( clickedY <= MinutesToSurface( minutesFrom) + 20) then
						result := FSchedules[ currSchedule];

				AFTER_PRESENCE :
					if ( clickedX = 0) and ( MinutesToSurface( minutesFrom) <= clickedY) and ( clickedY <= MinutesToSurface( minutesFrom) + 20) then
						result := FSchedules[ currSchedule];
				end;

			TRANSFER_SCHEDULE :
				case presence of
				BEFORE_PRESENCE :
					if ( clickedX = 1) and ( MinutesToSurface( minutesFrom) - 20 <= clickedY) and ( clickedY <= MinutesToSurface( minutesFrom)) then
						result := FSchedules[ currSchedule];

				AFTER_PRESENCE :
					if ( clickedX = 1) and ( MinutesToSurface( minutesFrom) <= clickedY) and ( clickedY <= MinutesToSurface( minutesFrom) + 20) then
						result := FSchedules[ currSchedule];

				FULL_PRESENCE :
					if ( clickedX = 1) and ( MinutesToSurface( minutesFrom) <= clickedY) and ( clickedY <= MinutesToSurface( minutesFrom) + 20) then
						result := FSchedules[ currSchedule];
				end;
			end;
		end;
		currSchedule := FSchedules[ currSchedule].nextIndex;
	end;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TClientPrivateGraphFrame.ResetSelection( Col, keySchedule : Integer);

	procedure InvalidateCol( Col : Integer);
	var
		rectRedraw : TRect;
	begin
		rectRedraw := gridGraph.CellRect( Col, 0);
		rectRedraw.Top := 0;
		rectRedraw.Bottom := gridGraph.Height;
		InvalidateRect( gridGraph.Handle, @rectRedraw, false);
	end;
var
	currSchedule : Integer;
	addressFrom, addressTo, supplier, summary : String;
begin
	Col := Max( Low( FCalendars), Min( Col, High( FCalendars)));

	if Col <> CursorCol then
	begin
		InvalidateCol( CursorCol);
		FCursorCol := Col;
		InvalidateCol( CursorCol);
	end;

	Selection.DisableControls;
	Selection.EmptyDataSet;
	currSchedule := FCalendars[ Col].scheduleIndex;
	while currSchedule >= 0 do
	begin
		with Schedule, FSchedules[ currSchedule] do
		begin
			RecNo := recordNo;

			summary := Format( '%.2d:%.2d ` %s', [ minutesFrom div 60, minutesFrom mod 60, Fields[ sc_strService].AsString]);

			if ( Fields[ sc_typAddressFrom].AsInteger = PROFILE_ADDRESS) and ( Fields[ sc_refResidentFrom].AsInteger = ClientKey) then
				addressFrom := 'p҂̎'
			else
				addressFrom := Fields[ sc_strAddressFrom].AsString;

			if ( Fields[ sc_typAddressTo].AsInteger = PROFILE_ADDRESS) and ( Fields[ sc_refResidentTo].AsInteger = ClientKey) then
				addressTo := 'p҂̎'
			else
				addressTo := Fields[ sc_strAddressTo].AsString;

			if not Fields[ sc_refServer].IsNull then
				supplier := Format( ' ~ %s', [ Fields[ sc_strServer].AsString])
			else
				supplier := '';

			case Fields[ sc_typSchedule].AsInteger of
			CARE_SCHEDULE, MENAGE_SCHEDULE :
				summary := Format( '%s %s%s', [ summary, addressFrom, supplier]);

			MIGRATION_SCHEDULE, TRANSFER_SCHEDULE :
				if presence = BEFORE_PRESENCE then
					summary := Format( '%s y%s  %sz%s', [ summary, addressTo, addressFrom, supplier])
				else
					summary := Format( '%s y%s  %sz%s', [ summary, addressFrom, addressTo, supplier]);
			end;

			case Fields[ sc_typSchedule].AsInteger of
			CARE_SCHEDULE, MENAGE_SCHEDULE :
				summary := Format( '%s ` %.2d:%.2d', [ summary, minutesTo div 60, minutesTo mod 60]);

			MIGRATION_SCHEDULE :
				if presence = BEFORE_PRESENCE then
					summary := Format( '%s ` %.2d:%.2d', [ summary, subjectAt div 60, subjectAt mod 60]);
			end;

			Selection.Append;
			Selection.FieldByName( 'keyCol').AsInteger := Col;
			Selection.FieldByName( 'keyInstance').AsInteger := instanceKey;
			Selection.FieldByName( 'typSchedule').AsInteger := scheduleType;
			Selection.FieldByName( 'intMinutesAt').AsInteger := minutesFrom;
			Selection.FieldByName( 'intMinutesFor').AsInteger := minutesTo - minutesFrom;
			Selection.FieldByName( 'strCaption').AsString := summary;
			Selection.Post;
			currSchedule := nextIndex;
		end;
	end;
	Selection.IndexFieldNames := 'intMinutesAt;intMinutesFor;strCaption';
	Selection.EnableControls;

	cboSelection.KeyValue := ZeroToNull( 0);
	cboSelection.KeyValue := keySchedule;
	btnSelectSchedule.Enabled := cboSelection.Text <> '';
	btnDeleteSchedule.Enabled := cboSelection.Text <> '';
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TClientPrivateGraphFrame.Reset( keyProfile : Integer; datFrom, datTo : TDate);
var
	defaultCol, defaultSchedule : Integer;

	ARow, ACol : Integer;
	currCalendar, emptyCalendar : Integer;
	currSchedule, nextSchedule : Integer;
	scheduleCascaded : Boolean;

	procedure Markup( Col, Row : Integer; ServerReady : Boolean);
	begin
		if gridGraph.Objects[ Col, Row] = gridGraph then exit;
		if ServerReady then
			gridGraph.Objects[ Col, Row] := Schedule
		else
			gridGraph.Objects[ Col, Row] := gridGraph;
	end;
begin
	defaultCol := Selection.FieldByName( 'keyCol').AsInteger;
	defaultSchedule := Selection.FieldByName( 'keyInstance').AsInteger;
	// cursor preservation

	FClientKey := keyProfile;
	FScheduleFrom := datFrom;
	FScheduleTo := datTo;
	// property update

	Profile.Close;
	Profile.ParamByName( 'keyInstance').AsInteger := ClientKey;
	Profile.ParamByName( 'datAvailableAt').AsDateTime := ScheduleTo;
	Profile.Open;

	if not Profile.IsEmpty then with Profile do
		txtProfileInfo.Text := Format( '%s / %s / %s / %s / %s/ %s', [
			Trim( FieldByName( 'strBranch').AsString),
			Trim( FieldByName( 'strRegion').AsString),
			Trim( FieldByName( 'strCode').AsString),
			Trim( FieldByName( 'strName').AsString),
			Trim( FieldByName( 'strNameRead').AsString),
			Trim( FieldByName( 'strPhoneNumber1').AsString),
			Trim( FieldByName( 'strAddress').AsString)
		])
	else
		txtProfileInfo.Text := 'o^敪 / Zn / o^ԍ / o^@ / ݁@ / dbԍP/ Z';
	// Profile update

	Schedule.Close;
	Schedule.Params.ParamByName( 'keyRecipient').AsInteger := ClientKey;
	Schedule.Params.ParamByName( 'datFrom').AsDate := ScheduleFrom;
	Schedule.Params.ParamByName( 'datTo').AsDate := ScheduleTo;
	Schedule.Open;
	// Schedule update

	if sc_keyInstance < 0 then
	begin
		sc_keyInstance := Schedule.FieldByName( 'keyInstance').Index;
		sc_typProgress := Schedule.FieldByName( 'typProgress').Index;
		sc_datSchedule := Schedule.FieldByName( 'datSchedule').Index;
		sc_typSchedule := Schedule.FieldByName( 'typSchedule').Index;
		sc_strService := Schedule.FieldByName( 'strService').Index;
		sc_intMinutesFrom := Schedule.FieldByName( 'intMinutesFrom').Index;
		sc_intMinutesTo  := Schedule.FieldByName( 'intMinutesTo').Index;
		sc_refSubject := Schedule.FieldByName( 'refSubject').Index;
		sc_refSubjectSubject := Schedule.FieldByName( 'refSubjectSubject').Index;
		sc_typSubject := Schedule.FieldByName( 'typSubject').Index;
		sc_intSubjectAt  := Schedule.FieldByName( 'intSubjectAt').Index;
		sc_intSubjectFor  := Schedule.FieldByName( 'intSubjectFor').Index;
		sc_refSubjectServer  := Schedule.FieldByName( 'refSubjectServer').Index;
		sc_typAddressFrom := Schedule.FieldByName( 'typAddressFrom').Index;
		sc_refAddressFrom := Schedule.FieldByName( 'refAddressFrom').Index;
		sc_strAddressFrom := Schedule.FieldByName( 'strAddressFrom').Index;
		sc_refResidentFrom := Schedule.FieldByName( 'refResidentFrom').Index;
		sc_typAddressTo := Schedule.FieldByName( 'typAddressTo').Index;
		sc_refAddressTo := Schedule.FieldByName( 'refAddressTo').Index;
		sc_strAddressTo := Schedule.FieldByName( 'strAddressTo').Index;
		sc_refResidentTo := Schedule.FieldByName( 'refResidentTo').Index;
		sc_refClient := Schedule.FieldByName( 'refClient').Index;
		sc_strClient := Schedule.FieldByName( 'strClient').Index;
		sc_refServer := Schedule.FieldByName( 'refServer').Index;
		sc_strServer := Schedule.FieldByName( 'strServer').Index;
		sc_strOpponent := Schedule.FieldByName( 'strOpponent').Index;
	end;
	// prepare index cache

	FCalendars := nil;
	SetLength( FCalendars, ( DaysBetween( ScheduleFrom, ScheduleTo) + 1) * 2);
	for ACol := Low( FCalendars) to High( FCalendars) do
	begin
		FCalendars[ ACol].calendar := IncDay( ScheduleFrom, ACol div 2);
		FCalendars[ ACol].scheduleIndex := -1;
		FCalendars[ ACol].overlapped := false;
		FCalendars[ ACol].cascaded := IntToBool( ACol mod 2);
	end;

	FSchedules := nil;
	nextSchedule := 0;
	SetLength( FSchedules, Schedule.RecordCount);

	Schedule.First;
	with Schedule do while not EOF do
	begin
		if ( not Fields[ sc_refSubject].IsNull) and (
			( not Fields[ sc_refSubjectSubject].IsNull) or
			( Fields[ sc_typSubject].AsInteger = Fields[ sc_typSchedule].AsInteger)
		) then
			scheduleCascaded := true
		else
			scheduleCascaded := false;

		currCalendar := DaysBetween( ScheduleFrom, Fields[ sc_datSchedule].AsDateTime) * 2 + BoolToInt( scheduleCascaded);
		if FCalendars[ currCalendar].scheduleIndex >= 0 then
		begin
			currSchedule := FCalendars[ currCalendar].scheduleIndex;
			while true do
			begin
				case Fields[ sc_typSchedule].AsInteger of CARE_SCHEDULE, MENAGE_SCHEDULE :
					if IntersectionOf(
						FSchedules[ currSchedule].minutesFrom, FSchedules[ currSchedule].minutesTo,
						Fields[ sc_intMinutesFrom].AsInteger, Fields[ sc_intMinutesTo].AsInteger
					) > 5 then
					begin
						FCalendars[ currCalendar].overlapped := true;
					end;
				else
					if ( FSchedules[ currSchedule].scheduleType = Fields[ sc_typSchedule].AsInteger)
					and ( FSchedules[ currSchedule].minutesFrom = Fields[ sc_intMinutesFrom].AsInteger)
					and ( FSchedules[ currSchedule].minutesTo = Fields[ sc_intMinutesTo].AsInteger)
					then
					begin
						FCalendars[ currCalendar].overlapped := true;
					end;
				end;
				if FSchedules[ currSchedule].nextIndex < 0 then break;
				currSchedule := FSchedules[ currSchedule].nextIndex;
			end;
			FSchedules[ currSchedule].nextIndex := nextSchedule;
		end
		else
		begin
			FCalendars[ currCalendar].scheduleIndex := nextSchedule;
		end;

		with FSchedules[ nextSchedule] do
		begin
			instanceKey := Fields[ sc_keyInstance].AsInteger;
			scheduleType := Fields[ sc_typSchedule].AsInteger;
			progress := Fields[ sc_typProgress].AsInteger;
			minutesFrom := Fields[ sc_intMinutesFrom].AsInteger;
			minutesTo := Fields[ sc_intMinutesTo].AsInteger;
			subjectAt := Fields[ sc_intSubjectAt].AsInteger;
			opponent := Fields[ sc_strOpponent].AsString;
			serverReady := not Fields[ sc_refServer].IsNull;
			subjectKey := Fields[ sc_refSubject].AsInteger;
			case scheduleType of
			MIGRATION_SCHEDULE :
				if minutesFrom < subjectAt + Fields[ sc_intSubjectFor].AsInteger div 2 then
					presence := BEFORE_PRESENCE
				else
					presence := AFTER_PRESENCE;

			TRANSFER_SCHEDULE :
				if ( subjectKey > 0) and ( Fields[ sc_typSubject].AsInteger <> TRANSFER_SCHEDULE) then
				begin
					if minutesFrom < subjectAt + Fields[ sc_intSubjectFor].AsInteger div 2 then
						presence := BEFORE_PRESENCE
					else
						presence := AFTER_PRESENCE;
				end
				else
					presence := FULL_PRESENCE;
			else
				presence := FULL_PRESENCE;
			end;
			recordNo := RecNo;
			nextIndex := -1;
		end;
		Inc( nextSchedule);

		Next;
	end;

	emptyCalendar := 0;
	for currCalendar := Low( FCalendars) to High( FCalendars) do
	begin
		if ( not FCalendars[ currCalendar].cascaded) or ( FCalendars[ currCalendar].scheduleIndex >= 0) then
		begin
			while emptyCalendar < currCalendar do
			begin
				if FCalendars[ emptyCalendar].cascaded and ( FCalendars[ emptyCalendar].scheduleIndex < 0) then
				begin
					FCalendars[ emptyCalendar] := FCalendars[ currCalendar];
					FCalendars[ currCalendar].cascaded := true;
					FCalendars[ currCalendar].scheduleIndex := -1;
					break;
				end;
				Inc( emptyCalendar);
			end;
		end;
	end;
	SetLength( FCalendars, Max( 1, emptyCalendar + 1));
	// Calendars and Packets setup

	gridGraph.ColCount := Length( FCalendars);
	for ACol := 0 to gridGraph.ColCount - 1 do
		for ARow := 0 to gridGraph.RowCount - 1 do
			gridGraph.Objects[ ACol, ARow] := nil;

	if not Schedule.IsEmpty then
	begin
		for ACol := 0 to gridGraph.ColCount - 1 do
		begin
			currSchedule := FCalendars[ ACol].scheduleIndex;
			while currSchedule >= 0 do
                        begin
				with FSchedules[ currSchedule] do case scheduleType of
				CARE_SCHEDULE, MENAGE_SCHEDULE :
					for ARow := minutesFrom div 60 to ( minutesTo - 1) div 60 do
					begin
						Markup( ACol, ARow, serverReady);
					end;

				MIGRATION_SCHEDULE :
					if presence = BEFORE_PRESENCE then
					begin
						Markup( ACol, Max( 0, ( minutesFrom - 10) div 60), serverReady);
						Markup( ACol, Max( 0, ( minutesFrom + 22) div 60), serverReady);
					end
					else
					begin
						Markup( ACol, Max( 0, minutesFrom div 60), serverReady);
						Markup( ACol, Max( 0, ( minutesFrom + 32) div 60), serverReady);
					end;

				TRANSFER_SCHEDULE :
					if presence = BEFORE_PRESENCE then
					begin
						Markup( ACol, Max( 0, ( minutesFrom - 32) div 60), serverReady);
						Markup( ACol, Max( 0, minutesFrom div 60), serverReady);
					end
					else
					begin
						Markup( ACol, Max( 0, ( minutesFrom - 10) div 60), serverReady);
						Markup( ACol, Max( 0, ( minutesFrom + 22) div 60), serverReady);
					end;
				end;
				currSchedule := FSchedules[ currSchedule].nextIndex;
	                end;
		end;
	end;
	// graph drawing information update

	ResetSelection( defaultCol, defaultSchedule);
	// Selection update

	Obsolete := false;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TClientPrivateGraphFrame.btnSelectProfile_Click(Sender: TObject);
begin
	TClientSet.Prepare.Select( ClientKey);
end;

procedure TClientPrivateGraphFrame.btnChooseProfile_Click(Sender: TObject);
var
	keyClient : Integer;
begin
	if ChooseClient( DataStore.MainTransaction, ScheduleTo, keyClient) then
	begin
		Reset( keyClient, ScheduleFrom, ScheduleTo);
	end;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TClientPrivateGraphFrame.cboSelection_Click(Sender: TObject);
begin
	btnSelectSchedule.Enabled := cboSelection.Text <> '';
	btnDeleteSchedule.Enabled := cboSelection.Text <> '';
end;

procedure TClientPrivateGraphFrame.btnSelectSchedule_Click(Sender: TObject);
begin
	case Selection.FieldByName( 'typSchedule').AsInteger of
	     CARE_SCHEDULE : TCareSet.Prepare.Select( Selection.FieldByName( 'keyInstance').AsInteger);
	   MENAGE_SCHEDULE : TMenageSet.Prepare.Select( Selection.FieldByName( 'keyInstance').AsInteger);
	MIGRATION_SCHEDULE : TMigrationSet.Prepare.Select( Selection.FieldByName( 'keyInstance').AsInteger);
	 TRANSFER_SCHEDULE : TTransferSet.Prepare.Select( Selection.FieldByName( 'keyInstance').AsInteger);
	end;
end;

procedure TClientPrivateGraphFrame.btnDeleteSchedule_Click(Sender: TObject);
begin
	case Selection.FieldByName( 'typSchedule').AsInteger of
	     CARE_SCHEDULE : TCareSet.Prepare.Delete( Selection.FieldByName( 'keyInstance').AsInteger);
	   MENAGE_SCHEDULE : TMenageSet.Prepare.Delete( Selection.FieldByName( 'keyInstance').AsInteger);
	MIGRATION_SCHEDULE : TMigrationSet.Prepare.Delete( Selection.FieldByName( 'keyInstance').AsInteger);
	 TRANSFER_SCHEDULE : TTransferSet.Prepare.Delete( Selection.FieldByName( 'keyInstance').AsInteger);
	end;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TClientPrivateGraphFrame.gridGraph_DrawCell(Sender: TObject; ACol, ARow: Integer; Rect: TRect; State: TGridDrawState);
var
	defBrushColor, defPenColor : TColor;
	defBrushStyle : TBrushStyle;
	defPenStyle : TPenStyle;
	defClipRect : TRect;
	topLimit, bottomLimit : Integer;
	labelX, labelY : Integer;
	currSchedule : Integer;
begin
	with gridGraph.Canvas do
	begin
		defBrushColor := Brush.Color;
		defBrushStyle := Brush.Style;
		defPenColor := Pen.Color;
		defPenStyle := Pen.Style;
		defClipRect := ClipRect;
		IntersectClipRect( Handle, Rect.Left, Rect.Top, Rect.Right, Rect.Bottom);
	end;
	// backup default status

	with gridGraph.Canvas do if ARow = 0 then
	begin
		Font.Color := clBtnText;
		Font.Size := 8;
		if ACol = CursorCol then
		begin
			case DayOfTheWeek( DateFromCol( ACol)) of
			DaySaturday : Brush.Color := TColor( $CCAAAA);
			DaySunday : Brush.Color := TColor( $AAAACC);
			else Brush.Color := TColor( $AAAAAA);
			end;
		end
		else
		begin
			case DayOfTheWeek( DateFromCol( ACol)) of
			DaySaturday : Brush.Color := TColor( $DCC0C0);
			DaySunday : Brush.Color := TColor( $C0C0DC);
			else Brush.Color := IfThen( not FCalendars[ ACol].cascaded, clSilver, gridGraph.Color);
			end;
		end;
		if FCalendars[ ACol].overlapped then Brush.Color := TColor( $9999EE);
		Brush.Style := bsSolid;
		if not FCalendars[ ACol].cascaded then
		begin
			TextRect( Rect, Rect.Left + 2, Rect.Top + 6,
				FormatDateTime( 'M"/"dd"("ddd")"', DateFromCol( ACol))
			);
		end
		else
		begin
			FillRect( Rect);
			Pen.Color := clLtGray;
			Pen.Style := psDot;
			MoveTo( Rect.Left + 31, Rect.Top);
			LineTo( Rect.Left + 31, Rect.Bottom);
		end;
		// draw header column
	end
	else
	begin
		if gridGraph.Objects[ ACol, ARow] = gridGraph then
			Brush.Color := clDefaultExcluded
		else if ACol = CursorCol then
			Brush.Color := TColor( $DDDDDD)
		else
			Brush.Color := gridGraph.Color;
		Brush.Style := bsSolid;
		FillRect( Rect);

		Pen.Color := clLtGray;
		Pen.Style := psDot;
		MoveTo( Rect.Left + 31, Rect.Top);
		LineTo( Rect.Left + 31, Rect.Bottom);
		// draw schedule axis

		if ( ACol mod 4 = 0) or ( ARow mod 4 = 0) then
		begin
			Font.Color := clWhite;
			Font.Color := clGray;
			Font.Size := 8;
			TextOut( Rect.Left + 1, Rect.Top + 1,
				IfThen( ( ARow div 12) mod 2 = 0,
					Format( 'AM%d:00', [ ARow mod 12]),
					Format( 'PM%d:00', [ ARow mod 12])
				)
			);
		end;
		// draw time label

		Font.Color := clBlack;
		Font.Size := 8;
		if gridGraph.Objects[ ACol, ARow] <> nil then
		begin
			currSchedule := FCalendars[ ACol].scheduleIndex;
			while currSchedule >= 0 do
                        begin
				with FSchedules[ currSchedule] do case scheduleType of
				CARE_SCHEDULE, MENAGE_SCHEDULE :
					begin
						Brush.Color := IfThen( scheduleType = CARE_SCHEDULE, clCare, clMenage);
						Brush.Style := bsSolid;
						if progress = UNCLEAR_PROGRESS then Brush.Style := bsDiagCross;

						Pen.Color := Brush.Color;
						Pen.Style := psSolid;

						topLimit := MinutesToSurface( minutesFrom, ARow);
						bottomLimit := MinutesToSurface( minutesTo, ARow);
						Rectangle( Rect.Left + 26, Rect.Top + topLimit, Rect.Left + 37, Rect.Top + bottomLimit);

						labelX := 64 - TextWidth( FSchedules[ currSchedule].opponent);
						labelY := ( topLimit + bottomLimit - TextHeight( FSchedules[ currSchedule].opponent)) div 2;
					end;

				MIGRATION_SCHEDULE :
					if presence = BEFORE_PRESENCE then
					begin
						topLimit := MinutesToSurface( minutesFrom, ARow);
						iglIcon.Draw( gridGraph.Canvas, Rect.Left + 42, Rect.Top + topLimit, IfThen( progress <> UNCLEAR_PROGRESS, 4, 5));
						labelX := 64 - TextWidth( FSchedules[ currSchedule].opponent);
						labelY := topLimit - TextHeight( FSchedules[ currSchedule].opponent);
					end
					else
					begin
						topLimit := MinutesToSurface( minutesFrom, ARow);
						bottomLimit := topLimit + 22;
						iglIcon.Draw( gridGraph.Canvas, Rect.Left + 2, Rect.Top + topLimit, IfThen( progress <> UNCLEAR_PROGRESS, 4, 5));
						labelX := 0;
						labelY := bottomLimit;
					end;

				TRANSFER_SCHEDULE :
					if presence = BEFORE_PRESENCE then
					begin
						topLimit := MinutesToSurface( minutesFrom, ARow) - 22;
						iglIcon.Draw( gridGraph.Canvas, Rect.Left + 21, Rect.Top + topLimit, IfThen( progress <> UNCLEAR_PROGRESS, 2, 3));
						labelX := 0;
						labelY := topLimit - TextHeight( FSchedules[ currSchedule].opponent) div 2;
					end
					else if presence = AFTER_PRESENCE then
					begin
						topLimit := MinutesToSurface( minutesFrom, ARow);
						iglIcon.Draw( gridGraph.Canvas, Rect.Left + 21, Rect.Top + topLimit, IfThen( progress <> UNCLEAR_PROGRESS, 2, 3));
						labelX := 64 - TextWidth( FSchedules[ currSchedule].opponent);
						labelY := topLimit - TextHeight( FSchedules[ currSchedule].opponent) div 2;
					end
					else
					begin
						topLimit := MinutesToSurface( minutesFrom, ARow);
						iglIcon.Draw( gridGraph.Canvas, Rect.Left + 21, Rect.Top + topLimit, IfThen( progress <> UNCLEAR_PROGRESS, 0, 1));
						labelX := 64 - TextWidth( FSchedules[ currSchedule].opponent);
						labelY := topLimit - TextHeight( FSchedules[ currSchedule].opponent) div 2;
					end;
				end;

				SetBkMode( Handle, TRANSPARENT);
				TextOut( Rect.Left + labelX, Rect.Top + labelY, FSchedules[ currSchedule].opponent);

				currSchedule := FSchedules[ currSchedule].nextIndex;
                        end;
		end;
		// draw schedule

		if ( ARow mod 24) = HourOf( Now) then
		begin
			Pen.Color := clRed;
			Pen.Style := psSolid;
			MoveTo( Rect.Left, Rect.Top + MinuteOf( Now));
			LineTo( Rect.Right, Rect.Top + MinuteOf( Now));
		end;
		// draw current time indicator
	end;

	with gridGraph.Canvas do
	begin
		Brush.Color := defBrushColor;
		Brush.Style := defBrushStyle;
		Pen.Color := defPenColor;
		Pen.Style := defPenStyle;
		SelectClipRgn( Handle, 0);
		IntersectClipRect( Handle, defClipRect.Left, defClipRect.Top, defClipRect.Right, defClipRect.Bottom);
	end;
	// restore default status
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TClientPrivateGraphFrame.gridGraph_SelectCell(Sender: TObject; ACol, ARow: Integer; var CanSelect: Boolean);
begin
	CanSelect := false;
end;

procedure TClientPrivateGraphFrame.gridGraph_MouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
	if Button = mbLeft then ResetSelection( ColFromX( X), PacketFromPos( X, Y).instanceKey);
end;

procedure TClientPrivateGraphFrame.gridGraph_DblClick(Sender: TObject);
var
	clickedPos : TPoint;
	scheduleKey : Integer;
begin
	clickedPos := gridGraph.ScreenToClient( Mouse.CursorPos);

	with gridGraph do if ( LeftCol + clickedPos.X div ( DefaultColWidth + 1)) >= ColCount then exit;

	scheduleKey := PacketFromPos( clickedPos.X, clickedPos.Y).instanceKey;
	if scheduleKey > 0 then
	begin
		cboSelection.KeyValue := ZeroToNull( 0);
		cboSelection.KeyValue := scheduleKey;
		btnSelectSchedule.Enabled := cboSelection.Text <> '';
		btnDeleteSchedule.Enabled := cboSelection.Text <> '';
		if btnSelectSchedule.Enabled then btnSelectSchedule.Click;
	end;
end;

procedure TClientPrivateGraphFrame.gridGraph_KeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
begin
	case Key of
	 VK_LEFT : gridGraph.LeftCol := Max( 0, gridGraph.LeftCol - 1);
	VK_RIGHT : gridGraph.LeftCol := Min( gridGraph.ColCount - gridGraph.VisibleColCount, gridGraph.LeftCol + 1);
	   VK_UP : gridGraph.TopRow := Max( 1, gridGraph.TopRow - 1);
	 VK_DOWN : gridGraph.TopRow := Min( gridGraph.RowCount - gridGraph.VisibleRowCount, gridGraph.TopRow + 1);
	end;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TClientPrivateGraphFrame.gridGraph_DragOver(Sender, Source: TObject; X, Y: Integer; State: TDragState; var Accept: Boolean);
begin
	gridGraph_MouseDown( Sender, mbLeft, [], X, Y);

	Accept := false;

	if Profile.IsEmpty then exit;
	with gridGraph do if ( LeftCol + X div ( DefaultColWidth + 1)) >= ColCount then exit;

	with PacketFromPos( X, Y) do case DragSourceToSchedulePalette( Source).ScheduleType of
	      PRIMARY_CARE : Accept := true;
	    SECONDARY_CARE : Accept := ( instanceKey > 0) and ( subjectKey <= 0) and ( scheduleType = CARE_SCHEDULE);

	    PRIMARY_MENAGE : Accept := true;
	  SECONDARY_MENAGE : Accept := ( instanceKey > 0) and ( subjectKey <= 0) and ( scheduleType = MENAGE_SCHEDULE);

	  HELPER_MIGRATION : Accept := ( instanceKey > 0) and ( scheduleType in [ CARE_SCHEDULE, MENAGE_SCHEDULE]);

	  PRIMARY_TRANSFER : Accept := true;
	SECONDARY_TRANSFER : Accept := ( instanceKey > 0) and ( subjectKey <= 0) and ( scheduleType = TRANSFER_SCHEDULE);
	   ESCORT_TRANSFER : Accept := ( instanceKey > 0) and ( subjectKey <= 0) and ( scheduleType in [ CARE_SCHEDULE, MENAGE_SCHEDULE]);
	end;
end;

procedure TClientPrivateGraphFrame.gridGraph_DragDrop(Sender, Source: TObject; X, Y: Integer);
var
        DroppedDate : TDate;
	DroppedAt : Integer;
begin
	DroppedDate := DateFromCol( ColFromX( X));
	DroppedAt := RowFromY( Y) * 60 + ( ( ( Y - gridGraph.RowHeights[ 0]) mod 60) div 15) * 15;
	with PacketFromPos( X, Y) do case DragSourceToSchedulePalette( Source).ScheduleType of
	      PRIMARY_CARE : TCareSet.Prepare( DroppedDate, DroppedAt, ClientKey).Append;
	    SECONDARY_CARE : TCareSet.Prepare( instanceKey).Append;

	    PRIMARY_MENAGE : TMenageSet.Prepare( DroppedDate, DroppedAt, ClientKey).Append;
	  SECONDARY_MENAGE : TMenageSet.Prepare( instanceKey).Append;

	  HELPER_MIGRATION : TMigrationSet.Prepare( instanceKey, DroppedAt).Append;

	  PRIMARY_TRANSFER : TTransferSet.Prepare( DroppedDate, DroppedAt, ClientKey).Append;
	SECONDARY_TRANSFER : TTransferSet.Prepare( instanceKey).Append;
	   ESCORT_TRANSFER : TTransferSet.Prepare( instanceKey, DroppedAt).Append;
	end;
end;

end.
