unit MainDatastore;

interface

uses
  SysUtils, Classes, IBDatabase, DB, IBSQL, IBCustomDataSet, IBQuery,
  ExtCtrls, Dialogs, Forms, Provider, DBLocalI, IB, IBErrorCodes,
  IBServices, Windows, StrUtils;

procedure ExecSQL( Query : TIBSQL; request : String; parameters : array of Variant);
procedure UpdateSQL( Query : TIBSQL; nameOfTable, selector : String; fields : array of String; values : array of Variant);
procedure InsertSQL( Query : TIBSQL; nameOfTable : String; fields : array of String; values : array of Variant);
procedure DeleteSQL( Query : TIBSQL; nameOfTable, selector : String; values : array of Variant);

function VerifyRecords( Query : TIBSQL; nameOfTable : String; whereSQL : String) : Boolean;
function VerifyRecord( Query : TIBSQL; nameOfTable : String; keyInstance : Integer) : Boolean;

function CancelRecords( Query : TIBSQL; nameOfTable : String; whereSQL : String) : Boolean;
function CancelRecord( Query : TIBSQL; nameOfTable : String; keyInstance : Integer) : Boolean;

function RestoreRecords( Query : TIBSQL; nameOfTable : String; whereSQL : String) : Boolean;
function RestoreRecord( Query : TIBSQL; nameOfTable : String; keyInstance : Integer) : Boolean;

function ReserveRevision( Query : TIBSQL; ReserveCount : Integer) : Integer;

type
  TDataStore = class(TDataModule)
    MainDatabase: TIBDatabase;
    MainTransaction: TIBTransaction;
    MainQuery: TIBSQL;
    BackupService: TIBBackupService;
    RestoreService: TIBRestoreService;
    IBConfigService1: TIBConfigService;
    procedure Module_Create(Sender: TObject);
  private
    { Private 錾 }
  public
    { Public 錾 }
    procedure Backup( DestFile : String);
    procedure Restore( SrcFile : String);

    procedure ExecSQL( request : String; parameters : array of Variant);
    procedure UpdateSQL( nameOfTable, selector : String; fields : array of String; values : array of Variant);
    procedure InsertSQL( nameOfTable : String; fields : array of String; values : array of Variant);
    procedure DeleteSQL( nameOfTable, selector : String; values : array of Variant);

    function VerifyRecords( nameOfTable : String; whereSQL : String) : Boolean;
    function VerifyRecord( nameOfTable : String; keyInstance : Integer) : Boolean;

    function CancelRecords( nameOfTable : String; whereSQL : String) : Boolean;
    function CancelRecord( nameOfTable : String; keyInstance : Integer) : Boolean;

    function RestoreRecords( nameOfTable : String; whereSQL : String) : Boolean;
    function RestoreRecord( nameOfTable : String; keyInstance : Integer) : Boolean;

    function ReserveRevision( ReserveCount : Integer) : Integer;

    procedure Commit();
    procedure Rollback();
  end;

var
  DataStore : TDataStore;

implementation

{$R *.dfm}

{*******************************************************************************
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =
 *******************************************************************************}

{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure ExecSQL( Query : TIBSQL; request : String; parameters : array of Variant); overload;
var
	index : Integer;
	paramName : String;
begin
	with Query do
	begin
		SQL.Text := request;

		for index := 0 to high( parameters) do
		begin
			paramName := 'param' + IntToStr( index + 1);
			ParamByName( paramName).AsVariant := parameters[ index];
		end;

		ExecQuery;
		Close;
	end;
end;


procedure UpdateSQL( Query : TIBSQL; nameOfTable, selector : String; fields : array of String; values : array of Variant);
var
	index : Integer;
	request : String;
begin
	Assert( high( fields) = high( values), 'EUpdateSQL : ' + nameOfTable);
	// parameter check

	request := 'UPDATE ' + nameOfTable + ' SET ' + fields[ 0] + ' = :param1';
	for index := 1 to high( fields) do
	begin
		request := request + ', ' + fields[ index] + ' = :param' + IntToStr( index + 1);
	end;
	// set up request string

	if selector <> '' then
		ExecSQL( Query, request + ' WHERE ' + selector, values)
	else
		ExecSQL( Query, request, values);
end;


procedure InsertSQL( Query : TIBSQL; nameOfTable : String; fields : array of String; values : array of Variant);
var
	index : Integer;
	request : String;
begin
	Assert( high( fields) = high( values), 'EInsertSQL : ' + nameOfTable);
	// parameter check

	request := 'INSERT INTO ' + nameOfTable + ' (' + fields[ 0];
	for index := 1 to high( fields) do
	begin
		request := request + ', ' + fields[ index];
	end;

	request := request + ') VALUES ( :param1';
	for index := 1 to high( fields) do
	begin
		request := request + ', :param' + IntToStr( index + 1);
	end;
	// set up request string

	ExecSQL( Query, request + ')', values);
end;


procedure DeleteSQL( Query : TIBSQL; nameOfTable, selector : String; values : array of Variant);
begin
	if selector <> '' then
		ExecSQL( Query, 'DELETE FROM ' + nameOfTable + ' WHERE ' + selector, values)
	else
		ExecSQL( Query, 'DELETE FROM ' + nameOfTable, values);
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function VerifyRecords( Query : TIBSQL; nameOfTable : String; whereSQL : String) : Boolean;
begin
	result := false;

	try
		with Query do
		begin
			SQL.Clear;
			SQL.Add( 'UPDATE ' + nameOfTable);
			SQL.Add( '   SET typStatus = -1');
			SQL.Add( ' WHERE ' + whereSQL);
			ExecQuery;
			Close;
		end;
		result := true;

	except
		on err : EIBInterBaseError do
		begin
			case err.IBErrorCode of
			isc_lock_conflict : Query.Close;
			end;
		end;
	end;
end;

function VerifyRecord( Query : TIBSQL; nameOfTable : String; keyInstance : Integer) : Boolean;
begin
	result := VerifyRecords( Query, nameOfTable, Format( 'keyInstance = %d', [ keyInstance]));
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function CancelRecords( Query : TIBSQL; nameOfTable : String; whereSQL : String) : Boolean;
begin
	result := false;

	try
		with Query do
		begin
			SQL.Clear;
			SQL.Add( 'UPDATE ' + nameOfTable);
			SQL.Add( '   SET typStatus = 1'); { DATA_DISABLED }
			SQL.Add( ' WHERE typStatus = 0 and (' + whereSQL + ')'); { DATA_ENABLED }
			ExecQuery;
			Close;
		end;
		result := true;

	except
		on err : EIBInterBaseError do
		begin
			case err.IBErrorCode of
			isc_lock_conflict : Query.Close;
			end;
		end;
	end;
end;

function CancelRecord( Query : TIBSQL; nameOfTable : String; keyInstance : Integer) : Boolean;
begin
	result := CancelRecords( Query, nameOfTable, Format( 'keyInstance = %d', [ keyInstance]));
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function RestoreRecords( Query : TIBSQL; nameOfTable : String; whereSQL : String) : Boolean;
begin
	result := false;

	try
		with Query do
		begin
			SQL.Clear;
			SQL.Add( 'UPDATE ' + nameOfTable);
			SQL.Add( '   SET typStatus = 0'); { DATA_ENABLED }
			SQL.Add( ' WHERE ' + whereSQL);
			ExecQuery;
			Close;
		end;
		result := true;

	except
		on err : EIBInterBaseError do
		begin
			case err.IBErrorCode of
			isc_lock_conflict : Query.Close;
			end;
		end;
	end;
end;

function RestoreRecord( Query : TIBSQL; nameOfTable : String; keyInstance : Integer) : Boolean;
begin
	result := RestoreRecords( Query, nameOfTable, Format( 'keyInstance = %d', [ keyInstance]));
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function ReserveRevision( Query : TIBSQL; ReserveCount : Integer) : Integer;
begin
	ExecSQL( Query, Format( 'SELECT Revision FROM ReservedRevision( %d)', [ ReserveCount]), []);

	result := Query.Fields[ 0].AsInteger + 1;
end;


{*******************************************************************************
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =
 *******************************************************************************}

{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TDataStore.Backup( DestFile : String);
begin
	if FileExists( MainDatabase.DatabaseName) then
	begin
		BackupService.Protocol := Local;
		BackupService.ServerName := '';
	end
	else
	begin
		BackupService.Protocol := TCP;
		BackupService.ServerName := LeftStr( MainDatabase.DatabaseName, Pos( ':', MainDatabase.DatabaseName) - 1);
	end;

	MainDatabase.Connected := false;
	BackupService.Active := true;
	try
		BackupService.DatabaseName := MainDatabase.DatabaseName;
		BackupService.BackupFile.Clear;
		BackupService.BackupFile.Add( DestFile);
		BackupService.ServiceStart();
		while not BackupService.Eof do
		begin
			BackupService.GetNextLine;
		end;
	finally
		BackupService.Active := false;
		MainDatabase.Connected := true;
	end;
end;

procedure TDataStore.Restore( SrcFile : String);
begin
	if FileExists( MainDatabase.DatabaseName) then
	begin
		RestoreService.Protocol := Local;
		RestoreService.ServerName := '';
	end
	else
	begin
		RestoreService.Protocol := TCP;
		RestoreService.ServerName := LeftStr( MainDatabase.DatabaseName, Pos( ':', MainDatabase.DatabaseName) - 1);
	end;

	MainDatabase.Connected := false;
	RestoreService.Active := true;
	try
		RestoreService.DatabaseName.Clear;
		RestoreService.DatabaseName.Add( MainDatabase.DatabaseName);
		RestoreService.BackupFile.Clear;
		RestoreService.BackupFile.Add( SrcFile);
		RestoreService.ServiceStart();
		while not RestoreService.Eof do
		begin
			RestoreService.GetNextLine;
		end;
	finally
		RestoreService.Active := false;
		MainDatabase.Connected := true;
	end;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TDataStore.ExecSQL( request : String; parameters : array of Variant);
begin
	MainDatastore.ExecSQL( MainQuery, request, parameters);
end;


procedure TDataStore.UpdateSQL( nameOfTable, selector : String; fields : array of String; values : array of Variant);
begin
	MainDatastore.UpdateSQL( MainQuery, nameOfTable, selector, fields, values);
end;


procedure TDataStore.InsertSQL( nameOfTable : String; fields : array of String; values : array of Variant);
begin
	MainDatastore.InsertSQL( MainQuery, nameOfTable, fields, values);
end;


procedure TDataStore.DeleteSQL( nameOfTable, selector : String; values : array of Variant);
begin
	MainDatastore.DeleteSQL( MainQuery, nameOfTable, selector, values);
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function TDataStore.VerifyRecords( nameOfTable : String; whereSQL : String) : Boolean;
begin
	result := MainDatastore.VerifyRecords( MainQuery, nameOfTable, whereSQL);
end;

function TDataStore.VerifyRecord( nameOfTable : String; keyInstance : Integer) : Boolean;
begin
	result := MainDatastore.VerifyRecord( MainQuery, nameOfTable, keyInstance);
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function TDataStore.CancelRecords( nameOfTable : String; whereSQL : String) : Boolean;
begin
	result := MainDatastore.CancelRecords( MainQuery, nameOfTable, whereSQL);
end;

function TDataStore.CancelRecord( nameOfTable : String; keyInstance : Integer) : Boolean;
begin
	result := MainDatastore.CancelRecord( MainQuery, nameOfTable, keyInstance);
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function TDataStore.RestoreRecords( nameOfTable : String; whereSQL : String) : Boolean;
begin
	result := MainDatastore.RestoreRecords( MainQuery, nameOfTable, whereSQL);
end;

function TDataStore.RestoreRecord( nameOfTable : String; keyInstance : Integer) : Boolean;
begin
	result := MainDatastore.RestoreRecord( MainQuery, nameOfTable, keyInstance);
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function TDataStore.ReserveRevision( ReserveCount : Integer) : Integer;
begin
	result := MainDatastore.ReserveRevision( MainQuery, ReserveCount);
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TDataStore.Commit();
begin
	MainTransaction.CommitRetaining;
end;

procedure TDataStore.Rollback();
begin
	MainTransaction.RollbackRetaining;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TDataStore.Module_Create(Sender: TObject);
begin
	if ParamStr( 1) <> '' then
		MainDatabase.DatabaseName := ParamStr( 1)
	else
		MainDatabase.DatabaseName := ExtractFilePath( Application.ExeName) + 'Workingset.gdb';
	MainDatabase.Open;
	MainTransaction.StartTransaction;
end;

end.
