unit Misc_Utilities;

interface

uses Windows, Classes, SysUtils, StrUtils, DateUtils, Variants, Math,
     Forms, Graphics, Controls, StdCtrls, Buttons,
     DB, DBClient, DBLocal, DBLocalI, DBCtrls, DBGrids, IBCustomDataSet,
     Misc_Constants, MainDataStore;

function TemporaryKey() : Integer;
function TemporaryFileName() : String;

function CatalogueType( radShowAvailable, radShowUnavailable, radShowDisabled : TRadioButton) : Integer;
function CatalogueFilter( typCatalogue : Integer) : String; overload;
function CatalogueFilter( radShowAvailable, radShowUnavailable, radShowDisabled : TRadioButton) : String; overload;

function ReadSelectedKey( DataSet : TDataSet; nameOfKey : String) : Integer;
function ReadInstanceKey( DataSet : TDataSet) : Integer;

type TDataSetPosition = record
	keyInstance : Integer;
	RecNo : Integer;
	IndexFieldNames : String;
end;

procedure DisableComboSet( DBCombo : TDBLookupComboBox; var Position : TDataSetPosition);
procedure EnableComboSet( DBCombo : TDBLookupComboBox; Position : TDataSetPosition);

procedure DisableGridSet( DBGrid : TDBGrid; var Position : TDataSetPosition);
procedure EnableGridSet( DBGrid : TDBGrid; Position : TDataSetPosition);

procedure SortByDefault( grid : TDBGrid);

procedure SelectComboWithoutFail( combo : TDBLookUpComboBox; key : Integer);

procedure OptionsToCheckBoxes( options : Integer; checkBoxes : array of TCheckBox);
function CheckBoxesToOptions( checkBoxes : array of TCheckBox) : Integer;

procedure OptionsToSpeedButtons( options : Integer; SpeedButtons : array of TSpeedButton);
function SpeedButtonsToOptions( SpeedButtons : array of TSpeedButton) : Integer;

function WithInRange( value, floor, ceil : String) : Boolean;
procedure ErrorCheck_Empty( caption, text : String; ErrorMessages : TStrings);
procedure ErrorCheck_Reading( caption, text : String; ErrorMessages : TStrings);
procedure ErrorCheck_Digits( caption, text : String; ErrorMessages : TStrings);
procedure ErrorCheck_Code( caption, text : String; ErrorMessages : TStrings);

function ZeroToNull( const AValue : Variant) : Variant;

function IntToBool( val : Integer) : Boolean;
function BoolToInt( val : Boolean) : Integer;
function VarToInt( val : Variant) : Integer;

function ToZipCode( text : String) : Integer;
function ToZipText( zip : Integer) : String;

function ToDigits( text : String) : String;

function ToAge( atDate, birthday : TDate) : Integer;
function ToAgeText( atDate, birthday : TDate) : String;

function MinutesToText( minutes : Integer) : String;
function PeriodsToText( minutesFrom, minutesTo : Integer) : String;
function RangeToText( minutesAt, minutesFor : Integer) : String;

function IntersectionOf( F1, T1, F2, T2 : Integer) : Integer;

function CurrToPoint( Curr : Currency) : Currency;
function CurrToMoney( Curr : Currency) : Currency;

function AmountOf( typCategory : Integer; curAmountPerCount, curAmountPerHour : Currency; intMinutes : Integer) : Currency;
function AmountToText( typCategory : Integer; curAmountPerCount, curAmountPerHour : Currency; intMinutes : Integer) : String;

implementation

{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

var TemporaryKeyCount : Integer = -1;

function TemporaryKey() : Integer;
begin
	result := TemporaryKeyCount;
	Dec( TemporaryKeyCount);
end;

var TemporaryFileCount : Cardinal = 1;

function TemporaryFileName() : String;
begin
	result := Format( '%stasukeai_%d.tmp', [ ExtractFilePath( Application.ExeName), GetTickCount() + TemporaryFileCount]);
	Inc( TemporaryFileCount);
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function CatalogueType( radShowAvailable, radShowUnavailable, radShowDisabled : TRadioButton) : Integer;
begin
	result := AVAILABLE_ITEM;
	if radShowAvailable.Checked then result := AVAILABLE_ITEM;
	if radShowUnavailable.Checked then result := UNAVAILABLE_ITEM;
	if radShowDisabled.Checked then result := DISABLED_ITEM;
end;

function CatalogueFilter( typCatalogue : Integer) : String;
begin
	case typCatalogue of
	AVAILABLE_ITEM :
		result := Format(
			'typStatus=%d and datAvailableFrom <= ''%s'' and ''%s'' <= datAvailableTo',
			[ DATA_ENABLED, DateToStr( Today), DateToStr( Today)]
		);

	UNAVAILABLE_ITEM :
		result := Format(
			'typStatus=%d and not ( datAvailableFrom <= ''%s'' and ''%s'' <= datAvailableTo)',
			[ DATA_ENABLED, DateToStr( Today), DateToStr( Today)]
		);

	DISABLED_ITEM :
		result := Format( 'typStatus=%d', [ DATA_DISABLED]);

	else
		result := '';
	end;
end;

function CatalogueFilter( radShowAvailable, radShowUnavailable, radShowDisabled : TRadioButton) : String;
begin
	result := CatalogueFilter( CatalogueType( radShowAvailable, radShowUnavailable, radShowDisabled));
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function ReadSelectedKey( DataSet : TDataSet; nameOfKey : String) : Integer;
begin
	if DataSet.IsEmpty then result := -1 else result := DataSet.FieldByName( nameOfKey).AsInteger;
end;


function ReadInstanceKey( DataSet : TDataSet) : Integer;
begin
	result := ReadSelectedKey( DataSet, 'keyInstance');
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure DisableComboSet( DBCombo : TDBLookupComboBox; var Position : TDataSetPosition);
begin
	with DBCombo.ListSource do
	begin
		Position.keyInstance := ReadInstanceKey( DataSet);
		DataSet.DisableControls;
	end;
end;

procedure EnableComboSet( DBCombo : TDBLookupComboBox; Position : TDataSetPosition);
begin
	with DBCombo.ListSource do
	begin
		DataSet.EnableControls;
		SelectComboWithoutFail( DBCombo, Position.keyInstance);
	end;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure DisableGridSet( DBGrid : TDBGrid; var Position : TDataSetPosition);
begin
	with DBGrid.DataSource do
	begin
		DataSet.DisableControls;
		Position.keyInstance := ReadInstanceKey( DataSet);
		if DataSet.IsEmpty then Position.RecNo := 1 else Position.RecNo := DataSet.RecNo;
		if DataSet is TIBClientDataSet then Position.IndexFieldNames := ( DataSet as TIBClientDataSet).IndexFieldNames;
	end;
end;

procedure EnableGridSet( DBGrid : TDBGrid; Position : TDataSetPosition);
begin
	with DBGrid.DataSource do
	begin
		if not DataSet.Locate( 'keyInstance', Position.keyInstance, []) then
			if not DataSet.IsEmpty then
				DataSet.RecNo := Min( DataSet.RecordCount, Position.RecNo);
		if DataSet is TIBClientDataSet then ( DataSet as TIBClientDataSet).IndexFieldNames := Position.IndexFieldNames;
		DataSet.EnableControls;
	end;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure SortByDefault( grid : TDBGrid);
var
	ACol : Integer;
begin
	if @grid.OnTitleClick = nil then exit;

	for ACol := 0 to grid.Columns.Count - 1 do
	begin
		if grid.Columns.Items[ ACol].Color = clSorted then
		begin
			grid.OnTitleClick( grid.Columns.Items[ ACol]);
			exit;
		end;
	end;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure SelectComboWithoutFail( combo : TDBLookUpComboBox; key : Integer);
begin
	combo.KeyValue := Null;
	if not combo.ListSource.DataSet.IsEmpty then
	begin
		combo.KeyValue := key;
		if combo.Text = '' then
		begin
			combo.KeyValue := combo.ListSource.DataSet.FieldByName( combo.KeyField).AsVariant;
		end;
	end;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure OptionsToCheckBoxes( options : Integer; checkBoxes : array of TCheckBox);
var
	index : Integer;
	mask : Integer;
begin
	mask := 1;
	for index := 0 to high( checkBoxes) do
	begin
		checkBoxes[ index].Checked := ( options and mask) <> 0;
		mask := mask shl 1;
	end;
end;


function CheckBoxesToOptions( checkBoxes : array of TCheckBox) : Integer;
var
	index : Integer;
begin
	result := 0;
	for index := high( checkBoxes) downto 0 do
	begin
		result := result shl 1;

		if checkBoxes[ index].Checked then
		begin
			result := result or 1;
		end;
	end;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure OptionsToSpeedButtons( options : Integer; SpeedButtons : array of TSpeedButton);
var
	index : Integer;
	mask : Integer;
begin
	mask := 1;
	for index := 0 to high( SpeedButtons) do
	begin
		SpeedButtons[ index].Down := ( options and mask) <> 0;
		mask := mask shl 1;
	end;
end;

function SpeedButtonsToOptions( SpeedButtons : array of TSpeedButton) : Integer;
var
	index : Integer;
begin
	result := 0;
	for index := high( SpeedButtons) downto 0 do
	begin
		result := result shl 1;

		if SpeedButtons[ index].Down then
		begin
			result := result or 1;
		end;
	end;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function WithInRange( value, floor, ceil : String) : Boolean;
begin
	result := ( CompareStr( floor, value) <= 0) and ( CompareStr( ceil, value) >= 0);
end;

procedure ErrorCheck_Empty( caption, text : String; ErrorMessages : TStrings);
begin
	if Trim( text) = '' then
	begin
		ErrorMessages.Add( caption + '͂Ă܂B');
	end;
end;

procedure ErrorCheck_Reading( caption, text : String; ErrorMessages : TStrings);
var
	index, size : Integer;
begin
	text := Trim( text);
	index := 1;
	size := 0;
	while index + size <= Length( text) do
	begin
		index := index + size;
		if ByteType( text, index) = mbLeadByte then size := 2 else size := 1;

		if WithInRange( Copy( text, index, size), '', '')
		or WithInRange( Copy( text, index, size), 'O', 'X')
		or WithInRange( Copy( text, index, size), '1', '9')
		or WithInRange( Copy( text, index, size), '`', 'y')
		or WithInRange( Copy( text, index, size), 'A', 'Z')
		or ( Copy( text, index, size) = '[')
		or ( Copy( text, index, size) = 'J')
		or ( Copy( text, index, size) = '@')
		or ( Copy( text, index, size) = ' ') then
		begin
			continue;
		end;

		ErrorMessages.Add( caption + ' Ђ炪 Ƒ_ASpAp݂̂œ͂ĂB');
		exit;
	end;
end;

procedure ErrorCheck_Digits( caption, text : String; ErrorMessages : TStrings);
var
	index, size : Integer;
begin
	text := Trim( text);
	index := 1;
	size := 0;
	while index + size <= Length( text) do
	begin
		index := index + size;
		if ByteType( text, index) = mbLeadByte then size := 2 else size := 1;

		if WithInRange( Copy( text, index, size), '0', '9')
		or ( Copy( text, index, size) = '-') then
		begin
			continue;
		end;

		ErrorMessages.Add( caption + '͔pƔpnCt݂̂œ͂ĂB');
		exit;
	end;
end;

procedure ErrorCheck_Code( caption, text : String; ErrorMessages : TStrings);
var
	index, size : Integer;
begin
	text := Trim( text);
	index := 1;
	size := 0;
	while index + size <= Length( text) do
	begin
		index := index + size;
		if ByteType( text, index) = mbLeadByte then size := 2 else size := 1;

		if WithInRange( Copy( text, index, size), '0', '9')
		or WithInRange( Copy( text, index, size), 'A', 'Z')
		or ( Copy( text, index, size) = '-') then
		begin
			continue;
		end;

		ErrorMessages.Add( caption + '͔pƔpnCt݂̂œ͂ĂB');
		exit;
	end;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function ZeroToNull( const AValue : Variant) : Variant; overload;
begin
	if AValue > 0 then result := AValue else result := Null;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function IntToBool( val : Integer) : Boolean;
begin
	result := ( val <> 0);
end;


function BoolToInt( val : Boolean) : Integer;
begin
	if val then result := 1 else result := 0;
end;


function VarToInt( val : Variant) : Integer;
begin
	if val = Null then result := 0 else result := val;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function ToZipCode( text : String) : Integer;
var
	index, size : Integer;
	zip : String;
begin
	text := Trim( text);
	if text = '' then
	begin
		result := -1;
		exit;
	end;

	index := 1;
	size := 0;
	zip := '';
	while index + size <= Length( text) do
	begin
		index := index + size;
		if ByteType( text, index) = mbLeadByte then size := 2 else size := 1;

		if WithInRange( Copy( text, index, size), '0', '9') then
		begin
			zip := zip + Copy( text, index, size);
			continue;
		end;

		if ( Copy( text, index, size) = '-') then
		begin
			continue;
		end;

		result := 0;
		exit;
	end;

	result := StrToIntDef( zip, 0);
end;

function ToZipText( zip : Integer) : String;
begin
	result := LeftStr( Format( '%.7d', [ zip]), 3)
		+ '-' + RightStr( Format( '%.7d', [ zip]), 4);
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function ToDigits( text : String) : String;
var
	index, size : Integer;
begin
	index := 1;
	size := 0;
	result := '';
	while index + size <= Length( text) do
	begin
		index := index + size;
		if ByteType( text, index) = mbLeadByte then size := 2 else size := 1;

		if WithInRange( Copy( text, index, size), '0', '9') then
		begin
			result := result + Copy( text, index, size);
		end;
	end;

	result := result;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function ToAge( atDate, birthday : TDate) : Integer;
begin
	result := YearsBetween( atDate, birthday);
end;

function ToAgeText( atDate, birthday : TDate) : String;
begin
	result := IntToStr( ToAge( atDate, birthday)) + ' ';
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function MinutesToText( minutes : Integer) : String;
begin
	result := Format( '%.2d:%.2d', [ minutes div 60, minutes mod 60]);
end;

function PeriodsToText( minutesFrom, minutesTo : Integer) : String;
begin
	result := Format( '%.2d:%.2d`%.2d:%.2d', [
		minutesFrom div 60, minutesFrom mod 60,
		minutesTo div 60, minutesTo mod 60
	]);
end;

function RangeToText( minutesAt, minutesFor : Integer) : String;
begin
	result := PeriodsToText( minutesAt, minutesAt + minutesFor);
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function IntersectionOf( F1, T1, F2, T2 : Integer) : Integer;
begin
	result := Max( 0, Max( 0, T2 - F1) - Max( 0, F2 - F1) - Max( 0, T2 - T1));
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function CurrToPoint( Curr : Currency) : Currency;
begin
	result := Int( Curr * 100) / 100;
end;

function CurrToMoney( Curr : Currency) : Currency;
begin
	result := Int( Curr);
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

function AmountOf( typCategory : Integer; curAmountPerCount, curAmountPerHour : Currency; intMinutes : Integer) : Currency;
begin
	if intMinutes <= 0 then
		result := curAmountPerCount + curAmountPerHour
	else
		result := curAmountPerCount + curAmountPerHour * Max( 1, ( intMinutes + 15) div 30) / 2;

	case typCategory of
	TICKET_EXCHANGE : result := CurrToPoint( result);
	MONEY_EXCHANGE : result := CurrToMoney( result);
	end;
end;

function AmountToText( typCategory : Integer; curAmountPerCount, curAmountPerHour : Currency; intMinutes : Integer) : String;
begin
	case typCategory of
	TICKET_EXCHANGE : result := Format( '%0.2n _', [ AmountOf( typCategory, curAmountPerCount, curAmountPerHour, intMinutes)]);
	MONEY_EXCHANGE : result := Format( '%0.0n ~', [ AmountOf( typCategory, curAmountPerCount, curAmountPerHour, intMinutes)]);
	else result := '* undefined exchange category *';
	end;
end;

end.
