unit Set_Transfer;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms,
  Dialogs, Base_Set, IBSQL, IBDatabase, StdCtrls, ExtCtrls, DB,
  IBCustomDataSet, IBQuery, ComCtrls, DBCtrls, Math, StrUtils, DateUtils,
  ActnList, Mask,  IBUpdateSQL, Misc_Constants, Misc_Utilities,
  Piece_ServiceSelector, Piece_MinutesSelector, Piece_AddressSelector,
  Piece_ProfileSelector, Piece_ClientSelector, Piece_ServerSelector,
  Piece_PatternSelector, MainDatastore, Misc_Scheduler;

type
  TTransferSet = class(TSetTemplate)
    Schedule: TIBQuery;
    UpdaterOf_Schedule: TIBUpdateSQL;

    Subject: TIBQuery;

    Pattern: TPatternSelector;
    Service: TServiceSelector;

    MinutesFrame: TShape;
    pnlMinutes: TPanel;
    lblMinutes: TLabel;
    MinutesAt: TMinutesSelector;

    AddressFrame: TShape;
    pnlAddress: TPanel;
    lblAddress: TLabel;
    AddressFrom: TAddressSelector;
    lblAddressOver: TLabel;
    AddressTo: TAddressSelector;

    ClientFrame: TShape;
    Client: TClientSelector;

    ServerFrame: TShape;
    Server: TServerSelector;

    btnQuoteFromPrevMonth: TButton;
    btnQuoteFromThisMonth: TButton;

    chkCopyNotice: TCheckBox;
    chkCopyServer: TCheckBox;
    btnBrowseSubject: TButton;

    ShortCuts: TActionList;
    Shortcut_ChooseAddressFrom: TAction;
    Shortcut_ChooseAddressTo: TAction;
    Shortcut_ChooseServer: TAction;

    procedure Client_Change(Sender: TObject);
    procedure Server_Change(Sender: TObject);

    procedure btnChooseServer_Click(Sender: TObject);

    procedure btnQuoteSchedule_Click(Sender: TObject);

    procedure btnBrowseSubject_Click(Sender: TObject);

    procedure ChooseAddressFrom_Execute(Sender: TObject);
    procedure ChooseAddressTo_Execute(Sender: TObject);
    procedure ChooseServer_Execute(Sender: TObject);

  private
    { Private 錾 }
    FSubjectKey : Integer;
    FSchedule : TDate;
    FMinutesAt : Integer;
    FClientKey : Integer;
    FServerKey : Integer;
  protected
    { Protected 錾 }
    procedure DefineTable( var nameOfTable : String; var modifyFlag, cancelFlag : Boolean); override;

    procedure BeforeInput(); override;
    procedure AfterInput(); override;

    procedure BeforeSelect(); override;
    procedure AfterSelect(); override;

    procedure BeforeAppend(); override;
    procedure AfterAppend(); override;

    procedure BeforePopup(); override;
    procedure AfterPopup(); override;
  public
    { Public 錾 }
    constructor Prepare(); overload; override;
    constructor Prepare( datSchedule : TDate; intMinutesAt, keyClient : Integer); overload;
    constructor Prepare( keySubject, intMinutesAt : Integer); overload;
    constructor Prepare( keySubject : Integer); overload;
    constructor Arrange( keyServer : Integer); overload;
  end;

var
  TransferSet: TTransferSet;

implementation

{$R *.dfm}

uses Chooser_Candidate, Chooser_Schedule, Set_Care, Set_Menage;

{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

constructor TTransferSet.Prepare();
begin
	inherited Prepare;

	FSubjectKey := -1;
	FSchedule := Today;
	FMinutesAt := 0;
	FClientKey := -1;
	FServerKey := -1;
end;

constructor TTransferSet.Prepare( datSchedule : TDate; intMinutesAt, keyClient : Integer);
begin
	inherited Prepare;

	FSubjectKey := -1;
	FSchedule := datSchedule;
	FMinutesAt := intMinutesAt;
	FClientKey := keyClient;
	FServerKey := -1;
end;

constructor TTransferSet.Prepare( keySubject, intMinutesAt : Integer);
begin
	inherited Prepare;

	FSubjectKey := keySubject;
	FSchedule := Today;
	FMinutesAt := intMinutesAt;
	FClientKey := -1;
	FServerKey := -1;
end;

constructor TTransferSet.Prepare( keySubject : Integer);
begin
	inherited Prepare;

	FSubjectKey := keySubject;
	FSchedule := Today;
	FMinutesAt := 9999;
	FClientKey := -1;
	FServerKey := -1;
end;

constructor TTransferSet.Arrange( keyServer : Integer);
begin
	inherited Prepare;

	FSubjectKey := -1;
	FSchedule := Today;
	FMinutesAt := 0;
	FClientKey := -1;
	FServerKey := keyServer;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TTransferSet.DefineTable( var nameOfTable : String; var modifyFlag, cancelFlag : Boolean);
begin
	nameOfTable := 'Schedule';
	modifyFlag := true;
	cancelFlag := true;

	Client.OnChange := Client_Change;
	Server.OnChange := Server_Change;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TTransferSet.BeforeInput();
begin
	Schedule.ParamByName( 'keyInstance').AsInteger := PrimaryKey;
	Schedule.Open;
end;

procedure TTransferSet.AfterInput();
var
	PatternKey : Integer;
begin
	if Appending and Pattern.Extendable then
		PatternKey := RealizedKey
	else
		PatternKey := Schedule.FieldByName( 'keyPattern').AsInteger;

	with Schedule do
	begin
		if Appending then Append else Edit;
		FieldByName( 'keyInstance').AsInteger := RealizedKey;
		FieldByName( 'datSchedule').AsDateTime := FSchedule;
		FieldByName( 'keyPattern').AsInteger := PatternKey;
		FieldByName( 'typPattern').AsInteger := Pattern.TypeCode;
		FieldByName( 'optPattern').AsInteger := Pattern.MaskCode;
		FieldByName( 'optElement').AsInteger := CheckBoxesToOptions( [ chkCopyServer, chkCopyNotice]);
		FieldByName( 'typProgress').AsInteger := Service.Progress;
		FieldByName( 'refSubject').AsVariant := ZeroToNull( FSubjectKey);
		FieldByName( 'refService').AsInteger := Service.ServiceKey;
		FieldByName( 'refExchange').AsInteger := Service.ExchangeKey;
		FieldByName( 'intMinutesAt').AsInteger := MinutesAt.Value;
		FieldByName( 'intMinutesFor').AsInteger := 0;
		FieldByName( 'refAddressFrom').AsInteger := AddressFrom.KeyValue;
		FieldByName( 'refAddressTo').AsInteger := AddressTo.KeyValue;
		FieldByName( 'strNotice').AsString := Service.Notice;
		FieldByName( 'refClient').AsVariant := ZeroToNull( Client.KeyValue);
		FieldByName( 'refServer').AsVariant := ZeroToNull( Server.KeyValue);
		Post;
		ApplyUpdates;
	end;

	if Appending and Pattern.Extendable then
	begin
		with Scheduler do
		begin
			Transaction := self.Transaction;

			datAnchor := FSchedule;

			keyPattern := PatternKey;
			typPattern := Pattern.TypeCode;
			optPattern := Pattern.MaskCode;
			optElement := CheckBoxesToOptions( [ chkCopyServer, chkCopyNotice]);

			typProgress := Service.Progress;
			refService := Service.ServiceKey;
			refExchange := Service.ExchangeKey;

			intMinutesAt := MinutesAt.Value;
			intMinutesFor := 0;
			refAddressFrom := AddressFrom.KeyValue;
			refAddressTo := AddressTo.KeyValue;
			strNotice := IfThen( chkCopyNotice.Checked, Service.Notice, '');

			refClient := Client.KeyValue;
			refServer := IfThen( chkCopyServer.Checked, Server.KeyValue, 0);
		end;

		if Subject.IsEmpty then
		begin
			Scheduler.CopyAsSubject;
		end
		else if ShowConfirmation( '\Ƃĕʂ܂H', 'ʉ\ł', MB_YESNO) = IDYES then
		begin
			Scheduler.CopyAsLemma( Subject.FieldByName( 'keyPattern').AsInteger);
		end;
	end;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TTransferSet.BeforeSelect();
begin
	with Schedule do
	begin
		ParamByName( 'keyInstance').AsInteger := PrimaryKey;
		Open;

		FSubjectKey := FieldByName( 'refSubject').AsInteger;
		FSchedule := FieldByName( 'datSchedule').AsDateTime;
		FMinutesAt := FieldByName( 'intMinutesAt').AsInteger;
		FClientKey := FieldByName( 'refClient').AsInteger;
		if ( FServerKey <= 0) or Browsing then FServerKey := FieldByName( 'refServer').AsInteger;

		Subject.ParamByName( 'keyInstance').AsInteger := FSubjectKey;
		Subject.Open;
		// value setup

		if not Subject.IsEmpty then
			Pattern.Reset( FSchedule, Subject.FieldByName( 'typPattern').AsInteger, Subject.FieldByName( 'optPattern').AsInteger)
		else
			Pattern.Reset( FSchedule, FieldByName( 'typPattern').AsInteger, FieldByName( 'optPattern').AsInteger);

		Service.Reset( Transaction, FSchedule, TRANSFER_SERVICE,
			FieldByName( 'typProgress').AsInteger,
			FieldByName( 'refService').AsInteger,
			FieldByName( 'refExchange').AsInteger,
			FieldByName( 'strNotice').AsString
		);

		MinutesAt.Value := FMinutesAt;

		AddressFrom.Reset( Transaction, FSchedule, FieldByName( 'refAddressFrom').AsInteger, FClientKey, FServerKey);
		AddressTo.Reset( Transaction, FSchedule, FieldByName( 'refAddressTo').AsInteger, FClientKey, FServerKey);

		Client.Reset( Transaction, FSchedule, FClientKey);
		Server.Reset( Transaction, FSchedule, FServerKey);

		OptionsToCheckBoxes( FieldByName( 'optElement').AsInteger, [ chkCopyServer, chkCopyNotice]);
	end;
end;

procedure TTransferSet.AfterSelect();
begin
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TTransferSet.BeforeAppend();
var
	keyAddressFrom, keyAddressTo : Integer;
begin
	with Subject do
	begin
		ParamByName( 'keyInstance').AsInteger := FSubjectKey;
		Open;

		if not IsEmpty then
		begin
			if FieldByName( 'typStatus').AsInteger <> DATA_ENABLED then
			begin
				ErrorMessages.Add( 'w肳ꂽ\݂͑܂B');
				exit;
			end;

			FSchedule := FieldByName( 'datSchedule').AsDateTime;
			FClientKey := FieldByName( 'refClient').AsInteger;

			Pattern.Reset( FSchedule, FieldByName( 'typPattern').AsInteger, FieldByName( 'optPattern').AsInteger);

			if FieldByName( 'typService').AsInteger <> TRANSFER_SERVICE then
			begin
				if FMinutesAt < FieldByName( 'intMinutesAtMid').AsInteger then
					FMinutesAt := FieldByName( 'intMinutesFrom').AsInteger
				else
					FMinutesAt := FieldByName( 'intMinutesTo').AsInteger;

				if FMinutesAt <= Subject.FieldByName( 'intMinutesAt').AsInteger then
				begin
					keyAddressFrom := -1;
					keyAddressTo := FieldByName( 'refAddressFrom').AsInteger;
				end
				else
				begin
					keyAddressFrom := FieldByName( 'refAddressTo').AsInteger;
					keyAddressTo := -1;
				end;
			end
			else
			begin
				FMinutesAt := FieldByName( 'intMinutesAt').AsInteger;
				keyAddressFrom := FieldByName( 'refAddressFrom').AsInteger;
				keyAddressTo := FieldByName( 'refAddressTo').AsInteger;
			end;
		end
		else
		begin
			Pattern.Reset( FSchedule, PATTERN_SINGLETON, 0);

			keyAddressFrom := -1;
			keyAddressTo := -1;
		end;
	end;

	Service.Reset( Transaction, FSchedule, TRANSFER_SERVICE, UNCLEAR_PROGRESS, -1, -1, '');

	MinutesAt.Value := FMinutesAt;

	AddressFrom.Reset( Transaction, FSchedule, keyAddressFrom, FClientKey, FServerKey);
	AddressTo.Reset( Transaction, FSchedule, keyAddressTo, FClientKey, FServerKey);

	Client.Reset( Transaction, FSchedule, FClientKey);
	Server.Reset( Transaction, FSchedule, FServerKey);

	Client.btnChooseProfile.Visible := Subject.IsEmpty;
end;

procedure TTransferSet.AfterAppend();
begin
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TTransferSet.BeforePopup();
begin
	Pattern.ReadOnly := ( not Appending) or ( not Subject.IsEmpty);
	Service.ReadOnly := Browsing;

	MinutesAt.ReadOnly := Browsing;

	AddressFrom.ReadOnly := Browsing;
	AddressTo.ReadOnly := Browsing;

	Client.ReadOnly := Browsing;
	Server.ReadOnly := Browsing;

	btnQuoteFromPrevMonth.Enabled := not Browsing;
	btnQuoteFromThisMonth.Enabled := not Browsing;

	chkCopyServer.Enabled := not Browsing;
	chkCopyNotice.Enabled := not Browsing;

	btnBrowseSubject.Visible := not Subject.IsEmpty;

	if Appending then Service.SetFocus;
	if Selecting then MinutesAt.SetFocus;
	// enable controls

	if Subject.IsEmpty then
		Caption := FormatDateTime( '"ڑ"yyyy"N"MM""dd"("ddd")"', FSchedule)
	else
		case Subject.FieldByName( 'typService').AsInteger of
		CARE_SERVICE :
			begin
				Caption := FormatDateTime( '"̈ڑ"yyyy"N"MM""dd"("ddd")"', FSchedule);
				Service.Color := clMoneyGreen;
			end;

		MENAGE_SERVICE :
			begin
				Caption := FormatDateTime( '"Ǝ̈ڑ"yyyy"N"MM""dd"("ddd")"', FSchedule);
				Service.Color := clMoneyGreen;
			end;

		TRANSFER_SERVICE :
			Caption := FormatDateTime( '"ڑ̕⏕"yyyy"N"MM""dd"("ddd")"', FSchedule);
		end;

	Pattern.Color := Service.Color;

	if not Appending then Caption := Format( '%s(%d)', [ Caption, PrimaryKey]);
	// update caption
end;

procedure TTransferSet.AfterPopup();
var
	Verifications : TStringList;
begin
	if Pattern.IsInvalid then
		ErrorMessages.Add( '\tɂȂĂ܂B');

	if Service.ServiceKey <= 0 then
		ErrorMessages.Add( 'T[rXIĂ܂B');

	if Service.ExchangeKey <= 0 then
		ErrorMessages.Add( 'ӗ炪IĂ܂B');

	if not AddressFrom.Selected then
		ErrorMessages.Add( 'onIĂ܂B');

	if not AddressTo.Selected then
		ErrorMessages.Add( 'ړInIĂ܂B');

	if not Client.Selected then
		ErrorMessages.Add( 'p҂IĂ܂B');

	if Client.KeyValue = Server.KeyValue then
		ErrorMessages.Add( 'hCo[Ɨp҂ɓ̓o^҂IԂƂ͂ł܂B');

	if ErrorMessages.Count > 0 then exit;
	// error check

	Verifications := TStringList.Create;
	try
		if AddressFrom.Unclear then Verifications.Add( 'on͂ɂȂĂ܂B');
		if AddressTo.Unclear then Verifications.Add( 'ړIn͂ɂȂĂ܂B');

		if Verifications.Count = 0 then
		begin
			if Service.Progress = UNCLEAR_PROGRESS then
			begin
				if ShowConfirmation( 'Kvȏ񂪑Sē͂܂B̗\m肵܂H', '\̊m', MB_YESNO) = IDYES then
				begin
					Service.Progress := DEFINED_PROGRESS;
				end;
			end;
		end
		else
		begin
			if Service.Progress <> UNCLEAR_PROGRESS then
			begin
				ShowNotify( Verifications.Text + 'L̂悤ȕs邽߁A̗\͊m/~܂B', '\̊m');
				Service.Progress := UNCLEAR_PROGRESS;
			end;
		end;
	finally
		Verifications.Free;
	end;
	// progress check
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TTransferSet.Client_Change(Sender: TObject);
begin
	AddressFrom.ClientKey := Client.KeyValue;
	AddressTo.ClientKey := Client.KeyValue;
end;

procedure TTransferSet.Server_Change(Sender: TObject);
begin
	AddressFrom.ServerKey := Server.KeyValue;
	AddressTo.ServerKey := Server.KeyValue;

	if ( not Preparing) and btnOK.Enabled then btnOK.SetFocus;
end;


{*=========================================================*
  @subject: 
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TTransferSet.btnChooseServer_Click(Sender: TObject);
var
	keyServer : Integer;
begin
	if ChooseCandidate(
		Transaction, FSchedule,
		Pattern.TypeCode, Pattern.MaskCode,
		MinutesAt.Value, MinutesAt.Value + 15,
		keyServer
	) then
		Server.KeyValue := keyServer;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TTransferSet.btnQuoteSchedule_Click(Sender: TObject);
var
	info : TScheduleInfo;
begin
	if Sender = btnQuoteFromThisMonth then
		QuoteSchedule( Transaction, FSchedule, FClientKey, TRANSFER_SERVICE, info)
	else
		QuoteSchedulePrevMonth( Transaction, FSchedule, FClientKey, TRANSFER_SERVICE, info);

	if info.refService > 0 then
	begin
		Service.Reset( info.refService, info.refExchange, info.strNotice);
	end;

	if info.intMinutesAt >= 0 then
	begin
		MinutesAt.Value := info.intMinutesAt;
	end;

	if info.refAddressFrom > 0 then
	begin
		AddressFrom.KeyValue := info.refAddressFrom;
		AddressTo.KeyValue := info.refAddressTo;
	end;

	if info.refServer > 0 then
	begin
		Server.KeyValue := info.refServer;
	end;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TTransferSet.btnBrowseSubject_Click(Sender: TObject);
begin
	case Subject.FieldByName( 'typService').AsInteger of
	    CARE_SCHEDULE : TCareSet.Prepare.Browse( Subject.FieldByName( 'keyInstance').AsInteger);
	  MENAGE_SCHEDULE : TMenageSet.Prepare.Browse( Subject.FieldByName( 'keyInstance').AsInteger);
	TRANSFER_SCHEDULE : TTransferSet.Prepare.Browse( Subject.FieldByName( 'keyInstance').AsInteger);
	end;
end;


{*=========================================================*
  @subject:
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*}

procedure TTransferSet.ChooseAddressFrom_Execute(Sender: TObject);
begin
	if not Browsing then AddressFrom.Choose;
end;

procedure TTransferSet.ChooseAddressTo_Execute(Sender: TObject);
begin
	if not Browsing then AddressTo.Choose;
end;

procedure TTransferSet.ChooseServer_Execute(Sender: TObject);
begin
	if not Browsing then Server.Choose;
end;

end.
