
/*******************************************************************************
  @file: Journal_Utility.sql
  @subject: miscellaneous definition for journal tables
  @update: 2004/05/08 (Sat) 00:00:00
  = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =
 *******************************************************************************/

INPUT 'Locator.sql';

/*=========================================================*
  @subject: procedure definition - Schedules, Jobs
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*/

SET TERM !! ;
RECREATE PROCEDURE Schedules ( datAt  DATE, optService  INTEGER) RETURNS (
		keyInstance  INTEGER,
		typStatus  INTEGER,
		typSchedule  INTEGER,

		keyPattern  INTEGER, typPattern  INTEGER,

		refSubject  INTEGER, refSubjectSubject  INTEGER,
		typSubject  INTEGER, intSubjectAt  INTEGER, intSubjectFor  INTEGER,
		refSubjectServer  INTEGER,

		datSchedule  DATE,
		typProgress  INTEGER,
		refService  INTEGER, strService  VARCHAR(32), refExchange  INTEGER,
		intMinutesAt  INTEGER, intMinutesFor  INTEGER,
		refAddressFrom  INTEGER, refAddressTo  INTEGER,
		strNotice  VARCHAR(48),

		refRecipient  INTEGER, refClient  INTEGER, refServer  INTEGER
	) AS
	BEGIN
		FOR SELECT
			sch.keyInstance,
			sch.typStatus,
			sv.typCategory,

			sch.keyPattern, sch.typPattern,

			sch.refSubject, sb.refSubject,
			sbsv.typCategory, sb.intMinutesAt, sb.intMinutesFor,
			sb.refServer,

			sch.datSchedule,
			sch.typProgress,
			sch.refService, sv.strName, sch.refExchange,
			sch.intMinutesAt, sch.intMinutesFor,
			sch.refAddressFrom, sch.refAddressTo,
			sch.strNotice,

			EmptyInteger( sb.refClient, sch.refClient), sch.refClient, sch.refServer

		FROM Schedule sch
			LEFT JOIN Service sv ON sv.keyInstance = sch.refService

			LEFT JOIN Schedule sb ON sb.keyInstance = sch.refSubject
			LEFT JOIN Service sbsv ON sbsv.keyInstance = sb.refService

		WHERE sch.datSchedule = :datAt
			and bitAnd( shiftLeft( 1, sv.typCategory), :optService) <> 0

		INTO	:keyInstance,
			:typStatus,
			:typSchedule,

			:keyPattern, :typPattern,

			:refSubject, :refSubjectSubject,
			:typSubject, :intSubjectAt, :intSubjectFor,
			:refSubjectServer,

			:datSchedule,
			:typProgress,
			:refService, :strService, :refExchange,
			:intMinutesAt, :intMinutesFor,
			:refAddressFrom, :refAddressTo,
			:strNotice,

			:refRecipient, :refClient, :refServer
		DO
		BEGIN
			SUSPEND;
		END
	END !!
SET TERM ; !!

SET TERM !! ;
RECREATE PROCEDURE Jobs ( datAt  DATE, optService  INTEGER) RETURNS (
		keyInstance  INTEGER,
		typStatus  INTEGER,
		typSchedule  INTEGER,

		keyPattern  INTEGER, typPattern  INTEGER,

		refSubject  INTEGER, refSubjectSubject  INTEGER,
		typSubject  INTEGER, intSubjectAt  INTEGER, intSubjectFor  INTEGER,
		refSubjectServer  INTEGER,

		datSchedule  DATE,
		typProgress  INTEGER,
		refService  INTEGER, strService  VARCHAR(32), refExchange  INTEGER,
		intMinutesAt  INTEGER, intMinutesFor  INTEGER,
		refAddressFrom  INTEGER, refAddressTo  INTEGER,
		strNotice  VARCHAR(48),

		refClient  INTEGER, refServer  INTEGER
	) AS
	BEGIN
		FOR SELECT
			sch.keyInstance,
			sch.typStatus,
			sv.typCategory,

			sch.keyPattern, sch.typPattern,

			sch.refSubject, sb.refSubject,
			sbsv.typCategory, sb.intMinutesAt, sb.intMinutesFor,
			sb.refServer,

			sch.datSchedule,
			sch.typProgress,
			sch.refService, sv.strName, sch.refExchange,
			sch.intMinutesAt, sch.intMinutesFor,
			sch.refAddressFrom, sch.refAddressTo,
			sch.strNotice,

			sch.refClient, sch.refServer

		FROM Schedule sch
			LEFT JOIN Service sv ON sv.keyInstance = sch.refService

			LEFT JOIN Schedule sb ON sb.keyInstance = sch.refSubject
			LEFT JOIN Service sbsv ON sbsv.keyInstance = sb.refService

		WHERE sch.datSchedule = :datAt
			and sch.refServer IS NULL
			and sch.typProgress = 1 /* DEFINED_PROGRESS */
			and sch.typStatus = 0 /* DATA_ENABLED */
			and bitAnd( shiftLeft( 1, sv.typCategory), :optService) <> 0

		INTO	:keyInstance,
			:typStatus,
			:typSchedule,

			:keyPattern, :typPattern,

			:refSubject, :refSubjectSubject,
			:typSubject, :intSubjectAt, :intSubjectFor,
			:refSubjectServer,

			:datSchedule,
			:typProgress,
			:refService, :strService, :refExchange,
			:intMinutesAt, :intMinutesFor,
			:refAddressFrom, :refAddressTo,
			:strNotice,

			:refClient, :refServer
		DO
		BEGIN
			SUSPEND;
		END
	END !!
SET TERM ; !!


/*=========================================================*
  @subject: procedure definition - ClientSchedules, ServerSchedules
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*/

SET TERM !! ;
RECREATE PROCEDURE ClientSchedules ( keyClient  INTEGER, datFrom  DATE, datTo  DATE) RETURNS (
		keyInstance  INTEGER,
		typStatus  INTEGER,
		typSchedule  INTEGER,

		keyPattern  INTEGER, typPattern  INTEGER, optPattern  INTEGER, optElement  INTEGER,

		refSubject  INTEGER, refSubjectSubject  INTEGER,
		typSubject  INTEGER, intSubjectAt  INTEGER, intSubjectFor  INTEGER,
		refSubjectServer  INTEGER,

		datSchedule  DATE,
		typProgress  INTEGER,
		refService  INTEGER, strService  VARCHAR(32), refExchange  INTEGER,
		intMinutesAt  INTEGER, intMinutesFor  INTEGER,
		refAddressFrom  INTEGER, refAddressTo  INTEGER,
		strNotice  VARCHAR(48),

		refRecipient  INTEGER, refClient  INTEGER, refServer  INTEGER
	) AS
	BEGIN
		FOR SELECT
			sch.keyInstance,
			sch.typStatus,
			sv.typCategory,

			sch.keyPattern, sch.typPattern, sch.optPattern, sch.optElement,

			sch.refSubject, sb.refSubject,
			sbsv.typCategory, sb.intMinutesAt, sb.intMinutesFor,
			sb.refServer,

			sch.datSchedule,
			sch.typProgress,
			sch.refService, sv.strName, sch.refExchange,
			sch.intMinutesAt, sch.intMinutesFor,
			sch.refAddressFrom, sch.refAddressTo,
			sch.strNotice,

			EmptyInteger( sb.refClient, sch.refClient), sch.refClient, sch.refServer

		FROM Schedule sch
			LEFT JOIN Service sv ON sv.keyInstance = sch.refService

			LEFT JOIN Schedule sb ON sb.keyInstance = sch.refSubject
			LEFT JOIN Service sbsv ON sbsv.keyInstance = sb.refService

		WHERE sch.datSchedule BETWEEN :datFrom and :datTo
			and ( sch.refClient = :keyClient or sb.refClient = :keyClient)

		INTO	:keyInstance,
			:typStatus,
			:typSchedule,

			:keyPattern, :typPattern, :optPattern, :optElement,

			:refSubject, :refSubjectSubject,
			:typSubject, :intSubjectAt, :intSubjectFor,
			:refSubjectServer,

			:datSchedule,
			:typProgress,
			:refService, :strService, :refExchange,
			:intMinutesAt, :intMinutesFor,
			:refAddressFrom, :refAddressTo,
			:strNotice,

			:refRecipient, :refClient, :refServer
		DO
		BEGIN
			SUSPEND;
		END
	END !!
SET TERM ; !!

SET TERM !! ;
RECREATE PROCEDURE ServerSchedules ( keyServer  INTEGER, datFrom  DATE, datTo  DATE) RETURNS (
		keyInstance  INTEGER,
		typStatus  INTEGER,
		typSchedule  INTEGER,

		keyPattern  INTEGER, typPattern  INTEGER,

		refSubject  INTEGER, refSubjectSubject  INTEGER,
		typSubject  INTEGER, intSubjectAt  INTEGER, intSubjectFor  INTEGER,
		refSubjectServer  INTEGER,

		datSchedule  DATE,
		typProgress  INTEGER,
		refService  INTEGER, strService  VARCHAR(32), refExchange  INTEGER,
		intMinutesAt  INTEGER, intMinutesFor  INTEGER,
		refAddressFrom  INTEGER, refAddressTo  INTEGER,
		strNotice  VARCHAR(48),

		refRecipient  INTEGER, refClient  INTEGER, refServer  INTEGER
	) AS
	BEGIN
		FOR SELECT
			sch.keyInstance,
			sch.typStatus,
			sv.typCategory,

			sch.keyPattern, sch.typPattern,

			sch.refSubject, sb.refSubject,
			sbsv.typCategory, sb.intMinutesAt, sb.intMinutesFor,
			sb.refServer,

			sch.datSchedule,
			sch.typProgress,
			sch.refService, sv.strName, sch.refExchange,
			sch.intMinutesAt, sch.intMinutesFor,
			sch.refAddressFrom, sch.refAddressTo,
			sch.strNotice,

			EmptyInteger( sb.refClient, sch.refClient), sch.refClient, sch.refServer

		FROM Schedule sch
			LEFT JOIN Service sv ON sv.keyInstance = sch.refService

			LEFT JOIN Schedule sb ON sb.keyInstance = sch.refSubject
			LEFT JOIN Service sbsv ON sbsv.keyInstance = sb.refService

		WHERE sch.datSchedule BETWEEN :datFrom and :datTo
			and sch.refServer = :keyServer

		INTO	:keyInstance,
			:typStatus,
			:typSchedule,

			:keyPattern, :typPattern,

			:refSubject, :refSubjectSubject,
			:typSubject, :intSubjectAt, :intSubjectFor,
			:refSubjectServer,

			:datSchedule,
			:typProgress,
			:refService, :strService, :refExchange,
			:intMinutesAt, :intMinutesFor,
			:refAddressFrom, :refAddressTo,
			:strNotice,

			:refRecipient, :refClient, :refServer
		DO
		BEGIN
			SUSPEND;
		END
	END !!
SET TERM ; !!


/*=========================================================*
  @subject: procedure definition - Clients, Servers
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*/

SET TERM !! ;
RECREATE PROCEDURE Clients ( datFrom  DATE, datTo  DATE) RETURNS ( keyInstance  INTEGER) AS
	DECLARE VARIABLE keySupport  INTEGER;
	BEGIN
		FOR
		SELECT DISTINCT sch.refClient
		FROM Schedule sch LEFT JOIN Profile pr ON pr.keyInstance = sch.refClient
		WHERE sch.datSchedule BETWEEN :datFrom and :datTo
		and sch.typStatus = 0 /* DATA_ENABLED */
		and pr.typCategory = 0 /* CLIENT_PROFILE */
		INTO :keyInstance
		DO
		BEGIN
			SUSPEND;
		END
	END !!
SET TERM ; !!

SET TERM !! ;
RECREATE PROCEDURE Servers ( datFrom  DATE, datTo  DATE) RETURNS ( keyInstance  INTEGER) AS
	DECLARE VARIABLE keySupport  INTEGER;
	BEGIN
		FOR
		SELECT DISTINCT sch.refServer
		FROM Schedule sch LEFT JOIN Profile pr ON pr.keyInstance = sch.refServer
		WHERE sch.datSchedule BETWEEN :datFrom and :datTo
		and sch.typStatus = 0 /* DATA_ENABLED */
		and pr.typCategory = 1 /* SERVER_PROFILE */
		INTO :keyInstance
		DO
		BEGIN
			SUSPEND;
		END
	END !!
SET TERM ; !!


/*=========================================================*
  @subject: procedure definition - Reserves
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*/

SET TERM !! ;
RECREATE PROCEDURE Reserves ( datFrom  DATE, datTo  DATE, optService  INTEGER) RETURNS (
		keyInstance  INTEGER,
		typStatus  INTEGER,
		typSchedule  INTEGER,

		keyPattern  INTEGER, typPattern  INTEGER, optPattern  INTEGER,

		datSchedule  DATE,
		typProgress  INTEGER,
		refService  INTEGER, strService  VARCHAR(32), refExchange  INTEGER,
		intMinutesAt  INTEGER, intMinutesFor  INTEGER,
		refAddressFrom  INTEGER, refAddressTo  INTEGER,
		strNotice  VARCHAR(48),

		refClient  INTEGER, refServer  INTEGER
	) AS
	BEGIN
		FOR SELECT
			sch.keyInstance,
			sch.typStatus,
			sv.typCategory,

			sch.keyPattern, sch.typPattern, sch.optPattern,

			sch.datSchedule,
			sch.typProgress,
			sch.refService, sv.strName, sch.refExchange,
			sch.intMinutesAt, sch.intMinutesFor,
			sch.refAddressFrom, sch.refAddressTo,
			sch.strNotice,

			sch.refClient, sch.refServer

		FROM Schedule sch
			LEFT JOIN Service sv ON sv.keyInstance = sch.refService

		WHERE sch.datSchedule BETWEEN :datFrom and :datTo
			and bitAnd( shiftLeft( 1, sv.typCategory), :optService) <> 0
			and sch.refSubject IS NULL
			and sch.typSequence = 1 /* SEQUENCE_TAIL */

		INTO	:keyInstance,
			:typStatus,
			:typSchedule,

			:keyPattern, :typPattern, :optPattern,

			:datSchedule,
			:typProgress,
			:refService, :strService, :refExchange,
			:intMinutesAt, :intMinutesFor,
			:refAddressFrom, :refAddressTo,
			:strNotice,

			:refClient, :refServer
		DO
		BEGIN
			SUSPEND;
		END
	END !!
SET TERM ; !!


/*=========================================================*
  @subject: procedure definition - Paybacks, Billings, Payments
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*/

SET TERM !! ;
RECREATE PROCEDURE Paybacks ( datFrom  DATE, datTo  DATE) RETURNS ( keyInstance  INTEGER) AS
	DECLARE VARIABLE keySupport  INTEGER;
	BEGIN
		FOR
			SELECT DISTINCT sch.refServer
			FROM Schedule sch
				LEFT JOIN Exchange ex ON ex.keyInstance = sch.refExchange
				LEFT JOIN Profile pr ON pr.keyInstance = sch.refServer
			WHERE sch.datSchedule BETWEEN :datFrom and :datTo
				and ex.typCategory = 0 /* TICKET_EXCHANGE */
				and sch.typStatus = 0 /* DATA_ENABLED */
				and pr.typCategory = 1 /* SERVER_PROFILE */
		UNION
			SELECT DISTINCT refProfile
			FROM Payback py
				WHERE py.datMonthFirstday <= :datTo
					and py.datMonthLastday >= :datFrom
					and py.typStatus = 0 /* DATA_ENABLED */
		INTO :keyInstance
		DO
		BEGIN
			SUSPEND;
		END
	END !!
SET TERM ; !!

SET TERM !! ;
RECREATE PROCEDURE Billings ( datFrom  DATE, datTo  DATE) RETURNS ( keyInstance  INTEGER) AS
	DECLARE VARIABLE keySupport  INTEGER;
	BEGIN
		FOR
			SELECT DISTINCT sch.refClient
			FROM Schedule sch
				LEFT JOIN Exchange ex ON ex.keyInstance = sch.refExchange
				LEFT JOIN Profile pr ON pr.keyInstance = sch.refClient
			WHERE sch.datSchedule BETWEEN :datFrom and :datTo
				and ex.typCategory = 1 /* MONEY_EXCHANGE */
				and sch.typStatus = 0 /* DATA_ENABLED */
				and pr.typCategory = 0 /* CLIENT_PROFILE */
		UNION
			SELECT DISTINCT refProfile
			FROM Billing bg
				WHERE bg.datMonthFirstday <= :datTo
					and bg.datMonthLastday >= :datFrom
					and bg.typStatus = 0 /* DATA_ENABLED */
		INTO :keyInstance
		DO
		BEGIN
			SUSPEND;
		END
	END !!
SET TERM ; !!

SET TERM !! ;
RECREATE PROCEDURE Payments ( datFrom  DATE, datTo  DATE) RETURNS ( keyInstance  INTEGER) AS
	DECLARE VARIABLE keySupport  INTEGER;
	BEGIN
		FOR
			SELECT DISTINCT sch.refServer
			FROM Schedule sch
				LEFT JOIN Exchange ex ON ex.keyInstance = sch.refExchange
				LEFT JOIN Profile pr ON pr.keyInstance = sch.refServer
			WHERE sch.datSchedule BETWEEN :datFrom and :datTo
				and ex.typCategory = 1 /* MONEY_EXCHANGE */
				and sch.typStatus = 0 /* DATA_ENABLED */
				and pr.typCategory = 1 /* SERVER_PROFILE */
		UNION
			SELECT DISTINCT refProfile
			FROM Payment py
				WHERE py.datMonthFirstday <= :datTo
					and py.datMonthLastday >= :datFrom
					and py.typStatus = 0 /* DATA_ENABLED */
		INTO :keyInstance
		DO
		BEGIN
			SUSPEND;
		END
	END !!
SET TERM ; !!


/*=========================================================*
  @subject: procedure definition - Candidates
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*/

SET TERM !! ;
RECREATE PROCEDURE Candidates (
		datAt  DATE,
		optAvailability  INTEGER,
		strAvailabilityReq  VARCHAR(64),
		keyBranch  INTEGER,
		keyRegion  INTEGER,
		strCodeReq  VARCHAR(32),
		strFamilyNameRead  VARCHAR(64),
		strFamilyName  VARCHAR(64),
		strFirstNameRead  VARCHAR(64),
		strFirstName  VARCHAR(64),
		typSex  INTEGER,
		useAddress  INTEGER,
		strAddressReq  VARCHAR(64),
		useNotice  INTEGER,
		strTitle   VARCHAR(64),
		strContent  VARCHAR(64)
	) RETURNS (
		keyInstance  INTEGER,
		strCode  VARCHAR(32),
		strName  VARCHAR(64),
		strNameRead  VARCHAR(128),
		strAvailability  VARCHAR(64),
		strBranch  VARCHAR(32),
		strRegion  VARCHAR(32),
		refZip  INTEGER,
		strAddress  VARCHAR(128),
		strAddressRead  VARCHAR(128),
		strPhoneNumber1  VARCHAR(32),
		strFaxNumber1  VARCHAR(32)
	) AS
	BEGIN
		FOR SELECT
			pr.keyInstance,
			pr.strCode,
			pr.strName,
			pr.strNameRead,
			pr.strAvailability,
			br.strName,
			rg.strName,
			adx.refZip,
			zpx.strAddress || adx.strAddress,
			zpx.strAddressRead || adx.strAddressRead,
			adx.strPhoneNumber1,
			adx.strFaxNumber1

		FROM Profile pr
			LEFT JOIN Branch br ON br.keyInstance = pr.refBranch
			LEFT JOIN Region rg ON rg.keyInstance = pr.refRegion
			LEFT JOIN Address adx ON adx.refProfile = pr.keyInstance
				and adx.codRevision = (
					SELECT Max( adx2.codRevision) FROM Address adx2
					 WHERE adx2.refProfile = pr.keyInstance
					   and adx2.typStatus = 0 /* DATA_ENABLED */
					   and adx2.typCategory = 0 /* PROFILE_ADDRESS */
					   and :datAt BETWEEN adx2.datAvailableFrom and adx2.datAvailableTo
				)
			LEFT JOIN Zip zpx ON zpx.keyZIP = adx.refZip

		WHERE pr.typStatus = 0 /* DATA_ENABLED */
			and pr.typCategory = 1 /* SERVER_PROFILE */
			and :datAt BETWEEN pr.datAvailableFrom and pr.datAvailableTo
			and pr.optAvailability <> 0
			and ( :optAvailability = 0 or bitAnd( pr.optAvailability, :optAvailability) = :optAvailability)
			and pr.strAvailability CONTAINING :strAvailabilityReq
			and ( :keyBranch <= 0 or pr.refBranch = :keyBranch)
			and ( :keyRegion <= 0 or pr.refRegion = :keyRegion)
			and pr.strCode STARTING WITH :strCodeReq
			and pr.strFamilyNameRead STARTING WITH :strFamilyNameRead
			and pr.strFamilyName STARTING WITH :strFamilyName
			and pr.strFirstNameRead STARTING WITH :strFirstNameRead
			and pr.strFirstName STARTING WITH :strFirstName
			and ( :typSex = 2 /* ANY_SEX */ or pr.typSex = :typSex)
			and ( :useAddress = 0 or EXISTS (
				SELECT * FROM Address ad LEFT JOIN Zip zp ON zp.keyZIP = ad.refZip
				WHERE ad.refProfile = pr.keyInstance
				and :datAt BETWEEN ad.datAvailableFrom and ad.datAvailableTo
				and ad.typStatus = 0 /* DATA_ENABLED */
				and ad.typCategory = 0 /* PROFILE_ADDRESS */
				and ( zp.strAddress || ad.strAddress) CONTAINING :strAddressReq
			))
			and ( :useNotice = 0 or EXISTS (
				SELECT * FROM Notice nt
				WHERE nt.refProfile = pr.keyInstance
				and :datAt BETWEEN nt.datAvailableFrom and nt.datAvailableTo
				and nt.typStatus = 0 /* DATA_ENABLED */
				and nt.strTitle CONTAINING :strTitle
				and nt.strContent CONTAINING :strContent
			))

		ORDER BY pr.strFamilyNameRead, pr.strFirstNameRead

		INTO	:keyInstance,
			:strCode,
			:strName,
			:strNameRead,
			:strAvailability,
			:strBranch,
			:strRegion,
			:refZip,
			:strAddress,
			:strAddressRead,
			:strPhoneNumber1,
			:strFaxNumber1
		DO
		BEGIN
			SUSPEND;
		END
	END !!
SET TERM ; !!


/*=========================================================*
  @subject: procedure definition - Locations, Maps
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*/

SET TERM !! ;
RECREATE PROCEDURE Locations ( keyServer  INTEGER, datFrom  DATE, datTo  DATE) RETURNS ( keyAddress  INTEGER) AS
	DECLARE VARIABLE keyAddressFrom  INTEGER;
	DECLARE VARIABLE keyAddressTo  INTEGER;
	BEGIN
		FOR SELECT
			sch.refAddressFrom,
			sch.refAddressTo

		FROM Schedule sch

		WHERE sch.datSchedule BETWEEN :datFrom and :datTo
			and sch.refServer = :keyServer
			and sch.typStatus = 0 /* DATA_ENABLED */

		INTO	:keyAddressFrom,
			:keyAddressTo
		DO
		BEGIN
			keyAddress = keyAddressFrom;
			SUSPEND;

			keyAddress = keyAddressTo;
			SUSPEND;
		END
	END !!
SET TERM ; !!

SET TERM !! ;
RECREATE PROCEDURE Maps ( keyServer  INTEGER, datFrom  DATE, datTo  DATE) RETURNS ( keyPicture  INTEGER) AS
	DECLARE VARIABLE keyAddress  INTEGER;
	BEGIN
		FOR SELECT DISTINCT
			lc.keyAddress

		FROM Locations( :keyServer, :datFrom, :datTo) lc

		WHERE NOT EXISTS (
			SELECT * FROM Schedule sch
			 WHERE sch.datSchedule BETWEEN ( :datFrom - 90) and ( :datFrom - 1)
			   and sch.refServer = :keyServer
			   and ( sch.refAddressFrom = lc.keyAddress or sch.refAddressTo = lc.keyAddress)
			   and sch.typStatus = 0 /* DATA_ENABLED */
			)

		INTO	:keyAddress
		DO
		BEGIN
			FOR SELECT pc.keyInstance

			FROM Picture pc

			WHERE pc.refOwner = :keyAddress
			  and pc.datAvailableFrom <= :datTo
			  and pc.datAvailableTo >= :datFrom

			INTO :keyPicture
			DO
			BEGIN
				SUSPEND;
			END
		END
	END !!
SET TERM ; !!


/*=========================================================*
  @subject: access permission
  @update: 2004/05/08 (Sat) 00:00:00
  - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *=========================================================*/

GRANT EXECUTE ON PROCEDURE Schedules TO User_Client;
GRANT EXECUTE ON PROCEDURE Jobs TO User_Client;

GRANT EXECUTE ON PROCEDURE ClientSchedules TO User_Client;
GRANT EXECUTE ON PROCEDURE ServerSchedules TO User_Client;

GRANT EXECUTE ON PROCEDURE Clients TO User_Client;
GRANT EXECUTE ON PROCEDURE Servers TO User_Client;

GRANT EXECUTE ON PROCEDURE Reserves TO User_Client;

GRANT EXECUTE ON PROCEDURE Paybacks TO User_Client;
GRANT EXECUTE ON PROCEDURE Billings TO User_Client;
GRANT EXECUTE ON PROCEDURE Payments TO User_Client;

GRANT EXECUTE ON PROCEDURE Candidates TO User_Client;

GRANT EXECUTE ON PROCEDURE Locations TO PROCEDURE Maps;
GRANT EXECUTE ON PROCEDURE Maps TO User_Client;


