/* 
 * Copyright 2007 Tatooine Project <http://tatooine.sourceforge.jp/> 
 *  
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *  
 *     http://www.apache.org/licenses/LICENSE-2.0
 *  
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.sf.tatooine.gtx;

import static jp.sf.tatooine.base.XmlConsts.INPUT_BUFFER_SIZE;
import static jp.sf.tatooine.base.XmlConsts.LEXICAL_HANDLER;
import static jp.sf.tatooine.base.XmlConsts.SCHEMA_LANGUAGE;
import static jp.sf.tatooine.base.XmlConsts.XML_SCHEMA;

import java.io.IOException;
import java.io.InputStream;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import jp.sf.tatooine.gtx.node.ELUtils;

/**
 * GtxParser.
 *
 * @author  Tooru Noda
 * @version 1.0 2007/09/12
 * @since   JDK5.0 Tiger
 */
public final class GtxParser {
	
	/** 空白を無視するか否か. */
	private boolean _ignoreSpace = false;
	
	/** コメントを無視するか否か. */
	private boolean _ignoreComment = false;
	
	/** コンテキスト変数. */
	private GtxContext _context;

	public final void setIgnoreComment(boolean ignoreComment) {
		_ignoreComment = ignoreComment;
	}

	public final void setIgnoreSpace(boolean ignoreSpace) {
		_ignoreSpace = ignoreSpace;
	}
	
	/**
	 * コンテキストを設定する.
	 * 
	 * @param context	コンテキスト
	 */
	public final void setContext(GtxContext context) {
		_context = context;
	}
	
	/**
	 * カスタムフィルタをパーサに登録する.
	 * 
	 * @param filter	カスタムフィルタ
	 */
	public final void addFilter(Filter filter) {
		ELUtils.addFilter(filter);
	}

	/**
	 * XMLファイルに記述されたテンプレート命令を処理する.
	 * 
	 * @param stream			入力ストリーム
	 * @return テンプレート
	 * @throws IOException	入出力エラー発生時に送出する
	 * @throws SAXException	XML構文エラー発生時に送出する
	 * @throws GtxSyntaxException	gtx構文エラー発生時に送出する
	 */
	public final GtxTemplate parse(InputStream stream) 
			throws IOException, SAXException, GtxSyntaxException {
		
		GtxTemplateHandler handler = new GtxTemplateHandler();
		handler.setIgnoreSpace(_ignoreSpace);
		handler.setIgnoreComment(_ignoreComment);
		
		SAXParserFactory factory = SAXParserFactory.newInstance();
		factory.setNamespaceAware(true);
		try {
			SAXParser parser = factory.newSAXParser();
			parser.setProperty(SCHEMA_LANGUAGE, XML_SCHEMA);
			parser.setProperty(LEXICAL_HANDLER, handler);
			parser.setProperty(INPUT_BUFFER_SIZE, Integer.valueOf(10000));
			parser.parse(new InputSource(stream), handler);
			
			GtxTemplate template = handler.getGtxTemplate();
			if (_context != null) {
				template.setContext(_context);
			}
			template.setIgnoreSpace(_ignoreSpace);
			return template;
		}
		catch(SAXException e) {
			e.printStackTrace();
			throw new IOException(e.getLocalizedMessage());
		}
		catch(ParserConfigurationException e) {
			e.printStackTrace();
			throw new IOException(e.getLocalizedMessage());
		}
	}

}
