#ifndef _TMQUATERNION_HPP
#define _TMQUATERNION_HPP

/** 
 *  @file quaternion.hpp
 *  @brief Defines tempest::quaternion<T,Sz>.
 *  @author ototoi / Toru Matsuoka
 *  @date 2004/05/14 
 */
#include<cstddef>
#include<cmath>
#include<stdexcept>

#include <complex>

#include<ostream>
#include<sstream>

#include"vector_base.hpp"

namespace tempest{	

/**
 *  @class quaternion
 *  @brief Quaternion class template.
 *  @todo   
 *  @code
 *  tempest::quaternion<double> q;  
 *  @endcode
 *  @bug    
 *
 */
	
template<class T>
class quaternion;

template<class T>
class quaternion:public vector_base< quaternion<T>, T, 4>{
public:
	//-----------------------------------------------
    //type defines
	typedef		T										value_type;
	typedef		T&										reference;							  
	typedef		const T&								const_reference;
	typedef		quaternion<T>							this_type;
	
	typedef		T										scalar_type;
	typedef		vector<T,3>								vector_type;
	
	typedef		std::size_t								size_type;
	typedef		std::ptrdiff_t							difference_type;
	
	typedef		T*										pointer;
	typedef		const T*								const_pointer;
	
	typedef		pointer									iterator;
	typedef		const_pointer							const_iterator;
	
	
public:
	static const size_type c_size = 4;//container size
	
	//-----------------------------------------------
    //functions for iterator
	iterator		begin()			{return element;}
	iterator		end()			{return element+c_size;}
	const_iterator	begin()	const	{return element;}
	const_iterator	end()	const	{return element+c_size;}
	
	//-----------------------------------------------
    //constructors and destructor
	quaternion(){}
	
	quaternion(value_type a,value_type b,value_type c,value_type d):m0(a),m1(b),m2(c),m3(d){}
	
	quaternion(const quaternion<T>& rhs):m0(rhs.m0),m1(rhs.m1),m2(rhs.m2),m3(rhs.m3){}
	
	template<class X>
	explicit quaternion(const X* rhs):m0(rhs[0]),m1(rhs[1]),m2(rhs[2]),m3(rhs[3]){}//X const (&rhs)[c_size]
	
	template<class X>
    explicit quaternion(const quaternion<X> &rhs):m0(rhs[0]),m1(rhs[1]),m2(rhs[2]),m3(rhs[3]){}	
	
	explicit quaternion(
		std::complex<T> const & z0,
		std::complex<T> const & z1 = std::complex<T>()
	):
		m0(z0.real()),
		m1(z0.imag()),
		m2(z1.real()),
		m3(z1.imag()){} // nothing to do!
	
	
	
	template<class Self,class X>
	explicit quaternion(const vector_base<Self,X,4> & rhs):
		m0(static_cast<T>(static_cast<const Self &>(rhs)[0])),
		m1(static_cast<T>(static_cast<const Self &>(rhs)[1])),
		m2(static_cast<T>(static_cast<const Self &>(rhs)[2])),
		m3(static_cast<T>(static_cast<const Self &>(rhs)[3])){}
	
	//-----------------------------------------------
	//inserters
	this_type& operator= (const this_type &rhs){
		m0 = rhs.m0 ;
    	m1 = rhs.m1 ;
    	m2 = rhs.m2 ;
		m3 = rhs.m3 ;
    	return *this;
	}

    template<class X>
    this_type& operator= (const quaternion<X> &rhs){
		m0 = rhs[0];
		m1 = rhs[1];
		m2 = rhs[2];
		m3 = rhs[3];
    	return *this;
	}
	
	template<class Self,class X>
	this_type& operator= (const vector_base<Self,X,4> & rhs){
		m0 = static_cast<T>(static_cast<const Self &>(rhs)[0]);
    	m1 = static_cast<T>(static_cast<const Self &>(rhs)[1]);
    	m2 = static_cast<T>(static_cast<const Self &>(rhs)[2]);
		m3 = static_cast<T>(static_cast<const Self &>(rhs)[3]);
		return *this;
	}
	
	template<class IT>
	void assign( IT start, IT end ){
		assert(std::distance(start,end)<=c_size);//debug
		
		std::copy(start,end,begin());				
	}
	
	void assign( size_type num, value_type val ){
		std::fill_n(begin(),(num<c_size)?num:c_size,val);		
	}
	
	//-----------------------------------------------
	//capacity
	size_type size ()     const { return c_size; }
    size_type max_size () const { return c_size; }
    bool      empty ()    const { return false;	 }
	
	//-----------------------------------------------
	//operators
	this_type& negate(){
		m0 = -m0;
		m1 = -m1;
		m2 = -m2;
		m3 = -m3;
		
		return *this;
	}
	
#define DECLARE_OP_EQUAL_X(OP)							\
	template<class X>									\
	this_type& operator OP (const quaternion<X> & rhs){	\
		T a = static_cast<T>(rhs[0]);					\
		T b = static_cast<T>(rhs[0]);					\
		T c = static_cast<T>(rhs[0]);					\
		T d = static_cast<T>(rhs[0]);					\
		m0 OP a;										\
		m1 OP b;										\
		m2 OP c;										\
		m3 OP d;										\
		return *this;									\
	}													\
	template<class X>									\
	this_type& operator OP (const std::complex<X> &rhs){\
		T at = static_cast<T>(rhs.real());				\
		T bt = static_cast<T>(rhs.imag());				\
														\
		m0 OP at;										\
		m1 OP bt;										\
		return *this;									\
	}
	
//----use
	DECLARE_OP_EQUAL_X(+=)
	DECLARE_OP_EQUAL_X(-=)
//
	
	
	
	
#undef DECLARE_OP_EQUAL
	
	template<class X>
	this_type& operator*= (std::complex<X> const & rhs){
		T ar = rhs.real();
		T br = rhs.imag();

		T at =  (+m0*ar-m1*br);
		T bt =  (+m0*br+m1*ar);
		T ct =  (+m2*ar+m3*br);
		T dt =  (-m2*br+m3*ar);

		m0 = at;
		m1 = bt;
		m2 = ct;
		m3 = dt;

		return(*this);
	}
	
	template<class X>
	this_type& operator/= (std::complex<X> const & rhs){
		T ar = rhs.real();
		T br = rhs.imag();
		
		T denominator = T(1)/(ar*ar+br*br);  
		
		T    at =  (+m0*ar+m1*br)*denominator;//++
		T    bt =  (-m0*br+m1*ar)*denominator;//-+
		T    ct =  (+m2*ar-m3*br)*denominator;//+-
		T    dt =  (+m2*br+m3*ar)*denominator;//++

		m0 = at;
		m1 = bt;
		m2 = ct;
		m3 = dt;

		return(*this);
	}
		
	template<class X>
	this_type& operator*= (const quaternion<X> & rhs){
		T ar = static_cast<T>(rhs[0]);
		T br = static_cast<T>(rhs[1]);
		T cr = static_cast<T>(rhs[2]);
		T dr = static_cast<T>(rhs[3]);
		
		T at = +m0*ar-m1*br-m2*cr-m3*dr;
		T bt = +m0*br+m1*ar+m2*dr-m3*cr;	   //(m0*br+ar*b)+(m2*dr-cr*d);
		T ct = +m0*cr-m1*dr+m2*ar+m3*br;	   //(m0*cr+ar*c)+(m3*br-dr*b);
		T dt = +m0*dr+m1*cr-m2*br+m3*ar;	   //(m0*dr+ar*d)+(m1*cr-br*c);

		m0 = at;
		m1 = bt;
		m2 = ct;
		m3 = dt;
		return *this;
	}
	
	template<class X>
	this_type& operator/= (const quaternion<X> & rhs){
		T ar = static_cast<T>(rhs[0]);
		T br = static_cast<T>(rhs[1]);
		T cr = static_cast<T>(rhs[2]);
		T dr = static_cast<T>(rhs[3]);
		
		T denominator = T(1)/(ar*ar + br*br + cr*cr + dr*dr);
		
		T at = (+m0*ar+m1*br+m2*cr+m3*dr)*denominator;	 //(m0*ar+m1*br+m2*cr+m3*dr)/denominator;
		T bt = (-m0*br+m1*ar-m2*dr+m3*cr)*denominator;	 //((ar*b-m0*br)+(cr*d-m2*dr))/denominator;
		T ct = (-m0*cr+m1*dr+m2*ar-m3*br)*denominator;	 //((ar*c-m0*cr)+(dr*b-m3*br))/denominator;
		T dt = (-m0*dr-m1*cr+m2*br+m3*ar)*denominator;	 //((ar*d-m0*dr)+(br*c-m1*cr))/denominator;
				
		m0 = at;
		m1 = bt;
		m2 = ct;
		m3 = dt;
		
		return *this;
	}
/*
	template<class X>
	this_type& operator*= (const X rhs){
		m0 = static_cast<T>(m0 * rhs);
		m1 = static_cast<T>(m1 * rhs);
		m2 = static_cast<T>(m2 * rhs);
		m3 = static_cast<T>(m3 * rhs);
		return *this;
	}
	
	template<class X>
	this_type& operator/= (const X rhs){
		m0 = static_cast<T>(m0 / rhs);
		m1 = static_cast<T>(m1 / rhs);
		m2 = static_cast<T>(m2 / rhs);
		m3 = static_cast<T>(m3 / rhs);
		return *this;
	}
*/
	
// for T 	
#define DECLARE_OP_EQUAL(OP)							\
	this_type& operator OP (const this_type & rhs){		\
		m0 OP rhs[0];									\
		m1 OP rhs[1];									\
		m2 OP rhs[2];									\
		m3 OP rhs[3];									\
		return *this;									\
	}													\
	this_type& operator OP (const std::complex<T> &rhs){\
		m0 OP rhs.real();								\
		m1 OP rhs.imag();								\
		return *this;									\
	}
	
//----use
	DECLARE_OP_EQUAL(+=)
	DECLARE_OP_EQUAL(-=)
//
	
#undef DECLARE_OP_EQUAL

	this_type& operator*= (const std::complex<T> & rhs){
		T ar = rhs.real();
		T br = rhs.imag();

		m0 =  (+m0*ar-m1*br);
		m1 =  (+m0*br+m1*ar);
		m2 =  (+m2*ar+m3*br);
		m3 =  (-m2*br+m3*ar);

		return(*this);
	}
	
	this_type& operator/= (const std::complex<T> & rhs){
		T ar = rhs.real();
		T br = rhs.imag();
		
		T denominator = T(1)/(ar*ar+br*br);  
		
		m0 =  (+m0*ar+m1*br)*denominator;//++
		m1 =  (-m0*br+m1*ar)*denominator;//-+
		m2 =  (+m2*ar-m3*br)*denominator;//+-
		m3 =  (+m2*br+m3*ar)*denominator;//++

		return(*this);
	}
	
#define ar rhs[0]
#define br rhs[1]
#define cr rhs[2]
#define dr rhs[3]
	
	this_type& operator*= (const this_type & rhs){
		T t0 = m0;
		T t1 = m1;
		T t2 = m2;
		
		m0 = +t0*ar-t1*br-t2*cr-m3*dr;
		m1 = +t0*br+t1*ar+t2*dr-m3*cr;	   //(m0*br+ar*b)+(m2*dr-cr*d);
		m2 = +t0*cr-t1*dr+t2*ar+m3*br;	   //(m0*cr+ar*c)+(m3*br-dr*b);
		m3 = +t0*dr+t1*cr-t2*br+m3*ar;	   //(m0*dr+ar*d)+(m1*cr-br*c);

		return *this;
	}
	
	template<class X>
	this_type& operator/= (const this_type & rhs){
		T t0 = m0;
		T t1 = m1;
		T t2 = m2;
		
		T denominator = T(1)/(ar*ar + br*br + cr*cr + dr*dr);
		
		m0 = (+t0*ar+t1*br+t2*cr+m3*dr)*denominator;	 //(m0*ar+m1*br+m2*cr+m3*dr)/denominator;
		m1 = (-t0*br+t1*ar-t2*dr+m3*cr)*denominator;	 //((ar*b-m0*br)+(cr*d-m2*dr))/denominator;
		m2 = (-t0*cr+t1*dr+t2*ar-m3*br)*denominator;	 //((ar*c-m0*cr)+(dr*b-m3*br))/denominator;
		m3 = (-t0*dr-t1*cr+t2*br+m3*ar)*denominator;	 //((ar*d-m0*dr)+(br*c-m1*cr))/denominator;
		
		return *this;
	}
	
#undef ar
#undef br
#undef cr
#undef dr
	
	
	this_type& operator*= (value_type rhs){
		m0 *= rhs;
		m1 *= rhs;
		m2 *= rhs;
		m3 *= rhs;
		return *this;
	}
	
	this_type& operator/= (value_type rhs){
		m0 /= rhs;
		m1 /= rhs;
		m2 /= rhs;
		m3 /= rhs;
		return *this;
	}
	
	
	//-----------------------------------------------
	T & operator[](size_type i){
    	return element[i];
	}
	
	value_type operator[](size_type i) const {
    	return element[i];
	}
	
	reference at(size_type i){
		if(c_size<=i){throw std::out_of_range("tempest::quaternion");}
		return element[i];
	}
	const_reference at(size_type i) const {
		if(c_size<=i){throw std::out_of_range("tempest::quaternion");}
		return element[i];
	}
	
	//-----------------------------------------------
	//utilities
	
	T sqr_length()const{
		return m0*m0 + m1*m1 + m2*m2 + m3*m3;
	}
	T length()const{
		T temp = sqr_length();
		return std::sqrt(temp);
	}
	T dot(const quaternion<T>& rhs)const{
		return m0*rhs[0] + m1*rhs[1] + m2*rhs[2] + m3*rhs[3];
	}
	
	/*
	 * euc_
	 *
	 */
	quaternion<T>& normalize(){
		T length = sqr_length();//||q||^2
		//if (length == T()) return *this;

		length = RSQRT(length);	// 1 / ||q||
		
		m0 *= length;
		m1 *= length;
		m2 *= length;
		m3 *= length;
		
		return *this;
	}
private:
	
	union{
		struct{
			T m0,m1,m2,m3;
		};
		T element[4];
	};

};

/**
 * @relates quaternion
 */
//@{

template<class T>
inline quaternion<T> operator+ (const quaternion<T> & rhs){
	return rhs;
}
	
template<class T>
inline quaternion<T> operator- (const quaternion<T> & rhs){
	return quaternion<T>(rhs).negate();
}
//-----------------------------------------------
//binary operators

	
template<class T> 
inline quaternion<T> operator+ (const quaternion<T> &lhs, const quaternion<T> &rhs){
	return quaternion<T>(lhs) += rhs;
}
template<class T> 
inline quaternion<T> operator- (const quaternion<T> &lhs, const quaternion<T> &rhs){
	return quaternion<T>(lhs) -= rhs;
}


template<class T> 
inline quaternion<T> operator* (T lhs,const quaternion<T> & rhs){ 
	return quaternion<T>(rhs) *= lhs ; 
}

template<class T> 
inline quaternion<T> operator* (const quaternion<T> &lhs,T rhs){ 
	return quaternion<T>(lhs) *= rhs ;
}	
	
template<class T> 
inline quaternion<T> operator/ (const quaternion<T> &lhs,T rhs){ 
	return quaternion<T>(lhs) /= rhs ;
}
	
template<class T>
inline  quaternion<T> operator* (const quaternion<T>& lhs,const quaternion<T>& rhs){
	/*
	return quaternion<T>(
		lw*rx + lx*rw + ly*rz - lz*ry,//++-
		lw*ry - lx*rz + ly*rw + lz*rx,//-++
		lw*rz + lx*ry - ly*rx + lz*rw,//+-+
		lw*rw - lx*rx - ly*ry - lz*rz//---
	);
	*/
	
	//lx*rw + lw*rx - lz*ry + ly*rz ,
	//
	//
	//
	return quaternion<T>(
		lhs[0]*rhs[0] - lhs[1]*rhs[1] - lhs[2]*rhs[2] - lhs[3]*rhs[3],
		lhs[0]*rhs[1] + lhs[1]*rhs[0] + lhs[2]*rhs[3] - lhs[3]*rhs[2],
		lhs[0]*rhs[2] - lhs[1]*rhs[3] + lhs[2]*rhs[0] + lhs[3]*rhs[1],
		lhs[0]*rhs[3] + lhs[1]*rhs[2] - lhs[2]*rhs[1] + lhs[3]*rhs[0]
	);
}
	

	
//-----------------------------------------------	
// utility functions

template<class T,std::size_t Sz>
inline quaternion<T> normalize(const quaternion<T> &rhs){
    return quaternion<T>(rhs).normalize();
}

template<class T,std::size_t Sz>
inline T length(const quaternion<T> &rhs){
	return rhs.length();
}

template<class T,std::size_t Sz>
inline T sqr_length(const quaternion<T> &rhs){
	return rhs.sqr_length();
}
template<class T>
inline T dot(const quaternion<T>& lhs,const quaternion<T>& rhs){
	return lhs.dot(rhs);
}

template<class T> 
inline  quaternion<T> operator~(const quaternion<T>& rhs){
	T l = rhs.sqr_length();//lq|*|q|
	//if (l == T()) return rhs;
	l = T(1)/l;
	
	return quaternion<T>(rhs[0]*l,-rhs[1]*l,-rhs[2]*l,-rhs[3]*l); 
}
	
template<class T>
inline quaternion<T> conj(const quaternion<T>& rhs){
	return quaternion<T>(rhs[0],-rhs[1],-rhs[2],-rhs[3]);
}
	
template<class T>
inline T norm(const quaternion<T> & rhs){
	return (rhs * conj(rhs))[0];		
}
	
template<class T>
inline quaternion<T> lerp(const quaternion<T> &lhs,const quaternion<T> &rhs, T t){
	return ((1-t)*lhs + t*rhs).normalize();
}
	
template<class T>
inline quaternion<T> slerp(const quaternion<T> &lhs,const quaternion<T> &rhs, T t){
	
	T theta = std::acos( dot(lhs,rhs) );//
	T s = static_cast<T>(T(1)/std::sin(theta));
	
	return (std::sin((1-t)*theta)*lhs + std::sin(t*theta)*rhs)*s;		
}
	
	/*
	template<class X,class Y>
	quaternion(X s, const vector<Y,3> & v):m0(s),m1(v[0]),m2(v[1]),m3(v[2]){}//scalar , vector
	*/
template<class Self, class T>
quaternion<T> scalar_vector(T s,const vector_base<Self,T,3> & bv){//scalar , vector
	const Self & v = static_cast<const Self &>(bv);
	return quaternion<T>(s,v[0],v[1],v[2]);
}
	
template<class Self,class T>
quaternion<T> pivot_angle(const vector_base<Self,T,3> & bv, T theta){//pivot , theta
	T s;
	const Self & v = static_cast<const Self &>(bv);
	
	theta /= 2;
	s = std::sin(theta);
	return quaternion<T>(std::cos(theta),s*v[0],s*v[1],s*v[2]);
}

template<typename T, typename _CharT, class _Traits>
std::basic_ostream<_CharT, _Traits>& operator<<(std::basic_ostream<_CharT, _Traits>& os, const quaternion<T>& rhs){
	
	std::basic_ostringstream<_CharT, _Traits> s;
	s.flags(os.flags());
	s.imbue(os.getloc());
	s.precision(os.precision());
	s << "(" << rhs[0] << "," << rhs[1] << "," << rhs[2] << rhs[3] <<")";
	
	return os << s.str();
}
	
//@}
	
	
//-----------------------------------------------
	//template<>    class quaternion<float>;
	//template<double>    class quaternion<double>;
	//template<long double>    class quaternion<long double>;

}//

#endif

