/*
 * Copyright 2009- kensir0u.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
/*
 * 쐬 (creation date)  F2009/02/14
 * pbP[W  (package name) Fnet.sf.thirdi.jdbc.impl
 * t@C  (file name)    FCSVConverterImpl.java
 */
package net.sf.thirdi.jdbc.impl;

import java.io.Closeable;
import java.io.IOException;
import java.io.StringWriter;
import java.io.Writer;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.util.Map;

import net.sf.thirdi.jdbc.CSVConverter;
import net.sf.thirdi.jdbc.CSVFormatMode;
import net.sf.thirdi.jdbc.CSVInfo;
import net.sf.thirdi.jdbc.MappingLetterType;
import net.sf.thirdi.jdbc.exception.IORuntimeException;
import net.sf.thirdi.jdbc.exception.SQLRuntimeException;
import net.sf.thirdi.jdbc.type.Type;
import net.sf.thirdi.jdbc.typeresolver.SQLTypeManager;

/**
 * <i>Tv(abstract)</i>F CSVConverterNX
 *@.<p>
 * CSVConverterNX
 * 
 *
 * @author  kensir0u
 * @version 1.0
 * @since   JDK 5.0
 *
 */
public class CSVConverterImpl implements CSVConverter{
	
	private static Map<Integer, Type> types = SQLTypeManager.getTypes();
	
	private static final boolean DEFAULT_ADDHEADERCOLUMN = true;
	
	private static final boolean DEFAULT_REMOVERETURN = false;
	
	private static final boolean DEFAULT_ESCAPEDOUBLEQUOTATION = false;
	
	private static final MappingLetterType DEFAULT_TYPE = MappingLetterType.ORIGINAL_LETTER;
	
	private static final CSVFormatMode DEFAULT_MODE = CSVFormatMode.ORIGINAL;
	
	private static final char DEFAULT_FIELDSEPARATORCHAR = ',';
	
	private static final String DEFAULT_RECORDSEPARATORCHAR = "\r\n";
	
	private static final Long DEFAULT_MAXCOUNT = Long.MAX_VALUE;
	
	private static final CSVInfo csvinfo = getDefaultCSVInfo();
	
	private static CSVInfo getDefaultCSVInfo(){
		
		CSVInfoImpl csvinfoimpl = new CSVInfoImpl();
		csvinfoimpl.setAddheaderColumn(DEFAULT_ADDHEADERCOLUMN);
		csvinfoimpl.setEscapeDoubleQuotation(DEFAULT_ESCAPEDOUBLEQUOTATION);
		csvinfoimpl.setFieldSeparatorChar(DEFAULT_FIELDSEPARATORCHAR);
		csvinfoimpl.setRemoveReturnMode(DEFAULT_REMOVERETURN);
		csvinfoimpl.setRecordSeparatorChar(DEFAULT_RECORDSEPARATORCHAR);
		csvinfoimpl.setMappingLetterType(DEFAULT_TYPE);
		csvinfoimpl.setFormtMode(DEFAULT_MODE);
		return csvinfoimpl;
	}
	/* (non-Javadoc)
	 * @see net.sf.thirdi.jdbc.CSVConverter#toCSV(java.sql.ResultSet, net.sf.thirdi.jdbc.CSVInfo, long)
	 */
	@Override
	public Writer toCSV(ResultSet rs, CSVInfo info, long maxcount) {
		Writer writer = null;
		try {
			if (info == null) {
				info = csvinfo;
			} else {
				info = arrangesMode(info);
			}
			long count = 0;
			writer = getDefaultWriter();
			
			if (info.addHeaderColumn()){
				writeColumnHeader(rs,writer,info);
			}
			
			while(rs.next()){
				if (count > maxcount) break;
				write(rs,writer,info);
				count++;
			}
			
			finish(writer);
			
			return writer;
		} catch (SQLException e) {
			throw new SQLRuntimeException(e);
		} catch (IOException e) {
			throw new IORuntimeException(e);
		} finally{
			close(writer);
		}
		
	}
	
	private static CSVInfo arrangesMode(CSVInfo info){
		
		switch(info.getFormatMode()){
		case RFC4108:
			info.setEscapeDoubleQuotation(true);
			info.setFieldSeparatorChar(DEFAULT_FIELDSEPARATORCHAR);
			info.setRemoveReturnMode(DEFAULT_REMOVERETURN);
			info.setRecordSeparatorChar(DEFAULT_RECORDSEPARATORCHAR);
			break;
		case ORIGINAL:
			info.setEscapeDoubleQuotation(DEFAULT_ESCAPEDOUBLEQUOTATION);
			info.setFieldSeparatorChar(DEFAULT_FIELDSEPARATORCHAR);
			info.setRemoveReturnMode(DEFAULT_REMOVERETURN);
			info.setRecordSeparatorChar(DEFAULT_RECORDSEPARATORCHAR);
			break;
		default:
			break;
		}
		
		return info;
		
	}
	
	private void finish(Writer writer) throws IOException{
		writer.flush();
		writer.close();
	}
	
	private void close(Closeable writer){
		if (writer != null) {
			try {
				writer.close();
			} catch (IOException ignore) {}
		}
	}
	
	private void writeColumnHeader(ResultSet rs,Writer writer,CSVInfo info) {

		try {
			ResultSetMetaData rsmd = rs.getMetaData();
			int count = rsmd.getColumnCount();
			
			StringBuffer sb = new StringBuffer();
			for (int i = 0; i < count; i++) {

				String columnName = rsmd.getColumnLabel(i + 1);
				switch(info.getMappingLetterType()){
					case CAPITAL_LETTER:
						columnName = columnName.toUpperCase();
						break;
					case SMALL_LETTER:
						columnName = columnName.toLowerCase();
						break;
					default:
				}
				if (i == count - 1 ) {
					sb.append(columnName).append(info.getRecordSeparator());
				} else {
					sb.append(columnName).append(info.getFieldSeparator());
				}
			}
			
			writer.write(sb.toString());
			
		} catch (SQLException e) {
			throw new SQLRuntimeException(e);
		} catch (IOException e) {
			throw new IORuntimeException(e);
		}
	}
	
	private static String nullToBlank(String value){
		if (value == null) return "";
		return value;
	}
	
	private static String removeReturnChar(String value){
		if (value == null || "".equals(value)) return value;
		char[] c = value.toCharArray();
		StringBuffer sb = new StringBuffer();
		for (int i = 0; i < c.length; i++) {
			char d = c[i];
			if ('\n' != d && '\r' != d){
				sb.append(d);
			}
		}
		return sb.toString();
		
	}
	private static String formatCSV(String value){
		if (value == null || "".equals(value)) return value;
		if (isFormatCSVChar(value)){
			return new StringBuffer()
			.append('"')
			.append(value)
			.append('"').toString();
		} else {
			return value;
		}		
	}
	private static boolean isFormatCSVChar(String value){
		if (value == null || "".equals(value)) return false;
		char[] c = value.toCharArray();
		for (int i = 0; i < c.length; i++) {
			char d = c[i];
			if ('\n' == d || '\r' == d || ',' ==d || '"' == d){
				return true;
			}
		}
		return false;
	}
	
	private static String escapeDoubleQuotation(String value){
		if (value == null || "".equals(value)) return value;
		char[] c = value.toCharArray();
		StringBuffer sb = new StringBuffer();
		for (int i = 0; i < c.length; i++) {
			char d = c[i];
			if ('"' != d ){
				sb.append(d).append('"');
			} else {
				sb.append(d);
			}
		}
		return sb.toString();
	}
	
	private void write(ResultSet rs,Writer writer,CSVInfo info){
		
		try {			
			ResultSetMetaData rsmd = rs.getMetaData();

			int count = rsmd.getColumnCount();
			
			if (rs.isBeforeFirst())
				rs.next();
			
			StringBuffer sb = new StringBuffer();
			for (int i = 0; i < count; i++) {

				//String columnName = rsmd.getColumnLabel(i + 1);
				
				Type type = types.get(rsmd.getColumnType(i + 1));

				if (type != null) {
					String o = type.getResultsetDataToString(rs,i+1);
					String value = nullToBlank(o);
					if (info.isEscapeDoubleQuotation()){
						value = escapeDoubleQuotation(value);
					}
					if (info.getFormatMode() == CSVFormatMode.RFC4108){
						value = formatCSV(value);
					} else if (info.getFormatMode() == CSVFormatMode.MANUAL){
						if (info.isRemoveReturn()){
							value = removeReturnChar(value);
						}
					}

					if (i == count - 1 ) {
						sb.append(value).append(info.getRecordSeparator());
					} else {
						sb.append(value).append(info.getFieldSeparator());
					}	
				}
			}

			writer.write(sb.toString());
			
		} catch (SQLException e) {
			throw new SQLRuntimeException(e);
		} catch (IOException e) {
			throw new IORuntimeException(e);
		}
	}	
	
	private static Writer getDefaultWriter(){
		return new StringWriter();
	}
	

	/* (non-Javadoc)
	 * @see net.sf.thirdi.jdbc.CSVConverter#toCSV(java.sql.ResultSet, net.sf.thirdi.jdbc.CSVInfo)
	 */
	@Override
	public Writer toCSV(ResultSet rs, CSVInfo info) {
		return toCSV(rs, info, DEFAULT_MAXCOUNT);
	}

	/* (non-Javadoc)
	 * @see net.sf.thirdi.jdbc.CSVConverter#toCSV(java.sql.ResultSet)
	 */
	@Override
	public Writer toCSV(ResultSet rs) {
		return toCSV(rs, null, DEFAULT_MAXCOUNT);
	}

	/* (non-Javadoc)
	 * @see net.sf.thirdi.jdbc.CSVConverter#toCSV(java.sql.ResultSet, net.sf.thirdi.jdbc.CSVInfo, long, java.io.Writer)
	 */
	@Override
	public <T extends Writer> T toCSV(ResultSet rs, CSVInfo info,
			long maxcount, T writer) {
		try {
			if (writer == null)
				//TODO message move to property file.
				throw new IllegalArgumentException("writer must not be null.");
			if (info == null) {
				info = csvinfo;
			} else {
				info = arrangesMode(csvinfo);
			}
			
			long count = 0;

			if (info.addHeaderColumn()){
				writeColumnHeader(rs,writer,info);
			}
			
			while(rs.next()){
				if (count > maxcount) break;
				write(rs,writer,info);
				count++;
			}
			
			finish(writer);
			
			return writer;
			
		} catch (SQLException e) {
			throw new SQLRuntimeException(e);
		} catch (IOException e) {
			throw new IORuntimeException(e);
		} finally {
			close(writer);
		}
	}

	/* (non-Javadoc)
	 * @see net.sf.thirdi.jdbc.CSVConverter#toCSV(java.sql.ResultSet, net.sf.thirdi.jdbc.CSVInfo, java.io.Writer)
	 */
	@Override
	public <T extends Writer> T toCSV(ResultSet rs, CSVInfo info, T writer) {
		return toCSV(rs, info, DEFAULT_MAXCOUNT, writer);
	}

	/* (non-Javadoc)
	 * @see net.sf.thirdi.jdbc.CSVConverter#toCSV(java.sql.ResultSet, java.io.Writer)
	 */
	@Override
	public <T extends Writer> T toCSV(ResultSet rs, T writer) {
		return toCSV(rs, null, DEFAULT_MAXCOUNT, writer);
	}
	/* (non-Javadoc)
	 * @see net.sf.thirdi.jdbc.CSVConverter#createCSVInfo()
	 */
	@Override
	public CSVInfo createCSVInfo() {
		return new CSVInfoImpl();
	}

}
