/*
 * Copyright 2009- kensir0u.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
/*
 * 쐬 (creation date)  F2009/02/14
 * pbP[W  (package name) Fnet.sf.thirdi.jdbc.impl
 * t@C  (file name)    FCSVConverterImpl.java
 */
package net.sf.thirdi.jdbc.impl;

import java.io.Closeable;
import java.io.IOException;
import java.io.StringWriter;
import java.io.Writer;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.util.Map;

import net.sf.thirdi.jdbc.CSVColumn;
import net.sf.thirdi.jdbc.CSVConverter;
import net.sf.thirdi.jdbc.CSVFormatInfo;
import net.sf.thirdi.jdbc.CSVFormatMode;
import net.sf.thirdi.jdbc.CSVHeaderInfo;
import net.sf.thirdi.jdbc.CSVInfo;
import net.sf.thirdi.jdbc.exception.IORuntimeException;
import net.sf.thirdi.jdbc.exception.SQLRuntimeException;
import net.sf.thirdi.jdbc.type.Type;
import net.sf.thirdi.jdbc.typeresolver.SQLTypeManager;

/**
 * <i>Tv(abstract)</i>F CSVConverterNX @.
 * <p>
 * CSVConverterNX
 * 
 * 
 * @author kensir0u
 * @version 1.0
 * @since JDK 5.0
 * 
 */
public class CSVConverterImpl implements CSVConverter {

	private static final boolean DEFAULT_ADDHEADERCOLUMN = true;

	private static final boolean DEFAULT_ESCAPEDOUBLEQUOTATION = false;

	private static final char DEFAULT_FIELDSEPARATORCHAR = ',';

	private static final Long DEFAULT_MAXCOUNT = Long.MAX_VALUE;

	private static final CSVFormatMode DEFAULT_MODE = CSVFormatMode.ORIGINAL;

	private static final String DEFAULT_RECORDSEPARATORCHAR = "\r\n";

	private static final boolean DEFAULT_REMOVERETURN = false;

	private static final char ESCAPE_CHAR = '"';

	private static Map<Integer, Type> types = SQLTypeManager.getTypes();

	private static CSVInfo arrangesMode(CSVInfo info) {

		switch (info.getFormatMode()) {
		case RFC4180:
			info.getCSVFormatInfo().setEscapeDoubleQuotation(true);
			info.getCSVFormatInfo().setFieldSeparatorChar(
					DEFAULT_FIELDSEPARATORCHAR);
			info.getCSVFormatInfo().setRemoveReturnMode(DEFAULT_REMOVERETURN);
			info.getCSVFormatInfo().setRecordSeparatorChar(
					DEFAULT_RECORDSEPARATORCHAR);
			break;
		case ORIGINAL:
			info.getCSVFormatInfo().setEscapeDoubleQuotation(
					DEFAULT_ESCAPEDOUBLEQUOTATION);
			info.getCSVFormatInfo().setFieldSeparatorChar(
					DEFAULT_FIELDSEPARATORCHAR);
			info.getCSVFormatInfo().setRemoveReturnMode(DEFAULT_REMOVERETURN);
			info.getCSVFormatInfo().setRecordSeparatorChar(
					DEFAULT_RECORDSEPARATORCHAR);
			break;
		default:
			break;
		}

		return info;
	}

	private static CSVColumn createCSVColumn(String columnName) {
		return new CSVColumnImpl(columnName);
	}

	private static String escapeDoubleQuotation(String value) {
		if (value == null || "".equals(value))
			return value;
		char[] c = value.toCharArray();
		StringBuffer sb = new StringBuffer();
		for (int i = 0; i < c.length; i++) {
			char d = c[i];
			if (ESCAPE_CHAR == d) {
				sb.append(d).append(ESCAPE_CHAR);
			} else {
				sb.append(d);
			}
		}
		return sb.toString();
	}

	private static String formatCSV(String value) {
		if (value == null || "".equals(value))
			return value;
		if (isFormatCSVChar(value)) {
			return new StringBuffer().append(ESCAPE_CHAR).append(value).append(
					ESCAPE_CHAR).toString();
		} else {
			return value;
		}
	}

	private static CSVFormatInfo getDefaultCSVFormatInfo() {
		return new CSVFormatInfoImpl();
	}

	private static CSVHeaderInfo getDefaultCSVHeaderInfo(ResultSet rs) {

		try {
			ResultSetMetaData rsmd = rs.getMetaData();
			int count = rsmd.getColumnCount();

			CSVHeaderInfoImpl csvheaderinfoimpl = new CSVHeaderInfoImpl();

			for (int i = 0; i < count; i++) {
				String columnName = rsmd.getColumnLabel(i + 1);
				csvheaderinfoimpl.setColumn(columnName,
						createCSVColumn(columnName));
			}

			return csvheaderinfoimpl;

		} catch (SQLException e) {
			throw new SQLRuntimeException(e);
		}
	}

	private static CSVInfo getDefaultCSVInfo(ResultSet rs) {
		CSVInfoImpl csvinfoimpl = new CSVInfoImpl();
		csvinfoimpl.setAddHeaderColumn(DEFAULT_ADDHEADERCOLUMN);
		csvinfoimpl.setFormtMode(DEFAULT_MODE);
		csvinfoimpl.setCSVFormatInfo(getDefaultCSVFormatInfo());
		csvinfoimpl.setCSVHeaderInfo(getDefaultCSVHeaderInfo(rs));
		return csvinfoimpl;
	}

	private static Writer getDefaultWriter() {
		return new StringWriter();
	}

	private static boolean isFormatCSVChar(String value) {
		if (value == null || "".equals(value))
			return false;
		char[] c = value.toCharArray();
		for (int i = 0; i < c.length; i++) {
			char d = c[i];
			if ('\n' == d || '\r' == d || DEFAULT_FIELDSEPARATORCHAR == d
					|| ESCAPE_CHAR == d) {
				return true;
			}
		}
		return false;
	}

	private static String nullToBlank(String value) {
		if (value == null)
			return "";
		return value;
	}

	private static String removeReturnChar(String value) {
		if (value == null || "".equals(value))
			return value;
		char[] c = value.toCharArray();
		StringBuffer sb = new StringBuffer();
		for (int i = 0; i < c.length; i++) {
			char d = c[i];
			if ('\n' != d && '\r' != d) {
				sb.append(d);
			}
		}
		return sb.toString();

	}

	private void close(Closeable writer) {
		if (writer != null) {
			try {
				writer.close();
			} catch (IOException ignore) {
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see net.sf.thirdi.jdbc.CSVConverter#createCSVInfo(java.sql.ResultSet)
	 */
	@Override
	public CSVInfo createCSVInfo(ResultSet rs) throws IllegalArgumentException {
		return getDefaultCSVInfo(rs);
	}

	private void finish(Writer writer) throws IOException {
		writer.flush();
		writer.close();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see net.sf.thirdi.jdbc.CSVConverter#toCSV(java.sql.ResultSet)
	 */
	@Override
	public Writer toCSV(ResultSet rs) {
		return toCSV(rs, (CSVInfo) null, DEFAULT_MAXCOUNT);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see net.sf.thirdi.jdbc.CSVConverter#toCSV(java.sql.ResultSet,
	 * net.sf.thirdi.jdbc.CSVInfo)
	 */
	@Override
	public Writer toCSV(ResultSet rs, CSVInfo info) {
		return toCSV(rs, info, DEFAULT_MAXCOUNT);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see net.sf.thirdi.jdbc.CSVConverter#toCSV(java.sql.ResultSet,
	 * net.sf.thirdi.jdbc.CSVInfo, long)
	 */
	@Override
	public Writer toCSV(ResultSet rs, CSVInfo info, long maxcount) {
		Writer writer = null;
		try {
			if (info == null) {
				info = getDefaultCSVInfo(rs);
			} else {
				info = arrangesMode(info);
			}
			long count = 1;
			writer = getDefaultWriter();

			if (info.isAddHeaderColumn()) {
				writeColumnHeader(writer, info);
			}

			while (rs.next()) {
				if (count > maxcount)
					break;
				write(rs, writer, info);
				count++;
			}

			finish(writer);

			return writer;
		} catch (SQLException e) {
			throw new SQLRuntimeException(e);
		} catch (IOException e) {
			throw new IORuntimeException(e);
		} finally {
			close(writer);
		}

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see net.sf.thirdi.jdbc.CSVConverter#toCSV(java.sql.ResultSet, long)
	 */
	@Override
	public Writer toCSV(ResultSet rs, long maxcount) {
		return toCSV(rs, (CSVInfo) null, maxcount);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see net.sf.thirdi.jdbc.CSVConverter#toCSV(java.sql.ResultSet,
	 * java.io.Writer)
	 */
	@Override
	public <T extends Writer> T toCSV(ResultSet rs, T writer) {
		return toCSV(rs, writer, null, DEFAULT_MAXCOUNT);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see net.sf.thirdi.jdbc.CSVConverter#toCSV(java.sql.ResultSet,
	 * java.io.Writer, net.sf.thirdi.jdbc.CSVInfo)
	 */
	@Override
	public <T extends Writer> T toCSV(ResultSet rs, T writer, CSVInfo info) {
		return toCSV(rs, writer, info, DEFAULT_MAXCOUNT);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see net.sf.thirdi.jdbc.CSVConverter#toCSV(java.sql.ResultSet,
	 * java.io.Writer, net.sf.thirdi.jdbc.CSVInfo, long)
	 */
	@Override
	public <T extends Writer> T toCSV(ResultSet rs, T writer, CSVInfo info,
			long maxcount) {
		try {
			if (writer == null)
				// TODO message move to property file.
				throw new IllegalArgumentException("writer must not be null.");
			if (info == null) {
				info = getDefaultCSVInfo(rs);
			} else {
				info = arrangesMode(info);
			}

			long count = 1;

			if (info.isAddHeaderColumn()) {
				writeColumnHeader(writer, info);
			}

			while (rs.next()) {
				if (count > maxcount)
					break;
				write(rs, writer, info);
				count++;
			}

			finish(writer);

			return writer;

		} catch (SQLException e) {
			throw new SQLRuntimeException(e);
		} catch (IOException e) {
			throw new IORuntimeException(e);
		} finally {
			close(writer);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see net.sf.thirdi.jdbc.CSVConverter#toCSV(java.sql.ResultSet,
	 * java.io.Writer, long)
	 */
	@Override
	public <T extends Writer> T toCSV(ResultSet rs, T writer, long maxcount) {
		return toCSV(rs, writer, null, maxcount);
	}

	private void write(ResultSet rs, Writer writer, CSVInfo info) {

		try {
			ResultSetMetaData rsmd = rs.getMetaData();

			int count = rsmd.getColumnCount();

			if (rs.isBeforeFirst())
				rs.next();

			StringBuffer sb = new StringBuffer();
			for (int i = 0; i < count; i++) {

				Type type = types.get(rsmd.getColumnType(i + 1));

				if (type != null) {
					String o = type.getResultsetDataToString(rs, i + 1);
					String value = nullToBlank(o);

					value = info.getCSVHeaderInfo().getCSVColumn(i + 1)
							.getCSVColumnDecorator().decorate(value);

					if (info.getCSVFormatInfo().isEscapeDoubleQuotation()) {
						value = escapeDoubleQuotation(value);
					}
					if (info.getFormatMode() == CSVFormatMode.RFC4180) {
						value = formatCSV(value);
					} else if (info.getFormatMode() == CSVFormatMode.MANUAL) {
						if (info.getCSVFormatInfo().isRemoveReturn()) {
							value = removeReturnChar(value);
						}
					}

					if (i == count - 1) {
						sb.append(value).append(
								info.getCSVFormatInfo().getRecordSeparator());
					} else {
						sb.append(value).append(
								info.getCSVFormatInfo().getFieldSeparator());
					}
				}
			}

			writer.write(sb.toString());

		} catch (SQLException e) {
			throw new SQLRuntimeException(e);
		} catch (IOException e) {
			throw new IORuntimeException(e);
		}
	}

	private void writeColumnHeader(Writer writer, CSVInfo info) {

		try {
			CSVHeaderInfo hinfo = info.getCSVHeaderInfo();
			CSVFormatInfo finfo = info.getCSVFormatInfo();
			int count = hinfo.getColumnCount();

			StringBuffer sb = new StringBuffer();
			for (int i = 0; i < count; i++) {

				CSVColumn c = hinfo.getCSVColumn(i + 1);
				String columnName = null;
				if (c.hasAliasName()) {
					columnName = c.getColumnAliasName();
				} else {
					columnName = c.getColumnName();
					switch (hinfo.getMappingLetterType()) {
					case CAPITAL_LETTER:
						columnName = columnName.toUpperCase();
						break;
					case SMALL_LETTER:
						columnName = columnName.toLowerCase();
						break;
					default:
					}
				}

				if (i == count - 1) {
					sb.append(columnName).append(finfo.getRecordSeparator());
				} else {
					sb.append(columnName).append(finfo.getFieldSeparator());
				}
			}

			writer.write(sb.toString());

		} catch (IOException e) {
			throw new IORuntimeException(e);
		}
	}

}
