package jp.thunderclacker.svnmail;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.MissingResourceException;
import java.util.Properties;
import java.util.ResourceBundle;

import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeMessage;

/**
 * Subversionコミットメール送信プログラム。<br>
 * <br>
 * Subversionにてコミットしたときにメールを送信します。<br>
 * $REPOS/hooks/post-commitから呼ばれます。<br>
 *  
 * @author		thunderclacker
 * @version		1.0
 *
 */
public class SVNMailSender {
	
	/** ResourceBundle */
	private static final ResourceBundle RESOURCE = ResourceBundle.getBundle("svn-mailsender");

	/** ログ出力ON/OFFのフラグ。true--ログ出力ON、false--ログ出力OFF */
	private boolean logging = true;
	
	/** このインスタンスで扱うリポジトリのPATH文字列 */
	private String repository = ""; 
	
	/** このインスタンスで扱うリビジョン番号 */
	private String revision = "";

	/** 簡単なログ出力 */
	private PrintWriter writer = null; 

	
	/**
	 * 当プログラムを起動するメソッドです。
	 * 
	 * @param		args			args[0]--$REPOS、args[1]--$REV
	 */
	public static void main(String[] args) {
		SVNMailSender svnMailSender = new SVNMailSender(args);
    	svnMailSender.sendCommitMail();
    }

	
	/**
	 * 初期処理を行います。<br>
	 * 初期処理にてロギングの設定をします。失敗しても例外は投げません。
	 * 
	 * @param		args			args[0]--$REPOS、args[1]--$REV
	 */
	public SVNMailSender(String[] args) {

		// リポジトリパス取得
		this.repository = args[0];

		// リビジョン取得
		this.revision = args[1];

    	try {
    		// 設定ファイルで「logging=true」の場合はログファイルを出力する。
			String logging = getProperty("logging");
	    	if(logging.equalsIgnoreCase("true")) {
	    		this.logging=true;
	    	}
	    	
    	} catch (Exception e) {
    		e.printStackTrace();
    	}
	}

	
	/**
	 * コミットメールを送信します。<br>
	 * メール送信に必要な情報は「<code>svn-mailsernder.properties</code>」と「svnlookコマンド」
	 * にて取得します。
	 * 
	 */
    public void sendCommitMail() {

		try {
			
			writer = new PrintWriter(new BufferedWriter(new FileWriter(repository +"/mail/svncommit.log", true)));

	    	log("------------------ SVNMailSender ------------------");
	    	 // SMTPサーバ名を設定
			Properties mailprop =new Properties();
			mailprop.put("mail.smtp.host", getProperty("server.smtp") );

			// メールセッションを確立
			Session session=Session.getDefaultInstance(mailprop,null);

			// マイムメッセージを生成
			MimeMessage message=new MimeMessage(session);
			
			// Toヘッダ設定
			String mailTo = getProperty("mail.to");
			log("To: " + mailTo);
			message.setRecipients( MimeMessage.RecipientType.TO
								 , InternetAddress.parse(mailTo, true));
			
			// Fromヘッダ設定
			String mailFrom = getProperty("mail.from");
			log("From: " + mailFrom);
			InternetAddress objFrm=new InternetAddress(mailFrom);
			message.setFrom(objFrm);

			// Reply-Toヘッダ設定
			String mailReplyTo = getProperty("mail.replyto");
			log("Reply-To: " + mailReplyTo);
			message.setReplyTo( InternetAddress.parse(mailReplyTo, true));
			
			// 件名設定
			String subject = getSubject();
			log("Subject: " + subject);
			message.setSubject(subject,"ISO-2022-JP");
			   
			// 本文設定
			String text = getText();
			log("Text: " + text);
			message.setText(text,"ISO-2022-JP");
			  
			// メール送信
			Transport.send(message); 

			log("--------------------------------------------------\n");
	    } catch (Exception e) {
	    	e.printStackTrace();
	    	
	    	// StackTraceをログ出力
			StringWriter sw = new StringWriter();
			e.printStackTrace(new PrintWriter(sw));
			log(sw.toString());
	    } finally {
	    	writer.close();
	    	writer = null;
	    }
    }
    
    
    /**
     * メール本文を取得します。
     * 
     * @return			メール本文の文字列
     */
    private String getText() throws IOException { 

    	// メール本文
		StringBuffer msg = new StringBuffer();
		
		// TracのURL取得
		String tracHome = getProperty("trac.home");
    	
    	// 更新日付取得
    	String upddate = (new SimpleDateFormat("yyyy/MM/dd HH:mm")).format(new Date());

    	// 更新者取得
    	String author = execCommand("svnlook author " + repository + " -r " + revision);
    	
    	// コミットメッセージ取得
    	String log = execCommand("svnlook log " + repository + " -r " + revision);
    	
    	// 変更情報取得
    	String changed = execCommand("svnlook changed " + repository + " -r " + revision);
    	
		msg.append("\nこのメールはSubversionからのコミット通知メールです。\n");
    	msg.append("\n============================================");
    	msg.append("\n Repository: "	).append(repository);
		msg.append("\n   Revision: Rev.").append(revision);
		msg.append("\n   Commiter: "	).append(author);
		if(!tracHome.equals("")) {
			msg.append("\n   Trac-URL: "	).append(tracHome).append("/changeset/").append(revision);
		}
		msg.append("\n       Date: "	).append(upddate);
    	msg.append("\n============================================");
    	msg.append("\nLog Message:\n"	).append(log);
    	msg.append("\n");
    	msg.append("\nChanged:\n"		).append(changed);
    	msg.append("\n");
    	msg.append("\n---");
    	msg.append("\nReply-To: "		).append(getProperty("mail.replyto") );
    	
    	return msg.toString();
    }
    
    
    /**
     * メールヘッダのSubjectを取得します。
     * 
     * @return			Subjectに設定する文字列
     */
    private String getSubject() throws IOException {

    	// 変更内容を取得
    	String changed = execCommand("svnlook changed " + repository + " -r " + revision);

    	// 一行目だけ取り出す 
    	String firstLine = changed.split("\n")[0];
    	
    	// 一行目を"/"で分割して配列にする。
    	String[] tmp = firstLine.split("/");
    	
    	// 配列の数により戻り値を決める
    	String moduleName = "";
    	if(tmp.length == 1 ) {
    		
    		// "/"が無いときは、PATHの１階層目のみを取得する。
    		moduleName = tmp[0] + "/";
    		
    	} else 	if(tmp.length == 2 ) {
        	
    		// "/"が１つだけの時は、PATHの２階層目のみを取得する。
        	moduleName = tmp[1] + "/";
        	
    	} else {
    		
    		// "/"が２つ以上の時は、PATHの２階層目＋３階層目を取得する。
    		moduleName = tmp[1] + "/" + tmp[2]+ "/";
    	}
    	
    	// タイトル文字列取得
		String prefix  = getProperty("subject.prefix");
    	String subject = prefix + "Subversion commit to \"" + moduleName + "\"";
    	
    	return subject;
    }

    
    /**
     * 外部コマンドを実行します。<br>
     * 引数で引き渡されたコマンド文字列を実行し、結果の出力メッセージを返します。<br>
     * 
     * @param			command					実行するコマンド
     * @return			コマンド実行で出力されるメッセージ
     */
    
    private String execCommand(String command) throws IOException {

    	// プロセスオブジェクトを生成
        Process process = Runtime.getRuntime().exec(command);
        
        // 外部コマンドの標準出力を取得するための入力ストリームを取得
        InputStream is = process.getInputStream();
        BufferedReader br = new BufferedReader(new InputStreamReader(is));
        
        // 標準出力を１行づつ取り出す。
        StringBuffer buf = new StringBuffer("");
        String line;
        while ((line = br.readLine()) != null) {
          buf.append(line).append("\n");
        }
        
        // 最後の改行コードを消す
    	buf.delete(buf.length()-1, buf.length());
    	
        return buf.toString();
    }

    /**
     * プロパティを取得します。
     * 何も取得できなかった場合には空文字を返します。
     * 
     * @param		key		プロパティを取得するキー
     * @return		取得したプロパティ文字列。
     */
    private String getProperty(String key) {
		try {
			String res = RESOURCE.getString(key);
			if(res != null) {
				return res.trim();
			}
			
		} catch (MissingResourceException e) {
			;
		}
		return "";
    }
    
    
    /**
     * ログを出力します。
     * 引数で引き渡された文字列をログ出力します。
     * 
     * @param		msg					ログ出力する文字列
     */
    private void log(String msg) {

    	// ログ出力OFFの場合は何もしない
		if(!this.logging) {
			return ;
		}

		try {
			writer.println(msg);
			writer.flush();
		} catch (Exception e) {
    		e.printStackTrace();
    	}
    }
}
