/*
 * Copyright (c) 2007, 2008 University of Tsukuba
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the University of Tsukuba nor the names of its
 *    contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/**
 * @file	drivers/pci.c
 * @brief	PCI driver (core)
 * @author	T. Shinagawa
 */

/*
 * Copyright (c) 2010-2012 Yuichi Watanabe
 */

#include <common/common.h>
#include <core/assert.h>
#include <core/printf.h>
#include <core/spinlock.h>
#include <core/vm.h>
#include <io/io.h>
#include <io/pci.h>
#include "pci_internal.h"

static spinlock_t	pci_config_lock = SPINLOCK_INITIALIZER;
drvdata_hdl_t		pci_handle;
struct resource		mmio_resource;
struct resource		io_resource;

LIST2_DEFINE_HEAD(struct pci_device, pci_device_on_root_bus);
LIST_DEFINE_HEAD(pci_device_list);
LIST_DEFINE_HEAD(pci_driver_list);
LIST_DEFINE_HEAD(pci_mmconfig_list);

static pci_config_address_t
pci_make_config_address(u8 bus, u8 devfn, int reg)
{
	pci_config_address_t addr;

	addr.allow = 1;
	addr.reserved = 0;
	addr.bus_no = bus;
	addr.devfn = devfn;
	addr.reg_no = reg;
	addr.type = 0;
	return addr;
}

/********************************************************************************
 * PCI internal interfaces
 ********************************************************************************/

void
pci_init_root_resources(void)
{
	rm_init_resource(&mmio_resource, 0, 0xffffffffffffffffLL,
			 RESOURCE_TYPE_MEM, 0, "mmio_resource");
	rm_init_resource(&io_resource, 0, 0x0000ffff,
			 RESOURCE_TYPE_MEM, 0, "io_resource");

}

void
pci_dump_resources(void)
{
	rm_dump_resources(&mmio_resource);
	rm_dump_resources(&io_resource);
}

static void
pci_insert_recouces(struct pci_device *dev)
{
	struct pci_device *parent;
	struct resource *resource, *presource;
	vmmerr_t err;
	int i, j;

	parent = dev->parent;

	for (i = 0; i < PCI_RESOURCE_NUM; i++) {
		resource = dev->resource + i;
		if (resource->type == RESOURCE_TYPE_INVALID) {
			continue;
		}

		if (parent == NULL) {
			switch (resource->type) {
			case RESOURCE_TYPE_MMIO:
				rm_insert_resource(&mmio_resource, resource);
				break;
			case RESOURCE_TYPE_IO:
				rm_insert_resource(&io_resource, resource);
				break;
			}
		} else {
			for (j = 0; j < PCI_RESOURCE_NUM; j++) {
				presource = parent->resource + j;
				if ((presource->data & PCI_RESOURCE_WINDOW)
				    == 0) {
					continue;
				}
				if (resource->type != presource->type) {
					continue;
				}
				if ((presource->data &
				     PCI_RESOURCE_PREFETCHABLE) &&
				    (resource->data &
				     PCI_RESOURCE_PREFETCHABLE) == 0) {
					continue;
				}
				err = rm_insert_resource(presource, resource);
				if (err == VMMERR_SUCCESS) {
					break;
				}
			}
		}
		if (resource->parent == NULL) {
			printf("Fail to insert resource. %s"
			       " %016llx-%016llx type %x data %x\n",
			       dev->name, resource->start,
			       resource->end, resource->type, resource->data);
		}
	}
}

vmmerr_t
pci_alloc_resource(struct pci_device *dev, struct resource *resource,
		  size_t size, phys_t align, u32 type, u32 data)
{
	struct pci_device *parent;
	struct resource *presource;
	vmmerr_t err;
	int j;

	parent = dev->parent;
	if (parent == NULL) {
		return VMMERR_NODEV;
	}

	for (j = 0; j < PCI_RESOURCE_NUM; j++) {
		presource = parent->resource + j;
		if ((presource->data & PCI_RESOURCE_WINDOW) == 0) {
			continue;
		}
		if (presource->type != type) {
			continue;
		}
		if ((presource->data & PCI_RESOURCE_PREFETCHABLE) &&
		    (data & PCI_RESOURCE_PREFETCHABLE) == 0) {
			continue;
		}
		err = rm_alloc_resource(presource, resource, size,
					align, type, data, dev->name);
		if (err == VMMERR_SUCCESS) {
			break;
		}
	}
	if (resource->parent == NULL) {
		return VMMERR_BUSY;
	}
	return VMMERR_SUCCESS;
}

void
pci_register_device(struct pci_device *parent, struct pci_device *dev)
{
#ifdef PRINT_PCI_DEVICE
	printf("%s %04x:%04x %06x %02x\n",
	       dev->name,
	       dev->vendor_id, dev->device_id,
	       dev->class_code, dev->header_type);
#endif
	LIST_APPEND(pci_device_list, dev);
	if (parent) {
		dev->parent = parent;
		LIST2_ADD(parent->children, sibling, dev);
	} else {
		dev->parent = NULL;
		LIST2_ADD(pci_device_on_root_bus, sibling, dev);
	}
	pci_insert_recouces(dev);
}

struct pci_device *
pci_next_device_on_root_bus(struct pci_device *cur)
{
	return LIST2_NEXT(pci_device_on_root_bus, sibling, cur);
}

int
pci_config_addr_handler(iotype_t type, ioport_t port, void *data)
{
	struct pci_vm_data *pci_data;

	pci_data = (struct pci_vm_data *)vm_get_driver_data(pci_handle);

	switch(type) {
	case IOTYPE_OUTL:
		pci_data->config_addr.value = *(u32 *)data;
		break;
	case IOTYPE_INL:
		*(u32 *)data = pci_data->config_addr.value;
		break;
	default:
		break;
	}
	return 1; /* no pass */
}

int
pci_config_data_handler(iotype_t type, ioport_t port, void *data)
{
	struct pci_vm_data *pci_data;
	bool wr;
	ioport_t size;
	int ret;

	pci_data = (struct pci_vm_data *)vm_get_driver_data(pci_handle);

	if (!pci_data->config_addr.allow) {
		return 1; /* no pass */
	}
	ret = pci_check_assignment(pci_data->config_addr.bus_no,
				   pci_data->config_addr.devfn);
	switch (ret) {
	case PCI_NOT_ASSIGNED:
		/* nothing to do */
		break;
	case PCI_ASSIGNED:
		spinlock_lock(&pci_config_lock);
		out32(PCI_CONFIG_ADDR_PORT, pci_data->config_addr.value);
		do_io_pass(type, port, data);
		spinlock_unlock(&pci_config_lock);
		break;
	case PCI_DUMMY_FUNC0:
		iotype_to_size_wr(type, &size, &wr);
		return pci_dummy_func0((pci_data->config_addr.reg_no << 2)
				       + (port & 0x3), size, wr, data);
	}
	return 1; /* no pass */
}

/********************************************************************************
 * PCI service functions exported to PCI device drivers
 ********************************************************************************/
/* ------------------------------------------------------------------------------
   PCI driver registration
 ------------------------------------------------------------------------------ */

/**
 * @brief		PCI driver registration function
 * @param  driver	pointer to struct pci_driver
 */
void pci_register_driver(struct pci_driver *driver)
{
	struct pci_device *dev;

	LIST_APPEND(pci_driver_list, driver);

	LIST_FOREACH (pci_device_list, dev) {
		u32 id = dev->id;
		u32 class = dev->class_code;

		if (idmask_match(id, driver->id) &&
		    idmask_match(class, driver->class)) {
			dev->driver = driver;
			driver->new(dev);
		}
	}
	return;
}

/* ------------------------------------------------------------------------------
   PCI configuration registers access
 ------------------------------------------------------------------------------ */

#define DEFINE_pci_read_config_data_without_lock(size)		\
static inline u##size pci_read_config_data##size##_without_lock(pci_config_address_t addr, int offset) \
{								\
	u##size data;						\
	out32(PCI_CONFIG_ADDR_PORT, addr.value);		\
	in##size(PCI_CONFIG_DATA_PORT + offset, &data);		\
	return data;						\
}

#define DEFINE_pci_write_config_data_without_lock(size)		\
static inline void pci_write_config_data##size##_without_lock(pci_config_address_t addr, int offset, u##size data) \
{								\
	out32(PCI_CONFIG_ADDR_PORT, addr.value);		\
	out##size(PCI_CONFIG_DATA_PORT + offset, data);		\
}

DEFINE_pci_read_config_data_without_lock(8)
DEFINE_pci_read_config_data_without_lock(16)
DEFINE_pci_read_config_data_without_lock(32)
DEFINE_pci_write_config_data_without_lock(8)
DEFINE_pci_write_config_data_without_lock(16)
DEFINE_pci_write_config_data_without_lock(32)

#define MMCONFIG_OFFSET(start_bus_no, bus_no, devfn, offset) \
	((bus_no - start_bus_no) * 32 * 8 * 4096 + devfn * 4096 + offset) \

#define DEFINE_pci_read_mmconfig(size)					\
	u##size pci_read_mmconfig_##size(u16 seg, u8 bus, u8 devfn,	\
					 pci_off_t offset)		\
	{								\
		struct pci_mmconfig *mmconfig;				\
		LIST_FOREACH(pci_mmconfig_list, mmconfig) {		\
			if (mmconfig->seg != seg ||			\
			    mmconfig->start_bus_no > bus ||		\
			    mmconfig->end_bus_no < bus) {		\
				continue;				\
			}						\
			return *(vu##size*)((ulong)mmconfig->virt_base + \
				MMCONFIG_OFFSET(mmconfig->start_bus_no, \
					bus, devfn, offset));		\
		}							\
		printf("Out of range of MMCONFIG: %04x:%02x:%02x.%01x %x\n", \
		       seg, bus, PCI_DEV_NO(devfn), PCI_FUNC_NO(devfn),	\
		       offset);						\
		return (u##size)((i##size)-1);				\
	}

#define DEFINE_pci_write_mmconfig(size)					\
	void pci_write_mmconfig_##size(u16 seg, u8 bus, u8 devfn,	\
					pci_off_t offset, u##size data)	\
	{								\
		struct pci_mmconfig *mmconfig;				\
		LIST_FOREACH(pci_mmconfig_list, mmconfig) {		\
			if (mmconfig->seg != seg ||			\
			    mmconfig->start_bus_no > bus ||		\
			    mmconfig->end_bus_no < bus) {		\
				continue;				\
			}						\
			*(vu##size*)((ulong)mmconfig->virt_base +	\
				MMCONFIG_OFFSET(mmconfig->start_bus_no, \
					bus, devfn, offset)) = data;	\
			return;						\
		}							\
		printf("Out of range of MMCONFIG: %04x:%02x:%02x.%01x %x\n", \
		       seg, bus, PCI_DEV_NO(devfn), PCI_FUNC_NO(devfn),	\
		       offset);						\
	}

DEFINE_pci_read_mmconfig(8)
DEFINE_pci_read_mmconfig(16)
DEFINE_pci_read_mmconfig(32)
DEFINE_pci_read_mmconfig(64)
DEFINE_pci_write_mmconfig(8)
DEFINE_pci_write_mmconfig(16)
DEFINE_pci_write_mmconfig(32)
DEFINE_pci_write_mmconfig(64)

#define DEFINE_pci_read_legacy_config(size)				\
	u##size pci_read_legacy_config_##size(u8 bus, u8 devfn,		\
					      pci_off_t offset)		\
	{								\
		u##size data;						\
		pci_config_address_t addr;				\
		if (offset > 0xff)					\
			return (u##size)((i##size)-1);			\
		addr =  pci_make_config_address(bus, devfn, offset >> 2); \
		offset &= 0x03;						\
		ASSERT(offset + size / 8 <= 4);				\
		spinlock_lock(&pci_config_lock);			\
		data = pci_read_config_data##size##_without_lock(addr,	\
							       offset);	\
		spinlock_unlock(&pci_config_lock);			\
		return data;						\
	}

#define DEFINE_pci_write_legacy_config(size)				\
	void pci_write_legacy_config_##size(u8 bus, u8 devfn,		\
					    pci_off_t offset,		\
					    u##size data)		\
	{								\
		pci_config_address_t addr;				\
		if (offset > 0xff)					\
			return;						\
		addr =  pci_make_config_address(bus, devfn, offset >> 2); \
		offset &= 0x03;						\
		ASSERT(offset + size / 8 <= 4);			\
		spinlock_lock(&pci_config_lock);			\
		pci_write_config_data##size##_without_lock(addr, offset,\
							   data);	\
		spinlock_unlock(&pci_config_lock);			\
	}

DEFINE_pci_read_legacy_config(8)
DEFINE_pci_read_legacy_config(16)
DEFINE_pci_read_legacy_config(32)
DEFINE_pci_write_legacy_config(8)
DEFINE_pci_write_legacy_config(16)
DEFINE_pci_write_legacy_config(32)

#define DEFINE_pci_read_config(size)					\
	u##size pci_read_config_##size(/* u16 seg, */ u8 bus, u8 devfn,	\
				       pci_off_t offset)		\
	{								\
		u16 seg = 0;						\
		if (LIST_HEAD(pci_mmconfig_list)) {			\
			return pci_read_mmconfig_##size(seg, bus, devfn,\
							offset);	\
		} else {						\
			ASSERT(seg == 0);				\
			return pci_read_legacy_config_##size(bus,	\
							     devfn,	\
							     offset);	\
		}							\
	}

#define DEFINE_pci_write_config(size)					\
	void pci_write_config_##size(/* u16 seg, */ u8 bus, u8 devfn,	\
				     pci_off_t offset, u##size data)	\
	{								\
		u16 seg = 0;						\
		if (LIST_HEAD(pci_mmconfig_list)) {			\
			pci_write_mmconfig_##size(seg, bus, devfn,	\
						 offset, data);		\
		} else {						\
			ASSERT(seg == 0);				\
			pci_write_legacy_config_##size(bus, devfn,	\
						       offset, data);	\
		}							\
	}

DEFINE_pci_read_config(8)
DEFINE_pci_read_config(16)
DEFINE_pci_read_config(32)
DEFINE_pci_write_config(8)
DEFINE_pci_write_config(16)
DEFINE_pci_write_config(32)

u64
pci_read_config_64(/* u16 seg, */ u8 bus, u8 devfn, pci_off_t offset)
{
	u16 seg = 0;
	if (LIST_HEAD(pci_mmconfig_list)) {
		return pci_read_mmconfig_64(seg, bus, devfn, offset);
	}
	ASSERT(seg == 0);
	return (((u64)pci_read_config_32(bus, devfn, offset + 4) << 32) |
		pci_read_config_32(bus, devfn, offset));
}

void
pci_write_config_64(/* u16 seg, */ u8 bus, u8 devfn, pci_off_t offset, u64 data)
{
	u16 seg = 0;
	if (LIST_HEAD(pci_mmconfig_list)) {
		pci_write_mmconfig_64(seg, bus, devfn, offset, data);
	}
	ASSERT(seg == 0);
	pci_write_config_32(bus, devfn, offset + 4, data >> 32);
	pci_write_config_32(bus, devfn, offset, data);
}
