/*
 * Copyright (c) 2007, 2008 University of Tsukuba
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the University of Tsukuba nor the names of its
 *    contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
/*
 * Copyright (c) 2010-2012 Yuichi Watanabe
 */

#include <core/panic.h>
#include "asm.h"
#include "mm.h"
#include <core/types.h>

static void *
hphys_mapmem (phys_t phys, u32 flags, uint len, bool wr)
{
	void *p;

	p = mapmem(MAPMEM_HPHYS |
		   (wr ? MAPMEM_WRITE : 0) |
		   (flags & MAPMEM_CACHE_MASK), phys, len);
	if (p == NULL) {
		panic("Faild to map memory space for access.");
	}
	return p;
}

void
read_hphys_b (phys_t phys, void *data, u32 flags)
{
	u8 *p;

	p = (u8 *)hphys_mapmem (phys, flags, sizeof *p, false);
	*(u8 *)data = *p;
	unmapmem (p, sizeof *p);
}

void
write_hphys_b (phys_t phys, u32 data, u32 flags)
{
	u8 *p;

	p = (u8 *)hphys_mapmem (phys, flags, sizeof *p, true);
	*p = data;
	unmapmem (p, sizeof *p);
}

void
read_hphys_w (phys_t phys, void *data, u32 flags)
{
	u16 *p;

	p = (u16 *)hphys_mapmem (phys, flags, sizeof *p, false);
	*(u16 *)data = *p;
	unmapmem (p, sizeof *p);
}

void
write_hphys_w (phys_t phys, u32 data, u32 flags)
{
	u16 *p;

	p = (u16 *)hphys_mapmem (phys, flags, sizeof *p, true);
	*p = data;
	unmapmem (p, sizeof *p);
}

void
read_hphys_l (phys_t phys, void *data, u32 flags)
{
	u32 *p;

	p = (u32 *)hphys_mapmem (phys, flags, sizeof *p, false);
	*(u32 *)data = *p;
	unmapmem (p, sizeof *p);
}

void
write_hphys_l (phys_t phys, u32 data, u32 flags)
{
	u32 *p;

	p = (u32 *)hphys_mapmem (phys, flags, sizeof *p, true);
	*p = data;
	unmapmem (p, sizeof *p);
}

void
read_hphys_q (phys_t phys, void *data, u32 flags)
{
	u64 *p;

	p = (u64 *)hphys_mapmem (phys, flags, sizeof *p, false);
	*(u64 *)data = *p;
	unmapmem (p, sizeof *p);
}

void
write_hphys_q (phys_t phys, u64 data, u32 flags)
{
	u64 *p;

	p = (u64 *)hphys_mapmem (phys, flags, sizeof *p, true);
	*p = data;
	unmapmem (p, sizeof *p);
}

bool
cmpxchg_hphys_l (phys_t phys, u32 *olddata, u32 data, u32 flags)
{
	u32 *p;
	bool r;

	p = (u32 *)hphys_mapmem (phys, flags, sizeof *p, true);
	r = asm_lock_cmpxchgl (p, olddata, data);
	unmapmem (p, sizeof *p);
	return r;
}

bool
cmpxchg_hphys_q (phys_t phys, u64 *olddata, u64 data, u32 flags)
{
	u64 *p;
	bool r;

	p = (u64 *)hphys_mapmem (phys, flags, sizeof *p, true);
        r = asm_lock_cmpxchgq (p, olddata, data);
	unmapmem (p, sizeof *p);
	return r;
}

