/*
 * Copyright (c) 2010-2012 Yuichi Watanabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of Yuichi Watanabe nor the names of its
 *    contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <common/list.h>
#include <core/acpi.h>
#include <core/initfunc.h>
#include <core/mm.h>
#include <core/printf.h>
#include <core/string.h>
#include <io/pci.h>
#include "pci_internal.h"

#define MCFG_SIGNATURE		"MCFG"

struct mcfg {
	struct acpi_desc_header header;
	u8 reserved[8];
} __attribute__ ((packed));

struct mcfg_entry {
	u64 address;
	u16 pci_segment;
	u8 start_bus_number;
	u8 end_bus_number;
	u32 reserved;
} __attribute__ ((packed));

LIST_DEFINE_HEAD(acpi_mmconfig_list);

static acpi_parser_t parse_mcfg;
static void
parse_mcfg(void *table, u32 length)
{
	int count;
	int i;
	struct mcfg *mcfg = (struct mcfg *)table;
	struct mcfg_entry *mcfg_entry;
	struct acpi_mmconfig *mmconfig;

	printf ("ACPI MCFG found.\n");

	count = (mcfg->header.length - sizeof(struct mcfg))
		/ sizeof(struct mcfg_entry);
	for (i = 0; i < count; i++) {
		mcfg_entry = (struct mcfg_entry *)((ulong)mcfg
			+ sizeof(struct mcfg)
			+ sizeof(struct mcfg_entry) * i);
		mmconfig = (struct acpi_mmconfig*)alloc(sizeof(*mmconfig));
		if (!mmconfig) {
			printf("parse_mcfg: Can't alloc memory");
			return;
		}
		mmconfig->base = mcfg_entry->address;
		mmconfig->seg = mcfg_entry->pci_segment;
		mmconfig->start_bus_no = mcfg_entry->start_bus_number;
		mmconfig->end_bus_no = mcfg_entry->end_bus_number;
		LIST_APPEND(acpi_mmconfig_list, mmconfig);
	}

	acpi_hide_table(table);
	printf ("ACPI MCFG cleared.\n");
}

struct acpi_mmconfig*
acpi_get_next_mmconfig(struct acpi_mmconfig* cur)
{
	return LIST_NEXT(acpi_mmconfig_list, cur);
}

static void
pci_acpi_init(void)
{
	acpi_register_parser(MCFG_SIGNATURE, parse_mcfg);
}

INITFUNC("global3", pci_acpi_init);
