/*
 * Copyright (c) 2010-2013 Yuichi Watanabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the Yuichi Watanabe nor the names of its
 *    contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * 8259A programable interrupt controller hook.
 */
#include <core/initfunc.h>
#include <core/io.h>
#include <core/types.h>
#include <core/extint.h>
#include <core/vm.h>
#include <io/io.h>
#include <io/pic.h>

/* #define PIC_DEBUG */
#ifdef PIC_DEBUG
#include <core/printf.h>
#define PIC_DBG(...)						\
	do {							\
		printf("PIC: " __VA_ARGS__);			\
	} while (0)
#else
#define PIC_DBG(...)
#endif

static void
pic_master_cmd(u8 data)
{
	if ((data & PIC_ICW1_SELECT_MASK) == PIC_ICW1_SELECT_MATCH) {
		extint_reset_8259a();
	}
}

static int
pic_io_hook(iotype_t type, ioport_t port, void *data)
{
	switch (type) {
	case IOTYPE_OUTB:
		pic_master_cmd(*(u8 *)data);
		break;
	default:
		break;
	}
	return 0; /* pass */
}

static void
pic_init (void)
{
	if (vm_get_id() != 0) {
		/* 
		 * For VMs except vm0, we provide pic emulator.
		 * See io/pic_emu.c.
		 */
		return;
	}

	set_iofunc(PIC_MASTER_CMD_PORT, pic_io_hook);
}

static void
pic_reset(void)
{
	if (vm_get_id() != 0) {
		return;
	}

	PIC_DBG("Reset pic.\n");

	out8(PIC_MASTER_IMR_PORT, 0xff);
	out8(PIC_SLAVE_IMR_PORT, 0xff);

	out8(PIC_MASTER_CMD_PORT, 0x11);
	out8(PIC_MASTER_CMD_PORT, 0x08);
	out8(PIC_MASTER_CMD_PORT, 0x1 << 2);
	out8(PIC_MASTER_CMD_PORT, 0x01);

	out8(PIC_SLAVE_CMD_PORT, 0x11);
	out8(PIC_SLAVE_CMD_PORT, 0x70);
	out8(PIC_SLAVE_CMD_PORT, 2);
	out8(PIC_SLAVE_CMD_PORT, 0x01);

	out8(PIC_MASTER_IMR_PORT, 0xfb);
	out8(PIC_SLAVE_IMR_PORT, 0xff);
}

static void
pic_unmask(void)
{
	if (vm_get_id() != 0) {
		return;
	}

	out8(PIC_MASTER_IMR_PORT, 0xfa);
}

INTR_SETUPVM1(pic_init);
INTR_RESET1(pic_reset);
INTR_UNMASK(pic_unmask);
