/*
 * From NetBSD usr/src/usr.sbin/acpitools/amldb/amldb.c
 */
/*-
 * Copyright (c) 1999 Mitsuru IWASAKI <iwasaki@FreeBSD.org>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	Id: amldb.c,v 1.8 2000/08/08 14:12:24 iwasaki Exp 
 *	$FreeBSD: src/usr.sbin/acpi/amldb/amldb.c,v 1.3 2001/10/22 17:25:32 iwasaki Exp $
 */
/*
 * Copyright (c) 2010-2014 Yuichi Watanabe
 */

#include <core/assert.h>
#include <core/mm.h>
#include <core/printf.h>
#include <core/string.h>
#include "acpi.h"
#include "panic.h"
#include "aml/aml_amlmem.h"
#include "aml/aml_common.h"
#include "aml/aml_env.h"
#include "aml/aml_evalobj.h"
#include "aml/aml_name.h"
#include "aml/aml_parse.h"
#include "aml/aml_region.h"

static void
acpi_load_aml(u8 *start, u8 *end)
{
	struct aml_environ	env;

	aml_new_name_group(start);

	memset(&env, 0, sizeof(env));
	env.curname = aml_get_rootname();
	env.dp = start;
	env.end = end;

	aml_local_stack_push(aml_local_stack_create());
	aml_parse_objectlist(&env, 0);
	aml_local_stack_delete(aml_local_stack_pop());

	ASSERT(env.dp == env.end);
}

#ifdef TTY_SERIAL
static int
dis_callback(struct aml_name *name, va_list ap)
{
	union aml_object *dis;
	int saved_flag;

	aml_print_curname(name);
	AML_DEBUGPRINT("\n");

	dis = name->property;

	if (dis->type != aml_t_method) {
		return 0; /* continue */
	}

	printf("Disabling ");
	saved_flag = aml_debug;
	aml_debug = 1;
	aml_print_curname(name);
	aml_debug = saved_flag;
	printf("\n");

	if (memcmp(name->namep, "_DIS", 4) != 0) {
		printf("Unexpected method name. 0x%4.4s\n", name->namep);
		return 0; /* continue */
	}

	memcpy(name->namep, "DIS_", 4);

	return 0; /* continue */
}

static int
hid_callback(struct aml_name *name, va_list ap)
{
	union aml_object *hid;
	struct aml_name *parent;

	aml_print_curname(name);
	AML_DEBUGPRINT("\n");

	hid = name->property;;

	if (hid->type == aml_t_method) {
		aml_local_stack_push(aml_local_stack_create());
		hid = aml_invoke_method(name, 0, NULL);
		aml_local_stack_delete(aml_local_stack_pop());
	}

	if (hid->type != aml_t_num) {
		return 0; /* continue */
	}

	if (hid->num.number != 0x0105d041) { /* PNP0501 */
		return 0; /* continue */
	}

	parent = name->parent;

	aml_apply_foreach_found_objects(parent, "_DIS", dis_callback);

	return 0; /* continue */
}

static void
disable_pnp0501_dis(void)
{
	aml_apply_foreach_found_objects(aml_get_rootname(),
					"_HID", hid_callback);
}
#endif /* TTY_SERIAL */

static int
sx_callback(struct aml_name *name, va_list ap)
{
	union aml_object *sx;
	union aml_object *elem;
#ifdef DISABLE_SLEEP
	int saved_flag;
#endif
	int sleep;
	int i;

	aml_print_curname(name);
	AML_DEBUGPRINT("\n");

	sx = name->property;

	if (sx->type != aml_t_package) {
		return 0; /* continue */
	}

	sleep = name->namep[2] - '0';

	AML_DEBUGPRINT("elements %d\n", sx->package.elements);
	for (i = 0; i < sx->package.elements; i++) {
		if (i >= 2) {
			continue;
		}
		elem = sx->package.objects[i];
		if (elem->type == aml_t_num) {
			acpi_sleep[sleep].valid = 1; /* valid */
			acpi_sleep[sleep].type[i] = elem->num.number;
			AML_DEBUGPRINT("val %d\n", elem->num.number);
		}
	}

#ifdef DISABLE_SLEEP
	if (sleep < 1 || sleep > 4) {
		return 0; /* continue */
	}
	printf("Disabling ");
	saved_flag = aml_debug;
	aml_debug = 1;
	aml_print_curname(name);
	aml_debug = saved_flag;
	printf("\n");
	memcpy(name->namep, "Sx__", 4);
	name->namep[1] = '0' + sleep;
#endif /* DISABLE_SLEEP */
	return 0; /* continue */
}

static void
parse_sx(void)
{
	aml_apply_foreach_found_objects(aml_get_rootname(),
					"_S0_", sx_callback);
	aml_apply_foreach_found_objects(aml_get_rootname(),
					"_S1_", sx_callback);
	aml_apply_foreach_found_objects(aml_get_rootname(),
					"_S2_", sx_callback);
	aml_apply_foreach_found_objects(aml_get_rootname(),
					"_S3_", sx_callback);
	aml_apply_foreach_found_objects(aml_get_rootname(),
					"_S4_", sx_callback);
	aml_apply_foreach_found_objects(aml_get_rootname(),
					"_S5_", sx_callback);
}


void
acpi_modify_aml(u8 *start, u8 *end)
{
	acpi_load_aml(start, end);

	parse_sx();
#ifdef TTY_SERIAL
	disable_pnp0501_dis();
#endif /* TTY_SERIAL */
}
