/*
 * Copyright (c) 2013-2014 Yuichi Watanabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the Yuichi Watanabe nor the names of its
 *    contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <core/types.h>
#include <core/initfunc.h>
#include <core/io.h>
#include <core/panic.h>
#include <core/rm.h>
#include <core/printf.h>
#include <core/vm.h>
#include <io/pci.h>
#include <io/vmrm.h>

/* #define ACPI_DEBUG */
#ifdef ACPI_DEBUG
#define ACPI_DBG(...)						\
	do {							\
		printf("ACPIEMU: " __VA_ARGS__);			\
	} while (0)
#else
#define ACPI_DBG(...)
#endif

#define ACPI_REG_SIZE		0x100
#define ACPI_PM1A_CNT_OFFSET	0x04
#define PM1_CNT_SLP_TYPX_MASK	0x1C00
#define PM1_CNT_SLP_TYPX_SHIFT	10
#define PM1_CNT_SLP_EN_BIT	0x2000
#define PM1A_S5_TYPE		0 /* must match a value in
				     bios/src/ssdt-misc.dsl */

struct acpi_emu {
	struct resource resource;
};

static drvdata_hdl_t acpi_handle;

static void
acpi_emu_pm1_sleep (u32 val)
{
	u8 type;

	type = (val & PM1_CNT_SLP_TYPX_MASK) >> PM1_CNT_SLP_TYPX_SHIFT;
	ACPI_DBG("acpi_emu_pm1_sleep 0x%x 0x%x\n", val, type);
	if (type == PM1A_S5_TYPE) {
		ACPI_DBG("shutdown\n");
		vm_shutdown();
	}
}

static int
acpi_emu_pm1_cnt_reg (iotype_t type, ioport_t port, void *data)
{
	u32 val;

	ACPI_DBG("acpi_emu_pm1_ctrl_reg 0x%x\n", type);

	switch (type) {
	case IOTYPE_OUTB:
		val = *(u8 *)data;
		break;
	case IOTYPE_OUTW:
		val = *(u16 *)data;
		break;
	case IOTYPE_OUTL:
		val = *(u32 *)data;
		break;
	default:
		return 1; /* emulated */
	}
	if ((val & PM1_CNT_SLP_EN_BIT) == 0) {
		acpi_emu_pm1_sleep (val);
	}
	return 1; /* emulated */
}

static void
acpi_emu_setup(void)
{
	struct acpi_emu *acpi;
	vmmerr_t err;

	if (vm_get_id() == 0) {
		return;
	}

	acpi = vm_get_driver_data(acpi_handle);

	err = rm_alloc_resource(vmrm_io_resource(), &acpi->resource,
				ACPI_REG_SIZE, ACPI_REG_SIZE,
				RESOURCE_TYPE_IO, 0,
				"acpi_emu");
	if (err != VMMERR_SUCCESS) {
		panic("Can't find free ioport for ACPI device.");
	}

	set_iofunc(acpi->resource.start + ACPI_PM1A_CNT_OFFSET,
		   acpi_emu_pm1_cnt_reg);
}

ioport_t
acpi_get_dev_base(void)
{
	struct acpi_emu *acpi;
	acpi = vm_get_driver_data(acpi_handle);

	return acpi->resource.start;
}

static void
acpi_emu_init(void)
{
	acpi_handle = vm_alloc_driver_data(sizeof(struct acpi_emu));
}

DRIVER_SETUPVM(acpi_emu_setup);
DRIVER_INIT(acpi_emu_init);
