/*
 * From BitVisor 1.0.1 core/ap.c
 */
/*
 * Copyright (c) 2007, 2008 University of Tsukuba
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the University of Tsukuba nor the names of its
 *    contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
/*
 * Copyright (c) 2010-2014 Yuichi Watanabe
 */

#include <core/cpu.h>
#include <core/initfunc.h>
#include <core/mm.h>
#include <core/panic.h>
#include "apic.h"
#include "apic_regs.h"
#include "asm.h"
#include "constants.h"
#include "sleep.h"

void *apic_virt_base = NULL;

bool
apic_available(void)
{
	u32 a, b, c, d;
	u64 tmp;

	asm_cpuid (1, 0, &a, &b, &c, &d);
	if (!(d & CPUID_1_EDX_APIC_BIT))
		return false;
	asm_rdmsr64 (MSR_IA32_APIC_BASE_MSR, &tmp);
	if (!(tmp & MSR_IA32_APIC_BASE_MSR_APIC_GLOBAL_ENABLE_BIT))
		return false;
	return true;
}

void
apic_wait_for_idle(void)
{
	while ((apic_read32(APIC_ICR_LOW_OFFSET)
		& APIC_ICR_LOW_STATUS_BIT) != APIC_ICR_LOW_STATUS_IDLE);
}

static void
apic_assert_init(void)
{
	apic_wait_for_idle();
	apic_write32(APIC_ICR_LOW_OFFSET,
		     APIC_ICR_LOW_DSH_OTHERS | APIC_ICR_LOW_TRIGGER_LEVEL |
		     APIC_ICR_LOW_MT_INIT | APIC_ICR_LOW_LEVEL_ASSERT);
}

static void
apic_deassert_init(void)
{
	apic_wait_for_idle();
	apic_write32(APIC_ICR_LOW_OFFSET,
		     APIC_ICR_LOW_DSH_OTHERS | APIC_ICR_LOW_TRIGGER_LEVEL |
		     APIC_ICR_LOW_MT_INIT | APIC_ICR_LOW_LEVEL_DEASSERT);
}

static void
apic_send_startup_ipi(u8 addr)
{
	apic_wait_for_idle();
	apic_write32(APIC_ICR_LOW_OFFSET,
		     APIC_ICR_LOW_DSH_OTHERS | APIC_ICR_LOW_TRIGGER_EDGE |
		     APIC_ICR_LOW_MT_STARTUP | APIC_ICR_LOW_LEVEL_ASSERT |
		     addr);
}

apic_id_t
apic_read_id (void)
{
	if (apic_available ()) {
		u32 a, b, c, d;
		asm_cpuid (1, 0, &a, &b, &c, &d);
		return (b & APIC_ID_MASK) >> APIC_ID_AID_SHIFT;
	} else {
		return 0;
	}
}

void
apic_broadcast_nmi(void)
{
	if (apic_virt_base == NULL) {
		return;
	}
	apic_wait_for_idle();
	apic_write32(APIC_ICR_LOW_OFFSET,
		     APIC_ICR_LOW_DSH_OTHERS | APIC_ICR_LOW_TRIGGER_EDGE |
		     APIC_ICR_LOW_MT_NMI | APIC_ICR_LOW_LEVEL_ASSERT);
}

void
apic_start_ap (u8 addr)
{
	int i;

	apic_assert_init();
	usleep(200000);
	apic_deassert_init();
	usleep (200000);
	for (i = 0; i < 2; i++) {
		apic_send_startup_ipi(addr);
		usleep (200000);
	}
}

bool
apic_check_apic_interrupt(vector_t vector)
{
	u32 isr;

	isr = apic_read32(APIC_ISR_OFFSET + 0x10 * (vector / 32));
	if (isr & (1 << (vector % 32))) {
		return true;
	} else {
		return false;
	}
}

void
apic_write_eoi(void)
{
	apic_write32(APIC_EOI_OFFSET, 0);
}

u32
apic_read_icrh(void)
{
	return apic_read32(APIC_ICR_HIGH_OFFSET);
}

void
apic_send_ipi(vector_t vector, apic_id_t dest)
{
	apic_wait_for_idle();
	apic_write32(APIC_ICR_HIGH_OFFSET,
		     dest << APIC_ICR_HIGH_DEST_SHIFT);
	apic_write32(APIC_ICR_LOW_OFFSET,
		     APIC_ICR_LOW_DSH_DEST | APIC_ICR_LOW_TRIGGER_EDGE |
		     APIC_ICR_LOW_LEVEL_ASSERT | APIC_ICR_LOW_MT_FIXED |
		     vector);
}

void
apic_send_sipi(vector_t vector, apic_id_t dest)
{
	apic_wait_for_idle();
	apic_write32(APIC_ICR_HIGH_OFFSET,
		     dest << APIC_ICR_HIGH_DEST_SHIFT);
	apic_write32(APIC_ICR_LOW_OFFSET,
		     APIC_ICR_LOW_DSH_DEST | APIC_ICR_LOW_TRIGGER_EDGE |
		     APIC_ICR_LOW_LEVEL_ASSERT | APIC_ICR_LOW_MT_STARTUP |
		     vector);
}

void
apic_send_init(apic_id_t dest)
{
	apic_wait_for_idle();

	apic_write32(APIC_ICR_HIGH_OFFSET,
		     dest << APIC_ICR_HIGH_DEST_SHIFT);
	apic_write32(APIC_ICR_LOW_OFFSET,
		     APIC_ICR_LOW_DSH_DEST | APIC_ICR_LOW_TRIGGER_LEVEL
		     | APIC_ICR_LOW_MT_INIT | APIC_ICR_LOW_LEVEL_ASSERT);

	usleep(200000);

	apic_write32(APIC_ICR_HIGH_OFFSET,
		     dest << APIC_ICR_HIGH_DEST_SHIFT);
	apic_write32(APIC_ICR_LOW_OFFSET,
		     APIC_ICR_LOW_DSH_DEST | APIC_ICR_LOW_TRIGGER_LEVEL |
		     APIC_ICR_LOW_MT_INIT | APIC_ICR_LOW_LEVEL_DEASSERT);

}

static void
apic_init(void)
{
	if (!apic_available ()) {
		panic("APIC is not available.");
	}
	apic_virt_base = mapmem(MAPMEM_HPHYS | MAPMEM_WRITE | MAPMEM_UC,
				APIC_BASE, PAGESIZE);
	if (apic_virt_base == NULL) {
		panic("Failed to map APIC.");
	}
}

static void
apic_enable(void)
{
	u64 reg_val;

	reg_val = apic_read32(APIC_SPURIOUS_INTR_VECT_OFFSET);
	reg_val |= APIC_SOFTWARE_ENABLE_BIT;
	apic_write32(APIC_SPURIOUS_INTR_VECT_OFFSET, reg_val);
}

static void
apic_pcpu_init(void)
{
	u64 reg_val;

	/* Clear logical apic id */
	apic_write32(APIC_LOGICAL_APIC_ID_OFFSET, 0);

	/* Enable APIC */
	asm_rdmsr64(MSR_IA32_APIC_BASE_MSR, &reg_val);
	reg_val |= MSR_IA32_APIC_BASE_MSR_APIC_GLOBAL_ENABLE_BIT;
	asm_wrmsr64(MSR_IA32_APIC_BASE_MSR, reg_val);

	apic_enable();
}

void
apic_disable(void)
{
	u64 reg_val;

	/*
	 * Disable APIC. APIC can still issue IPIs.
	 */
	apic_wait_for_idle();
	reg_val = apic_read32(APIC_SPURIOUS_INTR_VECT_OFFSET);
	reg_val &= ~APIC_SOFTWARE_ENABLE_BIT;
	apic_write32(APIC_SPURIOUS_INTR_VECT_OFFSET, reg_val);
}

void
apic_maskall(void)
{
	apic_disable();
	apic_enable();
}

void
apic_set_priority(vector_t priority)
{
	apic_write32(APIC_TPR_OFFSET, priority);
}

INITFUNC("global5", apic_init);
INITFUNC("pcpu1", apic_pcpu_init);
