/*
 * Copyright (c) 2007, 2008 University of Tsukuba
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the University of Tsukuba nor the names of its
 *    contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
/*
 * Copyright (c) 2010-2014 Yuichi Watanabe
 */

#include <core/spinlock.h>
#include <core/serial.h>
#include <core/serial_regs.h>
#include <core/time.h>
#include <core/thread.h>
#include "asm.h"
#include "initfunc.h"
#include "io_io.h"
#include "process.h"
#include "serial.h"
#include "types.h"

#define INIT_RATELSB		RATELSB_115200
#define INIT_RATEMSB		0x0
#define INIT_INTR		0x0
#define INIT_FIFOCTL		0x0
#define INIT_LINECTL		LINECTL_DATA8BIT
#define INIT_MODECTL		0x3 /* RTS | DTR */

static spinlock_t serial_lock = SPINLOCK_INITIALIZER;
bool serial_disable = false;
static bool initialized;

static void
serial_init (unsigned int port)
{
	spinlock_lock(&serial_lock);
	asm_outb (port + PORT_LINECTL, LINECTL_DLAB_BIT);
	asm_outb (port + PORT_RATELSB, INIT_RATELSB);
	asm_outb (port + PORT_RATEMSB, INIT_RATEMSB);
	asm_outb (port + PORT_LINECTL, INIT_LINECTL);
	asm_outb (port + PORT_MODECTL, INIT_MODECTL);
	asm_outb (port + PORT_INTRCTL, INIT_INTR);
	asm_outb (port + PORT_FIFOCTL, INIT_FIFOCTL);
	spinlock_unlock(&serial_lock);
}

static void
serial_send (unsigned int port, unsigned char data)
{
	u8 status;
	u64 expire;
	bool detect_lockup = time_initialized ();

	spinlock_lock(&serial_lock);
	if (serial_disable) {
		spinlock_unlock(&serial_lock);
		return;
	}
	if (detect_lockup) {
		expire = get_cpu_time() + 1000 * 1000; /* 1 sec */
	}
	do {
		if (detect_lockup && time_after (get_cpu_time (), expire)) {
			serial_disable = true;
			spinlock_unlock(&serial_lock);
			return;
		}
		asm_inb (port + PORT_LINESTAT, &status);
	} while (!(status & LINESTAT_TX_BIT));
	asm_outb (port + PORT_DATA, data);
	spinlock_unlock(&serial_lock);
}

static unsigned char
serial_recv (unsigned int port)
{
	u8 status, data;

	spinlock_lock(&serial_lock);
	asm_inb (port + PORT_LINESTAT, &status);
	while (!(status & LINESTAT_RX_BIT)) {
		spinlock_unlock(&serial_lock);
		schedule();
		spinlock_lock(&serial_lock);
		asm_inb (port + PORT_LINESTAT, &status);
	}
	asm_inb (port + PORT_DATA, &data);
	spinlock_unlock(&serial_lock);
	return data;
}

static int
serialin_msghandler (int m, int c)
{
	if (!initialized) {
		serial_init (SERIAL_PORT1);
		initialized = true;
	}
	if (m == 0)
		return serial_recv (SERIAL_PORT1);
	return 0;
}

static int
serialout_msghandler (int m, int c)
{
	if (!initialized) {
		serial_init (SERIAL_PORT1);
		initialized = true;
	}
	if (m == 0)
		serial_send (SERIAL_PORT1, (unsigned char)c);
	return 0;
}

void
serial_putDebugChar (int c)
{
	serialout_msghandler (0, c);
}

int
serial_getDebugChar (void)
{
	return serialin_msghandler (0, 0);
}

void
serial_putchar (unsigned char c)
{
	if (!initialized) {
		serial_init (SERIAL_PORT1);
		initialized = true;
	}
	if (c == '\n')
		serial_send (SERIAL_PORT1, '\r');
	else if (c == '\0')
		c = ' ';
	serial_send (SERIAL_PORT1, c);
}

void
serial_init_iohook (void)
{
	/* nothing to do */
}

void
serial_out(unsigned char c)
{
	if (!initialized) {
		serial_init (SERIAL_PORT1);
		initialized = true;
	}
	serial_send(SERIAL_PORT1, c);
}

static void
serial_init_msg (void)
{
	initialized = false;
	msgregister ("serialin", serialin_msghandler);
	msgregister ("serialout", serialout_msghandler);
}

INITFUNC ("msg0", serial_init_msg);
