/*
 * Copyright (c) 2007, 2008 University of Tsukuba
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the University of Tsukuba nor the names of its
 *    contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
/*
 * Copyright (c) 2010-2014 Yuichi Watanabe
 */

/**
 * @file	drivers/pci_init.c
 * @brief	PCI driver (init)
 * @author	T. Shinagawa
 */
#include <core/rm.h>
#include <core/printf.h>
#include "pci.h"
#include "pci_internal.h"
#include "pci_init.h"
#include "pci_conceal.h"
#include <core/acpi.h>
#include <core/mmio.h>

#define PCI_MAX_BUSES		256
#define PCI_MAX_DEVICES		32
#define PCI_MAX_FUNCS		8

static const char driver_name[] = "pci_driver";

static u8 pci_scan_bus(struct pci_device *parent, u8 bn);

DEFINE_ALLOC_FUNC(pci_device)

static pci_config_address_t pci_make_config_address(int bus, int dev, int fn, int reg)
{
	pci_config_address_t addr;

	addr.allow = 1;
	addr.reserved = 0;
	addr.bus_no = bus;
	addr.device_no = dev;
	addr.func_no = fn;
	addr.reg_no = reg;
	addr.type = 0;
	return addr;
}

static u32 pci_get_base_address_mask(pci_config_address_t addr)
{
	u32 tmp, mask;

	tmp = pci_read_config_data32(addr, 0);
	pci_write_config_data32(addr, 0, 0xFFFFFFFF);
	mask = pci_read_config_data32(addr, 0);
	pci_write_config_data32(addr, 0, tmp);
	return mask;
}

static void pci_save_base_address_masks(struct pci_device *dev)
{
	int i;
	pci_config_address_t addr = dev->address;

	for (i = 0; i < PCI_CONFIG_BASE_ADDRESS_NUMS; i++) {
		addr.reg_no = PCI_CONFIG_ADDRESS_GET_REG_NO(base_address) + i;
		dev->base_address_mask[i] = pci_get_base_address_mask(addr);
	}
	addr.reg_no = PCI_CONFIG_ADDRESS_GET_REG_NO(ext_rom_base);
	dev->base_address_mask[6] = pci_get_base_address_mask(addr);
}

static void pci_read_config_space(struct pci_device *dev)
{
	int i;
	pci_config_address_t addr = dev->address;
	struct pci_config_space *cs = &dev->config_space;

//	for (i = 0; i < PCI_CONFIG_REGS32_NUM; i++) {
	for (i = 0; i < 16; i++) {
		addr.reg_no = i;
		cs->regs32[i] = pci_read_config_data32(addr, 0);
	}

	dev->vendor_id = pci_read_config16(dev, PCI_CONFIG_VENDOR_ID);
	dev->device_id = pci_read_config16(dev, PCI_CONFIG_DEVICE_ID);
	dev->class_code = pci_read_config32(dev, PCI_CONFIG_CLASS_CODE - 1)
		>> 8;
	dev->header_type = pci_read_config8(dev, PCI_CONFIG_HEADER_TYPE);
	if (dev->type != PCI_CONFIG_HEADER_TYPE_1) {
		return;
	}
	dev->sec_bus = pci_read_config8(dev,
					PCI_CONFIG_SUBORDINATE_BUS_NUMBER);
	dev->sub_bus = pci_read_config8(dev,
					PCI_CONFIG_SUBORDINATE_BUS_NUMBER);
}

struct pci_config_mmio_data *
pci_search_config_mmio (u16 seg_group, u8 bus_no)
{
	struct pci_config_mmio_data *p;

	for (p = pci_config_mmio_data_head; p; p = p->next)
		if (p->seg_group == seg_group &&
		    p->bus_start <= bus_no && bus_no <= p->bus_end)
			return p;
	return NULL;
}

u32
pci_get_bar_mask(struct pci_device *dev, int offset)
{
	u32 mask;
	u32 tmp;

	tmp = pci_read_config32(dev, offset);
	pci_write_config32(dev, offset, 0xFFFFFFFF);
	mask = pci_read_config32(dev, offset);
	pci_write_config32(dev, offset, tmp);
	return mask;
}

static void
pci_init_resource_window(struct pci_device *dev)
{
	struct resource *resource;
	u64 reg_val, base, limit;

	if (dev->type != PCI_CONFIG_HEADER_TYPE_1) {
		return;
	}

	/*
	 * Iitialize I/O window.
	 */
	resource = dev->resource + PCI_RESOURCE_IO_WINDOW_INDEX;
	reg_val = pci_read_config8(dev, PCI_CONFIG_IO_BASE);
	base = (reg_val & PCI_CONFIG_IO_BASE_ADDRESS_MASK) << 8;
	if ((reg_val & PCI_CONFIG_IO_BASE_CAP_MASK)
	    == PCI_CONFIG_IO_BASE_32_BIT) {
		reg_val = pci_read_config16(dev, PCI_CONFIG_IO_BASE_UPPER);
		base |= reg_val << 16;
	}
	reg_val = pci_read_config8(dev, PCI_CONFIG_IO_LIMIT);
	limit = (((reg_val & PCI_CONFIG_IO_LIMIT_ADDRESS_MASK)
		  + PCI_CONFIG_IO_LIMIT_CAP_MASK) << 8) + 0xff;
	if ((reg_val & PCI_CONFIG_IO_LIMIT_CAP_MASK)
	    == PCI_CONFIG_IO_LIMIT_32_BIT) {
		reg_val = pci_read_config16(dev, PCI_CONFIG_IO_LIMIT_UPPER);
		limit |= reg_val << 16;
	}
	if (base <= limit) {
		rm_init_resource(resource, base, limit, RESOURCE_TYPE_IO,
				 PCI_RESOURCE_WINDOW, dev->name);
	}

	/*
	 * Iitialize memory window.
	 */
	resource = dev->resource + PCI_RESOURCE_MEMORY_WINDOW_INDEX;
	reg_val = pci_read_config16(dev, PCI_CONFIG_MEMORY_BASE);
	base = (reg_val & PCI_CONFIG_MEMORY_BASE_ADDRESS_MASK) << 16;
	reg_val = pci_read_config16(dev, PCI_CONFIG_MEMORY_LIMIT);
	limit = (((reg_val & PCI_CONFIG_MEMORY_LIMIT_ADDRESS_MASK)
		  + PCI_CONFIG_MEMORY_LIMIT_RESERVED_MASK) << 16) + 0xffff;
	if (base <= limit) {
		rm_init_resource(resource, base, limit, RESOURCE_TYPE_MMIO,
				 PCI_RESOURCE_WINDOW, dev->name);
	}

	/*
	 * Iitialize prefetchable memory window.
	 */
	resource = dev->resource + PCI_RESOURCE_PREFETCHABLE_WINDOW_INDEX;
	reg_val = pci_read_config16(dev, PCI_CONFIG_PREFETCHABLE_BASE);
	base = ((reg_val & PCI_CONFIG_PREFETCHABLE_BASE_ADDRESS_MASK) << 16);
	if ((reg_val & PCI_CONFIG_PREFETCHABLE_BASE_CAP_MASK)
	    == PCI_CONFIG_PREFETCHABLE_BASE_64_BIT) {
		reg_val = pci_read_config32(dev,
				     PCI_CONFIG_PREFETCHABLE_BASE_UPPER);
		base |= reg_val << 32;
	}
	reg_val = pci_read_config16(dev,
				    PCI_CONFIG_PREFETCHABLE_LIMIT);
	limit = (((reg_val & PCI_CONFIG_PREFETCHABLE_LIMIT_ADDRESS_MASK)
		  + PCI_CONFIG_PREFETCHABLE_LIMIT_CAP_MASK) << 16) + 0xffff;
	if ((reg_val & PCI_CONFIG_PREFETCHABLE_LIMIT_CAP_MASK)
	    == PCI_CONFIG_PREFETCHABLE_LIMIT_64_BIT) {
		reg_val = pci_read_config32(dev,
				     PCI_CONFIG_PREFETCHABLE_LIMIT_UPPER);
		limit |= (reg_val << 32);
	}
	if (base <= limit) {
		rm_init_resource(resource, base, limit, RESOURCE_TYPE_MMIO,
			PCI_RESOURCE_WINDOW | PCI_RESOURCE_PREFETCHABLE,
			dev->name);
	}

	reg_val = pci_read_config16(dev, PCI_CONFIG_BRIDGE_CONTROL);
	if ((reg_val & PCI_CONFIG_VGA_ENABLE) == 0) {
		return;
	}
	resource = dev->resource + PCI_RESOURCE_VGA_MEM;
	rm_init_resource(resource, 0xa0000, 0xbffff,
			 RESOURCE_TYPE_MMIO,
			 PCI_RESOURCE_WINDOW | PCI_RESOURCE_PREFETCHABLE,
			 dev->name);

	resource = dev->resource + PCI_RESOURCE_VGA_IO1;
	rm_init_resource(resource, 0x3b0, 0x3bb,
			 RESOURCE_TYPE_IO, PCI_RESOURCE_WINDOW,
			 dev->name);
	resource = dev->resource + PCI_RESOURCE_VGA_IO2;
	rm_init_resource(resource, 0x3c0, 0x3df,
			 RESOURCE_TYPE_IO, PCI_RESOURCE_WINDOW,
			 dev->name);
}

static void
pci_init_resources(struct pci_device *dev)
{
	int i, bar_num;
	u32 reg_val;
	int offset;
	u32 mask;
	u32 data;
	phys_t start;
	struct resource *resource;

	switch (dev->type) {
	case PCI_CONFIG_HEADER_TYPE_0:
		bar_num = PCI_CONFIG_BAR_NUM;
		break;
	case PCI_CONFIG_HEADER_TYPE_1:
		bar_num = PCI_CONFIG_HEADER1_BAR_NUM;
		break;
	default:
		return;
	}

	for (i = 0; i < bar_num; i++) {
		offset = PCI_CONFIG_BASE_ADDRESS0 + i * 4;
		resource = dev->resource + i;

		reg_val = pci_read_config32(dev, offset);
		if (reg_val == 0x00000000 || reg_val == 0xFFFFFFFF) {
			/* Unimplemented register */
			continue;
		}
		mask = pci_get_bar_mask(dev, offset);
		if ((reg_val & PCI_CONFIG_BAR_SPACEMASK) ==
		    PCI_CONFIG_BAR_IOSPACE) {
			start = reg_val & PCI_CONFIG_BAR_IOMASK;
			if (start == 0x0) {
				continue;
			}

			mask &= PCI_CONFIG_BAR_IOMASK;
			mask |= 0xFFFF0000;
			rm_init_resource(resource,
					 start,
					 start + (~mask),
					 RESOURCE_TYPE_IO,
					 0,
					 dev->name);
		} else {
			if (reg_val & PCI_CONFIG_BAR_PREFETCHABLE) {
				data = PCI_RESOURCE_PREFETCHABLE;
			} else {
				data = 0;
			}
			start = reg_val & PCI_CONFIG_BAR_MEMMASK;
			if ((reg_val & PCI_CONFIG_BAR_MEMTYPE_MASK) ==
					   PCI_CONFIG_BAR_MEMTYPE_64) {
				i++;
				offset += 4;
				reg_val = pci_read_config32(dev,
							    offset);
				start |= (u64)reg_val << 32;
			}
			if (start == 0x0) {
				continue;
			}

			mask &= PCI_CONFIG_BAR_MEMMASK;

			rm_init_resource(resource,
					 start,
					 start + (~mask),
					 RESOURCE_TYPE_MMIO,
					 data,
					 dev->name);
		}
	}

	pci_init_resource_window(dev);

	resource = dev->resource + PCI_RESOURCE_EXPANTION_ROM_INDEX;
	reg_val = pci_read_config32(dev, PCI_CONFIG_EXPANSION_ROM);
	if (reg_val != 0x00000000 && reg_val != 0xFFFFFFFF) {
		start = reg_val & PCI_CONFIG_ROM_MEMMASK;
		mask = pci_get_bar_mask(dev, PCI_CONFIG_EXPANSION_ROM) &
			PCI_CONFIG_ROM_MEMMASK;
		rm_init_resource(resource,
				 start,
				 start + (~mask),
				 RESOURCE_TYPE_MMIO,
				 0,
				 dev->name);
	}
}

static struct pci_device *pci_new_device(pci_config_address_t addr)
{
	struct pci_device *dev;

	dev = alloc_pci_device();
	if (dev != NULL) {
		memset(dev, 0, sizeof(*dev));
		dev->driver = NULL;
		dev->address = addr;
		dev->bus_no = addr.bus_no;
		dev->devfn = addr.devfn;
		snprintf(dev->name, PCI_LOCATION_STR_SIZE, PCI_LOCATION_FORMAT,
			 PCI_LOCATION_VALUE(dev));
		dev->config_mmio = pci_search_config_mmio (0, addr.bus_no);
		pci_read_config_space(dev);
		pci_init_resources(dev);
		pci_save_base_address_masks(dev);
		dev->conceal = pci_conceal_new_device (dev);
		LIST2_HEAD_INIT(dev->children, sibling);
		pci_append_device(dev);
	}
	return dev;
}

struct pci_device *
pci_possible_new_device (pci_config_address_t addr)
{
	u16 data;
	struct pci_device *ret = NULL;

	data = pci_read_config_data16 (addr, 0);
	if (data != 0xFFFF) {
		ret = pci_new_device (addr);
	}
	return ret;
}

static u8
pci_scan_buses_behind_bridge(struct pci_device *bridge)
{
	int bn;

	for (bn = bridge->sec_bus; bn <= bridge->sub_bus; bn++) {
		bn = pci_scan_bus(bridge, bn);
	}

	return bn;
}

static u8
pci_scan_bus(struct pci_device *parent, u8 bn)
{
	int dn, fn;
	pci_config_address_t addr;
	u16 data;
	struct pci_device *dev;
	u8 ret;
	u8 max_bn = bn;

	for (dn = 0; dn < PCI_MAX_DEVICES; dn++) {
		for (fn = 0; fn < PCI_MAX_FUNCS; fn++) {
			addr = pci_make_config_address(bn, dn, fn, 0);
			data = pci_read_config_data16(addr, 0);
			if (data == 0xFFFF) /* not exist */
				continue;
			dev = pci_new_device(addr);
			if (dev == NULL) {
				panic("pci_scan_bus: Can't alloc memory");
			}
			pci_insert_device(parent, dev);

			if (dev->type == PCI_CONFIG_HEADER_TYPE_1) {
				ret = pci_scan_buses_behind_bridge(dev);
				if (max_bn < ret) {
					max_bn = ret;
				}
			}

			if (fn == 0 && dev->multi_function == 0)
				break;
		}
	}
	return max_bn;
}

static void pci_find_devices(void)
{
	int bn;

	for (bn = 0; bn < PCI_MAX_BUSES; bn++) {
		bn = pci_scan_bus(NULL, bn);
	}
	return;
}

static void
pci_read_mcfg (void)
{
	uint n;
	struct pci_config_mmio_data d, *tmp, **pnext;

	pnext = &pci_config_mmio_data_head;
	*pnext = NULL;
	for (n = 0; acpi_read_mcfg (n, &d.base, &d.seg_group, &d.bus_start,
				    &d.bus_end); n++) {
		d.next = NULL;
		d.phys = d.base + (d.bus_start << 20);
		d.len = ((u32)(d.bus_end - d.bus_start) + 1) << 20;
		d.map = mapmem_hphys (d.phys, d.len, MAPMEM_WRITE |
				      MAPMEM_PCD | MAPMEM_PWT);
		tmp = alloc (sizeof *tmp);
		memcpy (tmp, &d, sizeof *tmp);
		*pnext = tmp;
		pnext = &tmp->next;
	}
}

static void
pci_mcfg_register_handler (void)
{
	uint n;
	struct pci_config_mmio_data *p;

	for (n = 0, p = pci_config_mmio_data_head; p; n++, p = p->next) {
		if (vm_get_id() == 0) {
			printf ("MCFG [%u] %04X:%02X-%02X (%llX,%X)\n",
			n, p->seg_group, p->bus_start, p->bus_end, p->phys,
			p->len);
		}
		mmio_register_unlocked (p->phys, p->len,
					pci_config_mmio_handler, p);
	}
}

static void
pci_init()
{
	pci_init_root_resources();
	pci_read_mcfg ();
	pci_find_devices();
	pci_assign_all_devices();
	return;
}

static void
pci_vminit()
{
	core_io_register_handler(PCI_CONFIG_ADDR_PORT, 1, pci_config_addr_handler, NULL,
				 CORE_IO_PRIO_HIGH, driver_name);
	core_io_register_handler(PCI_CONFIG_DATA_PORT, 4, pci_config_data_handler, NULL,
				 CORE_IO_PRIO_HIGH, driver_name);
	pci_mcfg_register_handler ();
	return;
}

DRIVER_INIT(pci_init);
DRIVER_VMINIT(pci_vminit);
