/*
 * Copyright (c) 2013 Igel Co., Ltd
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the University of Tsukuba nor the names of its
 *    contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
/*
 * Copyright (c) 2014 Yuichi Watanabe
 */

#include <EfiCommon.h>
#include <EfiApi.h>
#include <Protocol/SimpleFileSystem/SimpleFileSystem.h>
#include <Protocol/LoadedImage/LoadedImage.h>
#include <Protocol/FileInfo/FileInfo.h>

/* #define LOAD_DEBUG */
#ifdef LOAD_DEBUG
#define LOAD_DBG(systab, str, val)				\
	do {							\
		print(systab, str, val);			\
	} while (0)
#else
#define LOAD_DBG(systab, str, val)
#endif

#define VMM_FILE L"vmm.elf"
#define PARAM_FILE L"param.txt"
#define MODULE_FILE L"module.bin"

#define VMM_LOAD_SIZE 0x10000
struct vmm_data {
	uint64_t param_paddr;
	uint64_t param_size;
	uint64_t module_paddr;
	uint64_t module_size;
};

struct allocated_memory {
	char vmm[VMM_LOAD_SIZE];
	struct vmm_data data;
	char buf[];
};

struct file_info {
	EFI_FILE_INFO info;
	CHAR16 file_path[4095];
};

typedef int EFIAPI entry_func_t (uint32_t loadaddr, uint32_t loadsize,
				 EFI_SYSTEM_TABLE *systab, EFI_HANDLE image,
				 EFI_FILE_HANDLE file, struct vmm_data *data);

static EFI_GUID LoadedImageProtocol = EFI_LOADED_IMAGE_PROTOCOL_GUID;
static EFI_GUID FileSystemProtocol = EFI_SIMPLE_FILE_SYSTEM_PROTOCOL_GUID;
static EFI_GUID FileInfoId = EFI_FILE_INFO_ID;

static void
printhex (EFI_SYSTEM_TABLE *systab, uint64_t val, int width)
{
	CHAR16 msg[2];

	if (width > 1 || val >= 0x10)
		printhex (systab, val >> 4, width - 1);
	msg[0] = L"0123456789ABCDEF"[val & 0xF];
	msg[1] = L'\0';
	systab->ConOut->OutputString (systab->ConOut, msg);
}

static void
print (EFI_SYSTEM_TABLE *systab, CHAR16 *msg, uint64_t val)
{
	systab->ConOut->OutputString (systab->ConOut, msg);
	printhex (systab, val, 8);
	systab->ConOut->OutputString (systab->ConOut, L"\r\n");
}

static void
create_file_path (EFI_DEVICE_PATH_PROTOCOL *dp, CHAR16 *newname, CHAR16 *buf,
		  int len)
{
	int pathlen, i, j;
	CHAR16 *p;

	i = 0;
	while (!EfiIsDevicePathEnd (dp)) {
		if (dp->Type != 4 || /* Type 4 - Media Device Path */
		    dp->SubType != 4) /* Sub-Type 4 - File Path */
			goto err;
		if (i > 0 && buf[i - 1] != L'\\') {
			if (i >= len)
				goto err;
			buf[i++] = L'\\';
		}
		pathlen = EfiDevicePathNodeLength (dp);
		p = (CHAR16 *)&dp[1];
		for (j = 4; j < pathlen; j += 2) {
			if (i >= len)
				goto err;
			if (*p == L'\0')
				break;
			buf[i++] = *p++;
		}
		dp = EfiNextDevicePathNode (dp);
	}
	while (i > 0 && buf[i - 1] != L'\\')
		i--;
	for (j = 0; newname[j] != L'\0'; j++) {
		if (i >= len)
			goto err;
		buf[i++] = newname[j];
	}
	if (i >= len)
		goto err;
	buf[i++] = L'\0';
	return;
err:
	for (i = 0; i < len - 1 && newname[i] != L'\0'; i++)
		buf[i] = newname[i];
	buf[i] = L'\0';
}

EFI_STATUS EFIAPI
efi_main (EFI_HANDLE image, EFI_SYSTEM_TABLE *systab)
{
	void *tmp;
	uint32_t entry;
	UINTN readsize;
	int boot_error;
	EFI_STATUS status;
	entry_func_t *entry_func;
	EFI_FILE_HANDLE file, file2, file3;
	static CHAR16 file_path[4096];
	EFI_SIMPLE_FILE_SYSTEM_PROTOCOL *fileio;
	EFI_LOADED_IMAGE_PROTOCOL *loaded_image;
	EFI_PHYSICAL_ADDRESS paddr = 0x40000000;
	static struct file_info file_info;
	UINTN file_info_size;
	UINTN param_size;
	UINTN alloc_pages;
	struct allocated_memory *mem;
	int param_opened = 0;
	UINTN module_size;
	int module_opened = 0;

	LOAD_DBG (systab, L"Start ", 0);

	status = systab->BootServices->
		HandleProtocol (image, &LoadedImageProtocol, &tmp);
	if (EFI_ERROR (status)) {
		print (systab, L"LoadedImageProtocol ", status);
		return status;
	}
	loaded_image = tmp;
	status = systab->BootServices->
		HandleProtocol (loaded_image->DeviceHandle,
				&FileSystemProtocol, &tmp);
	if (EFI_ERROR (status)) {
		print (systab, L"FileSystemProtocol ", status);
		return status;
	}
	fileio = tmp;
	status = fileio->OpenVolume (fileio, &file);
	if (EFI_ERROR (status)) {
		print (systab, L"OpenVolume ", status);
		return status;
	}

	/* Get size of param */
	create_file_path (loaded_image->FilePath, PARAM_FILE, file_path,
			  sizeof file_path / sizeof file_path[0]);
	status = file->Open (file, &file2, file_path, EFI_FILE_MODE_READ, 0);
	LOAD_DBG (systab, L"Open " PARAM_FILE " ", status);
	if (EFI_ERROR (status)) {
		param_size = 0;
	} else {
		param_opened = 1;
		file_info_size = sizeof(file_info);
		status = file2->GetInfo (file2, &FileInfoId, &file_info_size,
					&file_info);
		if (EFI_ERROR (status)) {
			print (systab, L"GetInfo ", status);
			return status;
		}
		param_size = file_info.info.FileSize;
	}
	LOAD_DBG (systab, L"param_size ", param_size);

	/* Get size of module */
	create_file_path (loaded_image->FilePath, MODULE_FILE, file_path,
			  sizeof file_path / sizeof file_path[0]);
	status = file->Open (file, &file3, file_path, EFI_FILE_MODE_READ, 0);
	LOAD_DBG (systab, L"Open " MODULE_FILE " ", status);
	if (EFI_ERROR (status)) {
		module_size = 0;
	} else {
		module_opened = 1;
		file_info_size = sizeof(file_info);
		status = file3->GetInfo (file3, &FileInfoId, &file_info_size,
					 &file_info);
		if (EFI_ERROR (status)) {
			print (systab, L"GetInfo ", status);
			return status;
		}
		module_size = file_info.info.FileSize;
	}
	LOAD_DBG (systab, L"module_size ", module_size);

	/* Allocate pages */
	alloc_pages = (sizeof(struct allocated_memory)
		       + param_size + module_size + 4095) / 4096;
	LOAD_DBG (systab, L"Alloc ", alloc_pages);

	status = systab->BootServices->AllocatePages (AllocateMaxAddress,
						      EfiLoaderData,
						      alloc_pages,
						      &paddr);
	if (EFI_ERROR (status)) {
		print (systab, L"AllocatePages ", status);
		return status;
	}

	mem = (void *)paddr;

	/* Load param */
	if (param_opened) {
		readsize = param_size;
		mem->data.param_paddr = (uint64_t)mem->buf + module_size;
		status = file2->Read (file2, &readsize,
				      (void *)mem->data.param_paddr);
		if (EFI_ERROR (status)) {
			print (systab, L"Read ", status);
			return status;
		}
		file2->Close (file2);
		mem->data.param_size = readsize;
		LOAD_DBG (systab, L"readsize ", readsize);
	} else {
		mem->data.param_paddr = 0;
		mem->data.param_size = 0;
	}

	/* Load module */
	if (module_opened) {
		readsize = module_size;
		mem->data.module_paddr = (uint64_t)mem->buf;
		status = file3->Read (file3, &readsize,
				      (void *)mem->data.module_paddr);
		if (EFI_ERROR (status)) {
			print (systab, L"Read ", status);
			return status;
		}
		file3->Close (file3);
		mem->data.module_size = readsize;
		LOAD_DBG (systab, L"readsize ", readsize);
	} else {
		mem->data.module_paddr = 0;
		mem->data.module_size = 0;
	}

	/* Load VMM */
	create_file_path (loaded_image->FilePath, VMM_FILE, file_path,
			  sizeof file_path / sizeof file_path[0]);
	status = file->Open (file, &file2, file_path, EFI_FILE_MODE_READ, 0);
	if (EFI_ERROR (status)) {
		print (systab, L"Open ", status);
		return status;
	}
	readsize = VMM_LOAD_SIZE;
	status = file2->Read (file2, &readsize, mem->vmm);
	if (EFI_ERROR (status)) {
		print (systab, L"Read ", status);
		return status;
	}

	/* Call VMM */
	entry = *(uint32_t *)(mem->vmm + 0x18);
	entry_func = (entry_func_t *)(mem->vmm + (entry & 0xFFFF));
	boot_error = entry_func ((uint32_t)(uint64_t)&mem->vmm,
				 readsize, systab,
		    image, file2, &mem->data);
	if (!boot_error)
		systab->ConOut->OutputString (systab->ConOut,
					      L"Boot failed\r\n");
	/* Free pages and close files */
	status = systab->BootServices->FreePages (paddr, alloc_pages);
	if (EFI_ERROR (status)) {
		print (systab, L"FreePages ", status);
		return status;
	}
	file2->Close (file2);
	file->Close (file);
	if (!boot_error)
		return EFI_LOAD_ERROR;
	return EFI_SUCCESS;
}
