/*
    fileio
    copyright (c) 1998-2006 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "fileio.h"
#include <time.h>
#ifdef HAVE_UTIME_H
# include <utime.h>
#endif /* HAVE_UTIME_H */
#if defined (TM_IN_SYS_TIME) && defined (HAVE_SYS_TIME_H)
# include <sys/time.h>
#endif /* defined (TM_IN_SYS_TIME) && defined (HAVE_SYS_TIME_H) */
#if GLIB_CHECK_VERSION(2,6,0)
# include <glib/gstdio.h>
#endif /* GLIB_CHECK_VERSION(2,6,0) */


struct _FileIO
{
  gint counter;
#ifdef G_OS_WIN32
  HANDLE hFile;
#else /* not G_OS_WIN32 */
  int fd;
  gchar *file;
  guint share;
#endif /* not G_OS_WIN32 */
};


/******************************************************************************
*                                                                             *
* ja:ファイル名関数群                                                         *
*                                                                             *
******************************************************************************/
/*  ja:ファイルの絶対パスを取得する
    file,ファイル名
    path,パス名
     RET,フルパス                                                           */
gchar *
fileio_get_absolute_path (const gchar *file,
                          const gchar *path)
{
#ifdef G_OS_WIN32
  gchar *utf8str, *filename;
  gint i = 0, j, s;
  DWORD dwSize;             /* フルパスの長さ */
  HANDLE hFind;             /* 検索ハンドル */
# if ! GLIB_CHECK_VERSION(2,6,0)
  LPTSTR p, lpszFile;
  TCHAR szFile[MAX_PATH];   /* フルパス */
  TCHAR lpszLongName[MAX_PATH];
# endif /* not GLIB_CHECK_VERSION(2,6,0) */

  if (!file)
    return NULL;
# if GLIB_CHECK_VERSION(2,6,0)
  if (G_WIN32_HAVE_WIDECHAR_API ())
    {
      LPWSTR p, lpszFile;
      WCHAR szFile[MAX_PATH];   /* フルパス */
      WCHAR szLongName[MAX_PATH];
# endif /* GLIB_CHECK_VERSION(2,6,0) */

# if GLIB_CHECK_VERSION(2,6,0) || defined (UNICODE)
      utf8str = g_filename_to_utf8 (file, -1, NULL, NULL, NULL);
      if (!utf8str)
        return g_strdup (file);
      lpszFile = g_utf8_to_utf16 (file, -1, NULL, NULL, NULL);
      g_free (utf8str);
      dwSize = GetFullPathNameW (lpszFile, MAX_PATH, szFile, &p);
      g_free (lpszFile);
      if (dwSize == 0 || dwSize >= MAX_PATH)
        return g_strdup (file);
      if (dwSize >= 2)
        {
          if (szFile[1] == ':')
            {
              szLongName[0] = (WCHAR)CharUpperW ((LPWSTR)szFile[0]);
              szLongName[1] = ':';
              if (G_IS_DIR_SEPARATOR (szFile[2]))
                {
                  szLongName[2] = G_DIR_SEPARATOR;
                  i = 3;
                }
              else
                {
                  i = 2;
                }
            }
          else if (G_IS_DIR_SEPARATOR (szFile[0])
                                            && G_IS_DIR_SEPARATOR (szFile[1]))
            {
              i = 2;
              while (i < dwSize)
                {
                  if (G_IS_DIR_SEPARATOR (szFile[i]))
                    break;
                  i++;
                }
              i++;
              g_memmove (szLongName, szFile, i * sizeof (WCHAR));
            }
        }
      szLongName[i] = '\0';
      j = i;
      while (i <= dwSize)
        {
          if (szFile[i] == '\0' || G_IS_DIR_SEPARATOR (szFile[i]))
            {
              WIN32_FIND_DATAW rcf;

              s = lstrlenW (szLongName);
              g_memmove (szLongName + s, szFile + j, (i - j) * sizeof (WCHAR));
              szLongName[s + i - j] = '\0';
              hFind = FindFirstFileW (szLongName, &rcf);
              if (hFind != INVALID_HANDLE_VALUE)
                FindClose (hFind);
              else
                lstrcpyW (rcf.cFileName, L".");
              if (lstrcmpW (rcf.cFileName, L"."))
                {
                  g_memmove (szLongName + s, rcf.cFileName,
                                    lstrlenW (rcf.cFileName) * sizeof (WCHAR));
                  s += lstrlenW (rcf.cFileName);
                }
              else
                {
                  s += i - j;
                }
              if (G_IS_DIR_SEPARATOR (szFile[i]))
                szLongName[s++] = G_DIR_SEPARATOR;
              szLongName[s] = '\0';
              j = i + 1;
            }
          i++;
        }
      utf8str = g_utf16_to_utf8 (szLongName, -1, NULL, NULL, NULL);
# endif /* GLIB_CHECK_VERSION(2,6,0) || defined (UNICODE) */
# if GLIB_CHECK_VERSION(2,6,0)
    }
  else
    {
      LPSTR p, lpszFile;
      CHAR szFile[MAX_PATH];   /* フルパス */
      CHAR szLongName[MAX_PATH];
# endif /* GLIB_CHECK_VERSION(2,6,0) */

# if GLIB_CHECK_VERSION(2,6,0) || ! defined (UNICODE)
#  if GLIB_CHECK_VERSION(2,8,0)
      lpszFile = g_win32_locale_filename_from_utf8 (file);
#  elif GLIB_CHECK_VERSION(2,6,0)
      lpszFile = g_locale_from_utf8 (file, -1, NULL, NULL, NULL);
#  else /* not GLIB_CHECK_VERSION(2,6,0) */
      lpszFile = g_strdup (file);
#  endif /* not GLIB_CHECK_VERSION(2,6,0) */
      dwSize = GetFullPathNameA (lpszFile, MAX_PATH, szFile, &p);
      g_free (lpszFile);
      if (dwSize == 0 || dwSize >= MAX_PATH)
        return g_strdup (file);
      if (dwSize >= 2)
        {
          if (szFile[1] == ':')
            {
              szLongName[0] = (CHAR)CharUpperA ((LPSTR)szFile[0]);
              szLongName[1] = ':';
              if (G_IS_DIR_SEPARATOR (szFile[2]))
                {
                  szLongName[2] = G_DIR_SEPARATOR;
                  i = 3;
                }
              else
                {
                  i = 2;
                }
            }
          else if (G_IS_DIR_SEPARATOR (szFile[0])
                                            && G_IS_DIR_SEPARATOR (szFile[1]))
            {
              i = 2;
              while (i < dwSize)
                {
                  if (G_IS_DIR_SEPARATOR (szFile[i]))
                    break;
                  i += IsDBCSLeadByteEx (CP_ACP, szFile[i]) ? 2 : 1;
                }
              i++;
              g_memmove (szLongName, szFile, i * sizeof (CHAR));
            }
        }
      szLongName[i] = '\0';
      j = i;
      while (i <= dwSize)
        {
          if (szFile[i] == '\0' || G_IS_DIR_SEPARATOR (szFile[i]))
            {
              WIN32_FIND_DATAA rcf;

              s = lstrlenA (szLongName);
              g_memmove (szLongName + s, szFile + j, (i - j) * sizeof (CHAR));
              szLongName[s + i - j] = '\0';
              hFind = FindFirstFileA (szLongName, &rcf);
              if (hFind != INVALID_HANDLE_VALUE)
                FindClose (hFind);
              else
                lstrcpyA (rcf.cFileName, ".");
              if (lstrcmpA (rcf.cFileName, "."))
                {
                  g_memmove (szLongName + s, rcf.cFileName,
                                    lstrlenA (rcf.cFileName) * sizeof (CHAR));
                  s += lstrlenA (rcf.cFileName);
                }
              else
                {
                  s += i - j;
                }
              if (G_IS_DIR_SEPARATOR (szFile[i]))
                szLongName[s++] = G_DIR_SEPARATOR;
              szLongName[s] = '\0';
              j = i + 1;
            }
          i += IsDBCSLeadByteEx (CP_ACP, szFile[i]) ? 2 : 1;
        }
      utf8str = g_locale_to_utf8 (szLongName, -1, NULL, NULL, NULL);
# endif /* GLIB_CHECK_VERSION(2,6,0) || ! defined (UNICODE) */
# if GLIB_CHECK_VERSION(2,6,0)
    }
  if (!utf8str)
    return g_strdup (file);
  filename = g_filename_from_utf8 (utf8str, -1, NULL, NULL, NULL);
  g_free (utf8str);
  return filename ? filename : g_strdup (file);
# endif /* GLIB_CHECK_VERSION(2,6,0) */
#else /* not G_OS_WIN32 */
  gchar *absolate;
  gint i, leng;

  if (!file)
    return NULL;
  if (!G_IS_DIR_SEPARATOR (file[0]))
    {
      if (path && g_path_is_absolute (path))
        {
          leng = g_strlen (path);
          absolate = leng > 0 && G_IS_DIR_SEPARATOR (path[leng - 1])
                        ? g_strconcat (path, file, NULL)
                        : g_strconcat (path, G_DIR_SEPARATOR_S, file, NULL);
        }
      else
        {
          gchar *dir;

          dir = g_get_current_dir ();
          absolate = g_strconcat (dir, G_DIR_SEPARATOR_S, file, NULL);
          g_free (dir);
        }
    }
  else
    {
      absolate = g_strdup (file);
    }
  leng = g_strlen (absolate);
  i = 0;
  while (i < leng)
    if (G_IS_DIR_SEPARATOR (absolate[i])
                                    && G_IS_DIR_SEPARATOR (absolate[i + 1]))
      {
        leng--;
        g_memmove (absolate + i + 1, absolate + i + 2, leng - i);
      }
    else
      {
        i++;
      }
  i = 0;
  while (i < leng)
    if (G_IS_DIR_SEPARATOR (absolate[i]) && absolate[i + 1] == '.'
                                    && G_IS_DIR_SEPARATOR (absolate[i + 2]))
      {
        leng -= 2;
        g_memmove (absolate + i + 1, absolate + i + 3, leng - i);
      }
    else
      {
        i++;
      }
  i = 0;
  while (i < leng)
    if (G_IS_DIR_SEPARATOR (absolate[i]) && absolate[i + 1] == '.'
            && absolate[i + 2] == '.' && G_IS_DIR_SEPARATOR (absolate[i + 3]))
      {
        gint j;

        leng -= 3;
        g_memmove (absolate + i + 1, absolate + i + 4, leng - i);
        for (j = i - 1; j >= 0; j--)
          if (G_IS_DIR_SEPARATOR (absolate[j]))
            break;
        if (j >= 0)
          {
            g_memmove (absolate + j + 1, absolate + i + 1, leng - i);
            leng -= i - j;
            i = j;
          }
      }
    else
      {
        i++;
      }
  return absolate;
#endif /* not G_OS_WIN32 */
}


/*  ja:ファイルの相対パスを取得する
    file,ファイル名(終点)
    path,パス名(起点)
     RET,フルパス                                                           */
gchar *
fileio_get_relative_path (const gchar *file,
                          const gchar *path)
{
  gchar *absolate, *relative = NULL, *name, *tmp;
  gchar *path_to, *path_from, **ary_to, **ary_from;
  gint i, j;

  if (!file)
    return NULL;
  /* ja:終点 */
  absolate = fileio_get_full_path (file);
  path_to = g_path_get_dirname (absolate);
  g_free (absolate);
  ary_to = g_strsplit (path_to, G_DIR_SEPARATOR_S, 0);
  /* ja:起点 */
  path_from = path ? fileio_get_full_path (path) : g_get_current_dir ();
  ary_from = g_strsplit (path_from, G_DIR_SEPARATOR_S, 0);
  if (!ary_to || !ary_from || !ary_to[0] || !ary_from[0]
#ifdef G_OS_WIN32
                              || g_strfilecmp (ary_to[0], ary_from[0]) != 0
    || (G_IS_DIR_SEPARATOR (path_to[0]) && G_IS_DIR_SEPARATOR (path_to[1])
    && G_IS_DIR_SEPARATOR (path_from[0]) && G_IS_DIR_SEPARATOR (path_from[1])
                                && g_strfilecmp (ary_to[2], ary_from[2]) != 0)
#endif /* G_OS_WIN32 */
                                                                            )
    {
      g_free (path_to);
      g_free (path_from);
      g_strfreev (ary_to);
      g_strfreev (ary_from);
      return NULL;
    }
  g_free (path_to);
  g_free (path_from);
  for (i = 0; ary_to[i] && ary_from[i]; i++)
    if (g_strfilecmp (ary_to[i], ary_from[i]) != 0)
      break;
  for (j = i; ary_from[j]; j++)
    if (*(ary_from[j]) != '\0')
      {
        tmp = relative ? g_strconcat (relative, ".."G_DIR_SEPARATOR_S, NULL)
                       : g_strdup (".."G_DIR_SEPARATOR_S);
        g_free (relative);
        relative = tmp;
      }
  for (j = i; ary_to[j]; j++)
    {
      tmp = relative
                ? g_strconcat (relative, ary_to[j], G_DIR_SEPARATOR_S, NULL)
                : g_strconcat (ary_to[j], G_DIR_SEPARATOR_S, NULL);
      g_free (relative);
      relative = tmp;
    }
  g_strfreev (ary_to);
  g_strfreev (ary_from);
  name = g_path_get_basename (file);
  if (name)
    {
      tmp = relative ? g_strconcat (relative, name, NULL) : g_strdup (name);
      g_free (relative);
      relative = tmp;
      g_free (name);
    }
  return relative;
}


/******************************************************************************
*                                                                             *
* ja:低レベルファイル入出力関数群                                             *
*                                                                             *
******************************************************************************/
#ifndef G_OS_WIN32
static GList *glist_fileio = NULL;
# ifdef USE_THREAD
G_LOCK_DEFINE_STATIC (critical);
static volatile gboolean critical = FALSE;
# endif /* USE_THREAD */
#endif /* not G_OS_WIN32 */


/*  ja:ファイルの作成
      file,ファイル名
    access,アクセス
     share,共有
      mode,モード
       RET,ファイルポインタ,NULL:エラー                                     */
FileIO *
fileio_open (const gchar *file,
             const guint access,
             const guint share,
             const guint mode)
{
  gchar *path;
  FileIO *fio;
#ifdef G_OS_WIN32
  HANDLE hFile;
# if ! GLIB_CHECK_VERSION(2,6,0)
  LPTSTR lpszFile;
#  ifdef UNICODE
  gchar *utf8str;
#  endif /* UNICODE */
# endif /* not GLIB_CHECK_VERSION(2,6,0) */
#else /* not G_OS_WIN32 */
  int fd, flags;
  gint i;
  guint length;
#endif /* not G_OS_WIN32 */

  if (!file)
    return NULL;
  /* ja:フルパスを求める */
  path = fileio_get_full_path (file);
  if (!path)
    return NULL;
  if (mode != FILEIO_MODE_OPEN_EXISTING)
    {
      gchar *dir;

      dir = g_path_get_dirname (path);
      if (!g_file_test (dir, G_FILE_TEST_EXISTS) && !fileio_mkdir (dir))
        {
          g_free (path);
          g_free (dir);
          return NULL;
        }
      g_free (dir);
    }
#ifdef G_OS_WIN32
  g_free (path);
# if GLIB_CHECK_VERSION(2,6,0)
  if (G_WIN32_HAVE_WIDECHAR_API ())
    {
      gchar *utf8str;
      LPWSTR lpszFile;
# endif /* GLIB_CHECK_VERSION(2,6,0) */

# if GLIB_CHECK_VERSION(2,6,0) || defined (UNICODE)
      utf8str = g_filename_to_utf8 (file, -1, NULL, NULL, NULL);
      if (!utf8str)
        return NULL;
      lpszFile = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
      hFile = CreateFileW (lpszFile, access, share, NULL, mode,
                                                FILE_ATTRIBUTE_NORMAL, NULL);
      g_free (lpszFile);
# endif /* GLIB_CHECK_VERSION(2,6,0) || defined (UNICODE) */
# if GLIB_CHECK_VERSION(2,6,0)
    }
  else
    {
      LPSTR lpszFile;
# endif /* GLIB_CHECK_VERSION(2,6,0) */

# if GLIB_CHECK_VERSION(2,6,0) || ! defined (UNICODE)
#  if GLIB_CHECK_VERSION(2,8,0)
      lpszFile = g_win32_locale_filename_from_utf8 (file);
#  elif GLIB_CHECK_VERSION(2,6,0)
      lpszFile = g_locale_from_utf8 (file, -1, NULL, NULL, NULL);
#  else /* not GLIB_CHECK_VERSION(2,6,0) */
      lpszFile = g_strdup (file);
#  endif /* not GLIB_CHECK_VERSION(2,6,0) */
      hFile = CreateFileA (lpszFile, access, share, NULL, mode,
                                                FILE_ATTRIBUTE_NORMAL, NULL);
      g_free (lpszFile);
# endif /* GLIB_CHECK_VERSION(2,6,0) || ! defined (UNICODE) */
# if GLIB_CHECK_VERSION(2,6,0)
    }
# endif /* GLIB_CHECK_VERSION(2,6,0) */
  if (hFile == INVALID_HANDLE_VALUE)
    return NULL;
  fio = g_malloc (sizeof (FileIO));
  fio->counter = 1;
  fio->hFile = hFile;
#else /* not G_OS_WIN32 */
  /* ja:フラグ設定 */
  switch (access)
    {
      case FILEIO_ACCESS_READ:  flags = O_RDONLY;   break;
      case FILEIO_ACCESS_WRITE: flags = O_WRONLY;   break;
      case FILEIO_ACCESS_ALL:   flags = O_RDWR;     break;
      default:                  g_free (path);      return NULL;
    }
  switch (mode)
    {
      case FILEIO_MODE_CREATE_NEW:      /* ja:無ければ新規作成,あれば失敗 */
        flags |= O_CREAT | O_EXCL;
        break;
      case FILEIO_MODE_CREATE_ALWAYS:   /* ja:常に新規作成 */
        flags |= O_CREAT;
        break;
      case FILEIO_MODE_OPEN_EXISTING:   /* ja:あれば開く,無ければ失敗 */
        break;
      case FILEIO_MODE_OPEN_ALWAYS:     /* ja:あれば開く,無ければ新規作成 */
        flags |= O_CREAT;
        break;
      default:
        g_free (path);
        return NULL;
    }
  /* ja:共有をチェック */
# ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
# endif /* USE_THREAD */
  length = g_list_length (glist_fileio);
  for (i = 0; i < length; i++)
    {
      fio = g_list_nth_data (glist_fileio, i);
      if (g_strfilecmp (fio->file, path) == 0
                                            && (fio->share & access) != access)
        break;
    }
# ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
# endif /* USE_THREAD */
  if (i < length)
    {
      g_free (path);
      return NULL;
    }

  fd = open (path, flags,
                    S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH | S_IWOTH);
  if (fd == -1)
    {
      g_free (path);
      return NULL;
    }
  if (
# ifdef HAVE_FLOCK
      flock (fd,
        (access == FILEIO_ACCESS_READ ? LOCK_SH : LOCK_EX) | LOCK_NB) != 0 ||
# endif /* HAVE_FLOCK */
        (mode == FILEIO_MODE_CREATE_ALWAYS && (
# ifdef HAVE_FTRUNCATE
                    ftruncate (fd, 0) || 
# endif /* HAVE_FTRUNCATE */
                    lseek (fd, 0, SEEK_SET) == -1)))
    {
      close (fd);
      g_free (path);
      return NULL;
    }
  fio = g_malloc (sizeof (FileIO));
  fio->counter = 1;
  fio->fd = fd;
  fio->file = path;
  fio->share = share;
# ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
# endif /* USE_THREAD */
  glist_fileio = g_list_append (glist_fileio, fio);
# ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
# endif /* USE_THREAD */
#endif /* not G_OS_WIN32 */
  return fio;
}


/*  ja:ファイルポインタの参照数を増やす
    fio,ファイルポインタ
    RET,TRUE:正常終了,FALSE:エラー                                          */
gboolean
fileio_ref (FileIO *fio)
{
#ifdef G_OS_WIN32
  return fio ? fio->counter++, TRUE : FALSE;
#else /* not G_OS_WIN32 */
  gboolean result = FALSE;

# ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
# endif /* USE_THREAD */
  if (fio && g_list_find (glist_fileio, fio))
    {
      fio->counter++;
      result = TRUE;
    }
# ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
# endif /* USE_THREAD */
  return result;
#endif /* not G_OS_WIN32 */
}


/*  ja:ファイルポインタの参照数を減らす
    fio,ファイルポインタ
    RET,TRUE:正常終了,FALSE:エラー                                          */
gboolean
fileio_unref (FileIO *fio)
{
  gboolean result = FALSE;

#ifdef G_OS_WIN32
  if (fio)
    {
      fio->counter--;
      if (fio->counter <= 0)
        {
          result = CloseHandle (fio->hFile);
          g_free (fio);
        }
      else
        {
          result = TRUE;
        }
    }
#else /* not G_OS_WIN32 */
# ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
# endif /* USE_THREAD */
  if (fio && g_list_find (glist_fileio, fio))
    {
      fio->counter--;
      if (fio->counter <= 0)
        {
          glist_fileio = g_list_remove (glist_fileio, fio);
          result = close (fio->fd) == 0;
          g_free (fio->file);
          g_free (fio);
        }
      else
        {
          result = TRUE;
        }
    }
# ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
# endif /* USE_THREAD */
#endif /* not G_OS_WIN32 */
  return result;
}


/*  ja:ファイルから読み込む
       fio,ファイルポインタ
      data,バッファ
    length,バイト数
       RET,バイト数,-1:エラー                                               */
gssize
fileio_read (FileIO       *fio,
             gpointer      data,
             const gssize  length)
{
#ifdef G_OS_WIN32
  DWORD dwRead;

  return fio && ReadFile (fio->hFile, data, length, &dwRead, NULL)
                                                                ? dwRead : -1;
#else /* not G_OS_WIN32 */
  gssize result;

# ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
# endif /* USE_THREAD */
  result = fio && g_list_find (glist_fileio, fio) != NULL
                                        ? read (fio->fd, data, length) : -1;
# ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
# endif /* USE_THREAD */
  return result;
#endif /* not G_OS_WIN32 */
}


/*  ja:ファイルに書き込む
       fio,ファイルポインタ
      data,バッファ
    length,バイト数
       RET,バイト数,-1:エラー                                               */
gssize
fileio_write (FileIO        *fio,
              gconstpointer  data,
              const gssize   length)
{
#ifdef G_OS_WIN32
  DWORD dwWrite;

  return fio && WriteFile (fio->hFile, data, length, &dwWrite, NULL)
                                                                ? dwWrite : -1;
#else /* not G_OS_WIN32 */
  gssize result;

# ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
# endif /* USE_THREAD */
  result = fio && g_list_find (glist_fileio, fio)
                                        ? write (fio->fd, data, length) : -1;
# ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
# endif /* USE_THREAD */
  return result;
#endif /* not G_OS_WIN32 */
}


/*  ja:ファイルをシークする
       fio,ファイルポインタ
    offset,位置
    whence,モード
       RET,位置,-1:エラー                                                   */
goffset
fileio_seek (FileIO        *fio,
             const goffset  offset,
             const guint    whence)
{
#ifdef G_OS_WIN32
# ifdef G_HAVE_GINT64
  LONG lHigh, lLow;

  if (!fio)
    return -1;
  lHigh = offset >> 32;
  lLow = SetFilePointer (fio->hFile, (LONG)offset, &lHigh, whence);
  return lLow != 0xFFFFFFFF || GetLastError () == NO_ERROR
                                ? ((gint64)lHigh << 32) | (guint32)lLow : -1;
# else /* not G_HAVE_GINT64 */
  return SetFilePointer (fio->hFile, offset, NULL, whence);
# endif /* not G_HAVE_GINT64 */
#else /* not G_OS_WIN32 */
  gssize result;

# ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
# endif /* USE_THREAD */
  result = fio && g_list_find (glist_fileio, fio)
                                        ? lseek (fio->fd, offset, whence) : -1;
# ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
# endif /* USE_THREAD */
  return result;
#endif /* not G_OS_WIN32 */
}


/*  ja:ファイルサイズを求める
    fio,ファイルポインタ
    RET,ファイルサイズ,-1:エラー                                            */
goffset
fileio_size (FileIO *fio)
{
#ifdef G_OS_WIN32
# ifdef G_HAVE_GINT64
  DWORD dwHigh, dwLow;

  if (!fio)
    return -1;
  dwLow = GetFileSize (fio->hFile, &dwHigh);
  return dwLow != 0xFFFFFFFF || GetLastError () == NO_ERROR
                                ? ((gint64)dwHigh << 32) | (guint32)dwLow : -1;
# else /* not G_HAVE_GINT64 */
  return GetFileSize (fio->hFile, NULL);
# endif /* not G_HAVE_GINT64 */
#else /* not G_OS_WIN32 */
  goffset offset, leng;

  if (!fio)
    return -1;
  offset = fileio_seek (fio, 0, FILEIO_SEEK_CUR);
  if (offset < 0)
    return -1;
  leng = fileio_seek (fio, 0, FILEIO_SEEK_END);
  return fileio_seek (fio, offset, FILEIO_SEEK_SET) < 0 ? -1 : leng;
#endif /* not G_OS_WIN32 */
}


/******************************************************************************
*                                                                             *
* ja:高レベルファイル入出力関数群                                             *
*                                                                             *
******************************************************************************/
/*  ja:ファイルをメモリに読み込む
      file,ファイル名
    length,ファイルサイズ
       RET,メモリ,NULL:エラー                                               */
gpointer
fileio_load (const gchar *file,
             gssize      *length)
{
  gpointer data;
  gssize leng;
  FileIO *fio;

  fio = fileio_open (file, FILEIO_ACCESS_READ, FILEIO_SHARE_READ,
                                                    FILEIO_MODE_OPEN_EXISTING);
  if (!fio || (leng = fileio_size (fio)) < 0)
    return NULL;
  data = g_malloc (leng);
  if (data && fileio_read (fio, data, leng) != leng)
    {
      g_free (data);
      data = NULL;
    }
  if (!fileio_close (fio))
    {
      g_free (data);
      data = NULL;
    }
  if (data && length)
    *length = leng;
  return data;
}


/*  ja:ファイルにメモリから書き込む
      file,ファイル名
      data,メモリ
    length,サイズ
       RET,TRUE:正常終了,FALSE:エラー                                       */
gboolean
fileio_save (const gchar   *file,
             gconstpointer  data,
             const gssize   length)
{
  gboolean result;
  FileIO *fio;

  fio = fileio_open (file, FILEIO_ACCESS_WRITE, FILEIO_SHARE_READ,
                                                    FILEIO_MODE_CREATE_ALWAYS);
  if (fio)
    {
      result = fileio_write (fio, data, length) == length;
      if (!fileio_close (fio))
        result = FALSE;
    }
  else
    {
      result = FALSE;
    }
  return result;
}


/*  ja:1文字読み込む
    fio,ファイルポインタ
    RET,文字,EOF:エラー                                                     */
gint
fileio_getc (FileIO *fio)
{
  gchar c;

  return fileio_read (fio, &c, sizeof (gchar)) == sizeof (gchar) ? c : EOF;
}


/*  ja:1文字書き込む
      c,文字
    fio,ファイルポインタ
    RET,文字,EOF:エラー                                                     */
gint
fileio_putc (gint    c,
             FileIO *fio)
{
  return fileio_write (fio, &c, sizeof (gchar)) == sizeof (gchar) ? c : EOF;
}


/*  ja:1行読み込む
      data,バッファ
    length,バイト数
       fio,ファイルポインタ
       RET,data:正常終了,NULL:エラー                                        */
gchar *
fileio_gets (gchar       *data,
             const gsize  length,
             FileIO      *fio)
{
  gchar c;
  gsize i = 0;
  gssize result;

  if (!data || length <= 0)
    return NULL;
  while (i <= length - 2)
    {
      result = fileio_read (fio, &c, sizeof (gchar));
      if (result < 0)
        return NULL;
      if (result < sizeof (gchar))
        break;
      data[i++] = c;
      if (c == '\n')
        break;
    }
  if (i == 0)
    return NULL;
  data[i] = '\0';
  return data;
}


/*  ja:1行書き込む
      data,バッファ
    length,バイト数
       fio,ファイルポインタ
       RET,正数:正常終了,EOF:エラー                                         */
gint
fileio_puts (const gchar *data,
             FileIO      *fio)
{
  gsize i;
  gssize result;

  if (!data)
    return EOF;
  for (i = 0; data[i] != '\0'; i++);
  result = fileio_write (fio, data, i * sizeof (gchar));
  return result >= 0 ? result : EOF;
}


/******************************************************************************
*                                                                             *
* ja:ディレクトリ関数群                                                       *
*                                                                             *
******************************************************************************/
/*  ja:ディレクトリを作る
    dir,ディレクトリ名
    RET,TRUE:正常終了,FALSE,エラー                                          */
gboolean
fileio_mkdir (const gchar *dir)
{
  gboolean result;
  gchar *path, *parent, *utf8str;
#if ! GLIB_CHECK_VERSION(2,6,0) && defined (G_OS_WIN32)
  LPTSTR lpszPath;
#endif /* ! GLIB_CHECK_VERSION(2,6,0) && defined (G_OS_WIN32) */

  if (!dir)
    return FALSE;
  path = fileio_get_full_path (dir);
  if (!path)
    return FALSE;
  parent = g_path_get_dirname (path);
  if (!parent)
    {
      g_free (path);
      return FALSE;
    }
  /* ja:parentの末尾がG_DIR_SEPARATORならば下はない */
  utf8str = g_filename_to_utf8 (parent, -1, NULL, NULL, NULL);
  if (!utf8str)
    {
      g_free (path);
      g_free (parent);
      return FALSE;
    }
  result = G_IS_DIR_SEPARATOR (utf8str[g_strlen (utf8str) - 1]);
  g_free (utf8str);
  if (result)
    {
      g_free (path);
      g_free (parent);
      return TRUE;
    }
  result = !g_file_test (parent, G_FILE_TEST_EXISTS) && !fileio_mkdir (parent);
  g_free (parent);
  if (result)
    {
      g_free (path);
      return FALSE;
    }
#ifdef G_OS_WIN32
# if GLIB_CHECK_VERSION(2,6,0)
  if (G_WIN32_HAVE_WIDECHAR_API ())
    {
      LPWSTR lpszPath;
# endif /* GLIB_CHECK_VERSION(2,6,0) */

# if GLIB_CHECK_VERSION(2,6,0) || defined (UNICODE)
      utf8str = g_filename_to_utf8 (path, -1, NULL, NULL, NULL);
      if (!utf8str)
        {
          g_free (path);
          return FALSE;
        }
      lpszPath = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
      g_free (utf8str);
      result = CreateDirectoryW (lpszPath, NULL);
      g_free (lpszPath);
# endif /* GLIB_CHECK_VERSION(2,6,0) || defined (UNICODE) */
# if GLIB_CHECK_VERSION(2,6,0)
    }
  else
    {
      LPSTR lpszPath;
# endif /* GLIB_CHECK_VERSION(2,6,0) */

# if GLIB_CHECK_VERSION(2,6,0) || ! defined (UNICODE)
#  if GLIB_CHECK_VERSION(2,8,0)
      lpszPath = g_win32_locale_filename_from_utf8 (path);
#  elif GLIB_CHECK_VERSION(2,6,0)
      lpszPath = g_locale_from_utf8 (path, -1, NULL, NULL, NULL);
#  else /* not GLIB_CHECK_VERSION(2,6,0) */
      lpszPath = g_strdup (path);
#  endif /* not GLIB_CHECK_VERSION(2,6,0) */
      result = CreateDirectoryA (lpszPath, NULL);
      g_free (lpszPath);
# endif /* GLIB_CHECK_VERSION(2,6,0) || ! defined (UNICODE) */
# if GLIB_CHECK_VERSION(2,6,0)
    }
# endif /* GLIB_CHECK_VERSION(2,6,0) */
#else /* not G_OS_WIN32 */
# if GLIB_CHECK_VERSION(2,6,0)
  result = g_mkdir (path, S_IRUSR | S_IWUSR | S_IXUSR | S_IRGRP | S_IXGRP
                                                    | S_IROTH | S_IXOTH) == 0;
# else /* not GLIB_CHECK_VERSION(2,6,0) */
  result = mkdir (path, S_IRUSR | S_IWUSR | S_IXUSR | S_IRGRP | S_IXGRP
                                                    | S_IROTH | S_IXOTH) == 0;
# endif /* not GLIB_CHECK_VERSION(2,6,0) */
#endif /* not G_OS_WIN32 */
  g_free (path);
  return result;
}


/******************************************************************************
*                                                                             *
* ja:ファイル時刻関数群                                                       *
*                                                                             *
******************************************************************************/
/*  ja:ファイルの時刻を取得する
     file,ファイル名
    atime,最終アクセス時刻
    mtime,最終修正時刻
      RET,TRUE:正常終了,FALSE,エラー                                        */
gboolean
fileio_gettime (const gchar *file,
                FileIOTime  *atime,
                FileIOTime  *mtime)
{
#ifdef G_OS_WIN32
  HANDLE hFind;
# if ! GLIB_CHECK_VERSION(2,6,0)
  LPTSTR lpszFile;
  WIN32_FIND_DATA rcFind;
#  ifdef UNICODE
  gchar *utf8str;
#  endif /* UNICODE */
# endif /* not GLIB_CHECK_VERSION(2,6,0) */
#else /* not G_OS_WIN32 */
  struct stat buf;
#endif /* not G_OS_WIN32 */

  if (!file)
    return FALSE;
#ifdef G_OS_WIN32
# if GLIB_CHECK_VERSION(2,6,0)
  if (G_WIN32_HAVE_WIDECHAR_API ())
    {
      gchar *utf8str;
      LPWSTR lpszFile;
      WIN32_FIND_DATAW rcFind;
# endif /* GLIB_CHECK_VERSION(2,6,0) */

# if GLIB_CHECK_VERSION(2,6,0) || defined (UNICODE)
      utf8str = g_filename_to_utf8 (file, -1, NULL, NULL, NULL);
      if (!utf8str)
        return FALSE;
      lpszFile = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
      g_free (utf8str);
      if (!lpszFile)
        return FALSE;
      hFind = FindFirstFileW (lpszFile, &rcFind);
      g_free (lpszFile);
      if (hFind == INVALID_HANDLE_VALUE || !FindClose (hFind))
        return FALSE;
      if (atime)
        *atime = rcFind.ftLastAccessTime;
      if (mtime)
        *mtime = rcFind.ftLastWriteTime;
# endif /* GLIB_CHECK_VERSION(2,6,0) || defined (UNICODE) */
# if GLIB_CHECK_VERSION(2,6,0)
    }
  else
    {
      LPSTR lpszFile;
      WIN32_FIND_DATAA rcFind;
# endif /* GLIB_CHECK_VERSION(2,6,0) */

# if GLIB_CHECK_VERSION(2,6,0) || !defined (UNICODE)
#  if GLIB_CHECK_VERSION(2,8,0)
      lpszFile = g_win32_locale_filename_from_utf8 (file);
#  elif GLIB_CHECK_VERSION(2,6,0)
      lpszFile = g_locale_from_utf8 (file, -1, NULL, NULL, NULL);
#  else /* not GLIB_CHECK_VERSION(2,6,0) */
      lpszFile = g_strdup (file);
#  endif /* not GLIB_CHECK_VERSION(2,6,0) */
      if (!lpszFile)
        return FALSE;
      hFind = FindFirstFileA (lpszFile, &rcFind);
      g_free (lpszFile);
      if (hFind == INVALID_HANDLE_VALUE || !FindClose (hFind))
        return FALSE;
      if (atime)
        *atime = rcFind.ftLastAccessTime;
      if (mtime)
        *mtime = rcFind.ftLastWriteTime;
# endif /* GLIB_CHECK_VERSION(2,6,0) || !defined (UNICODE) */
# if GLIB_CHECK_VERSION(2,6,0)
    }
# endif /* GLIB_CHECK_VERSION(2,6,0) */
#else /* not G_OS_WIN32 */
  if (stat (file, &buf))
    return FALSE;
  if (atime)
    *atime = buf.st_atime;
  if (mtime)
    *mtime = buf.st_mtime;
#endif /* not G_OS_WIN32 */
  return TRUE;
}


/*  ja:ファイルの時刻を設定する
     file,ファイル名
    atime,最終アクセス時刻
    mtime,最終修正時刻
      RET,TRUE:正常終了,FALSE,エラー                                        */
gboolean
fileio_settime (const gchar *file,
                FileIOTime  *atime,
                FileIOTime  *mtime)
{
#ifdef G_OS_WIN32
  HANDLE hFile;
# if ! GLIB_CHECK_VERSION(2,6,0)
  LPTSTR lpszFile;
#  ifdef UNICODE
  gchar *utf8str;
#  endif /* UNICODE */
# endif /* not GLIB_CHECK_VERSION(2,6,0) */
#else /* not G_OS_WIN32 */
  struct stat sbuf;
  struct utimbuf ubuf;
#endif /* not G_OS_WIN32 */

  if (!file)
    return FALSE;
#ifdef G_OS_WIN32
# if GLIB_CHECK_VERSION(2,6,0)
  if (G_WIN32_HAVE_WIDECHAR_API ())
    {
      gchar *utf8str;
      LPWSTR lpszFile;
# endif /* GLIB_CHECK_VERSION(2,6,0) */

# if GLIB_CHECK_VERSION(2,6,0) || defined (UNICODE)
      utf8str = g_filename_to_utf8 (file, -1, NULL, NULL, NULL);
      if (!utf8str)
        return FALSE;
      lpszFile = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
      g_free (utf8str);
      hFile = CreateFileW (lpszFile, GENERIC_WRITE,
                                FILE_SHARE_READ | FILE_SHARE_WRITE, NULL,
                                OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
      g_free (lpszFile);
# endif /* GLIB_CHECK_VERSION(2,6,0) || defined (UNICODE) */
# if GLIB_CHECK_VERSION(2,6,0)
    }
  else
    {
      LPSTR lpszFile;
# endif /* GLIB_CHECK_VERSION(2,6,0) */

# if GLIB_CHECK_VERSION(2,6,0) || ! defined (UNICODE)
#  if GLIB_CHECK_VERSION(2,8,0)
      lpszFile = g_win32_locale_filename_from_utf8 (file);
#  elif GLIB_CHECK_VERSION(2,6,0)
      lpszFile = g_locale_from_utf8 (file, -1, NULL, NULL, NULL);
#  else /* not GLIB_CHECK_VERSION(2,6,0) */
      lpszFile = g_strdup (file);
#  endif /* not GLIB_CHECK_VERSION(2,6,0) */
      hFile = CreateFileA (lpszFile, GENERIC_WRITE,
                                FILE_SHARE_READ | FILE_SHARE_WRITE, NULL,
                                OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
      g_free (lpszFile);
# endif /* GLIB_CHECK_VERSION(2,6,0) || ! defined (UNICODE) */
# if GLIB_CHECK_VERSION(2,6,0)
    }
# endif /* GLIB_CHECK_VERSION(2,6,0) */
  if (hFile == INVALID_HANDLE_VALUE)
    return FALSE;
  if (!SetFileTime (hFile, NULL, atime, mtime))
    {
      CloseHandle (hFile);
      return FALSE;
    }
  return CloseHandle (hFile);
#else /* not G_OS_WIN32 */
# ifdef HAVE_UTIME
  if (stat (file, &sbuf))
    return FALSE;
  ubuf.actime  = atime ? *atime : sbuf.st_atime;
  ubuf.modtime = mtime ? *mtime : sbuf.st_mtime;
  return !utime (file, &ubuf);
# else /* not HAVE_UTIME */
  return FALSE;
# endif /* not HAVE_UTIME */
#endif /* not G_OS_WIN32 */
}


/*  ja:ファイルの時刻を変換する
    ftime,ファイルの時刻
    stime,システム時間                                                      */
void
fileio_getsystime (FileIOTime    *ftime,
                   FileIOSysTime *stime)
{
#ifdef G_OS_WIN32
  SYSTEMTIME st;

  FileTimeToSystemTime (ftime, &st);
  stime->year = st.wYear;
  stime->mon  = st.wMonth;
  stime->mday = st.wDay;
  stime->hour = st.wHour;
  stime->min  = st.wMinute;
  stime->sec  = st.wSecond;
  stime->wday = st.wDayOfWeek;
#else /* not G_OS_WIN32 */
  struct tm *gmt;

  gmt = gmtime (ftime);
  stime->year = gmt->tm_year + 1900;
  stime->mon  = gmt->tm_mon + 1;
  stime->mday = gmt->tm_mday;
  stime->hour = gmt->tm_hour;
  stime->min  = gmt->tm_min;
  stime->sec  = gmt->tm_sec;
  stime->wday = gmt->tm_wday;
#endif /* not G_OS_WIN32 */
}


/******************************************************************************
*                                                                             *
* ja:応用ファイル関数群                                                       *
*                                                                             *
******************************************************************************/
/*  ja:ファイルのアイコンを取得する
         file,ファイル名
    icon_size,FILEIO_ICON_SIZE_SMALL/FILEIO_ICON_SIZE_LARGE
          RET,pixbuf,NULL:エラー                                            */
GdkPixbuf *
fileio_extract_icon (const gchar *file,
                     const guint  icon_size)
{
#if defined (G_OS_WIN32)
  GdkPixbuf *pixbuf = NULL;
  HICON hIcon = NULL;
  ICONINFO ii;
# if ! GLIB_CHECK_VERSION(2,6,0)
  LPTSTR lpszFile;
  SHFILEINFO shfi;
#  ifdef UNICODE
  gchar *utf8str;
#  endif /* UNICODE */
# endif /* not GLIB_CHECK_VERSION(2,6,0) */

  if (!file)
    return NULL;
# if GLIB_CHECK_VERSION(2,6,0)
  if (G_WIN32_HAVE_WIDECHAR_API ())
    {
      gchar *utf8str;
      LPWSTR lpszFile;
      SHFILEINFOW shfi;
# endif /* GLIB_CHECK_VERSION(2,6,0) */

# if GLIB_CHECK_VERSION(2,6,0) || defined (UNICODE)
      utf8str = g_filename_to_utf8 (file, -1, NULL, NULL, NULL);
      if (!utf8str)
        return NULL;
      lpszFile = g_utf8_to_utf16 (file, -1, NULL, NULL, NULL);
      if ((BOOL)SHGetFileInfoW (lpszFile, 0, &shfi, sizeof (SHFILEINFOW),
                                        SHGFI_ICON | SHGFI_USEFILEATTRIBUTES
                                        | (icon_size == FILEIO_ICON_SIZE_SMALL
                                        ? SHGFI_SMALLICON : SHGFI_LARGEICON)))
        hIcon = shfi.hIcon;
      g_free (lpszFile);
# endif /* GLIB_CHECK_VERSION(2,6,0) || defined (UNICODE) */
# if GLIB_CHECK_VERSION(2,6,0)
    }
  else
    {
      LPSTR lpszFile;
      SHFILEINFOA shfi;
# endif /* GLIB_CHECK_VERSION(2,6,0) */

# if GLIB_CHECK_VERSION(2,6,0) || defined (UNICODE)
#  if GLIB_CHECK_VERSION(2,8,0)
      lpszFile = g_win32_locale_filename_from_utf8 (file);
#  elif GLIB_CHECK_VERSION(2,6,0)
      lpszFile = g_locale_from_utf8 (file, -1, NULL, NULL, NULL);
#  else /* not GLIB_CHECK_VERSION(2,8,0) */
      lpszFile = g_strdup (file);
#  endif /* not GLIB_CHECK_VERSION(2,8,0) */
      if ((BOOL)SHGetFileInfoA (lpszFile, 0, &shfi, sizeof (SHFILEINFOW),
                                        SHGFI_ICON | SHGFI_USEFILEATTRIBUTES
                                        | (icon_size == FILEIO_ICON_SIZE_SMALL
                                        ? SHGFI_SMALLICON : SHGFI_LARGEICON)))
        hIcon = shfi.hIcon;
      g_free (lpszFile);
# endif /* GLIB_CHECK_VERSION(2,6,0) || defined (UNICODE) */
    }
  if (GetIconInfo (hIcon, &ii))
    {
      BITMAPINFOHEADER bmih;
      HDC hDC;

      g_memset (&bmih, 0, sizeof (BITMAPINFOHEADER));
      bmih.biSize = sizeof (BITMAPINFOHEADER);
      hDC = CreateCompatibleDC (NULL);
      if (GetDIBits (hDC, ii.hbmColor, 0, 1,
                                    NULL, (LPBITMAPINFO)&bmih, DIB_RGB_COLORS))
        {
          guchar *pixels;
          gint rowstride, x, y, w, h;
          gboolean alpha = FALSE;
          LPBYTE lpbBits;

          w = bmih.biWidth;
          h = bmih.biHeight;
          bmih.biBitCount = 32;
          bmih.biCompression = BI_RGB;
          bmih.biHeight = -bmih.biHeight;
          pixbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, TRUE, 8, w, h);
          lpbBits = g_malloc0 (w * h * 4);
          /* en:color data */
          GetDIBits (hDC, ii.hbmColor, 0, h,
                                lpbBits, (LPBITMAPINFO)&bmih, DIB_RGB_COLORS);
          pixels = gdk_pixbuf_get_pixels (pixbuf);
          rowstride = gdk_pixbuf_get_rowstride (pixbuf);
          for (y = 0; y < h; y++)
            {
              for (x = 0; x < w; x++)
                {
                  pixels[2] = lpbBits[(x + y * w) * 4];
                  pixels[1] = lpbBits[(x + y * w) * 4 + 1];
                  pixels[0] = lpbBits[(x + y * w) * 4 + 2];
                  pixels[3] = lpbBits[(x + y * w) * 4 + 3];
                  if (!alpha && pixels[3] > 0)
                    alpha = TRUE;
                  pixels += 4;
                }
              pixels += (w * 4 - rowstride);
            }
          /* en:mask */
          if (!alpha)
            {
              GetDIBits (hDC, ii.hbmMask, 0, h,
                                lpbBits, (LPBITMAPINFO)&bmih, DIB_RGB_COLORS);
              pixels = gdk_pixbuf_get_pixels (pixbuf);
              for (y = 0; y < h; y++)
                {
                  for (x = 0; x < w; x++)
                    {
                      pixels[3] = 255 - lpbBits[(x + y * w) * 4];
                      pixels += 4;
                    }
                  pixels += (w * 4 - rowstride);
                }
            }
          g_free (lpbBits);
        }
      DeleteDC (hDC);
      DeleteObject (ii.hbmColor);
      DeleteObject (ii.hbmMask);
    }
  return pixbuf;
#elif GLIB_CHECK_VERSION(2,4,0) && defined (MIMEFILE)
  GdkPixbuf *pixbuf = NULL;

  if (!file)
    return NULL;
  if (!g_file_test (file, G_FILE_TEST_IS_DIR))
    {
      gchar *text;
      gssize length;

      text = fileio_load (MIMEFILE, &length);
      if (text)
        {
          gchar **mimes, *utf8str, *mime_type = NULL;
          gint i;

          text = g_realloc (text, length + 1);
          text[length] = '\0';
          utf8str = g_filename_to_utf8 (file, -1, NULL, NULL, NULL);
          mimes = g_strsplit (text, "\n", 0);
          for (i = 0; mimes[i]; i++)
            {
              gchar **globs;

              globs = g_strsplit (mimes[i], G_SEARCHPATH_SEPARATOR_S, 0);
              if (globs[0])
                {
                  gint j;

                  for (j = 1; globs[j]; j++)
                    if (g_pattern_match_simple (globs[j], utf8str))
                      {
                        mime_type = g_strdup (globs[0]);
                        break;
                      }
                }
              g_strfreev (globs);
              if (mime_type)
                break;
            }
          g_strfreev (mimes);
          g_free (utf8str);
          g_free (text);
          if (mime_type)
            {
              gchar *icon_name, *separator;

              icon_name = g_strconcat ("gnome-mime-", mime_type, NULL);
              g_free (mime_type);
              separator = g_strchr (icon_name, '/');
              if (separator)
                {
                  GtkIconTheme *icon_theme;

                  *separator = '-';
                  icon_theme = gtk_icon_theme_get_default ();
                  if (icon_theme)
                    {
                      gboolean result;

                      result = gtk_icon_theme_has_icon (icon_theme, icon_name);
                      if (!result)
                        {
                          *separator = '\0';
                          result = gtk_icon_theme_has_icon (icon_theme,
                                                                    icon_name);
                        }
                      if (result)
# if GLIB_CHECK_VERSION(2,6,0)
                        {
                          gint *sizes;

                          sizes = gtk_icon_theme_get_icon_sizes (icon_theme,
                                                                    icon_name);
                          if (sizes)
                            {
                              gint n = 0, s;

                              s = icon_size == FILEIO_ICON_SIZE_SMALL
                                                                    ? 16 : 32;
                              for (i = 0; sizes[i] > 0; i++)
                                if (ABS (s - sizes[i]) < ABS (s - n))
                                  n = sizes[i];
                              g_free (sizes);
                              if (n > 0)
                                pixbuf = gtk_icon_theme_load_icon (icon_theme,
                                                        icon_name, n, 0, NULL);
                            }
                        }
# else /* not GLIB_CHECK_VERSION(2,6,0) */
                        pixbuf = gtk_icon_theme_load_icon (icon_theme,
                            icon_name,
                            icon_size == FILEIO_ICON_SIZE_SMALL ? 16 : 32,
                            0, NULL);
# endif /* not GLIB_CHECK_VERSION(2,6,0) */
                    }
                }
              g_free (icon_name);
            }
        }
    }
# if 0
  if (!pixbuf)
    {
      GtkWidget *image;

      image = gtk_image_new ();
      pixbuf = gtk_widget_render_icon (image,
                                    g_file_test (file, G_FILE_TEST_IS_DIR)
                                        ? GTK_STOCK_DIRECTORY : GTK_STOCK_FILE,
                                    icon_size == FILEIO_ICON_SIZE_SMALL
                                        ? GTK_ICON_SIZE_SMALL_TOOLBAR
                                        : GTK_ICON_SIZE_LARGE_TOOLBAR,
                                    NULL);
      gtk_widget_destroy (image);
    }
# endif
  return pixbuf;
#else /* not G_OS_WIN32 && not MIMEFILE */
  return NULL;
#endif /* not G_OS_WIN32 && not MIMEFILE */
}
