/*
    Text maid
    copyright (c) 1998-2011 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "find.h"
#include "keyfile.h"
#include "print.h"
#include "replace.h"
#include "misc/charuty.h"
#include "misc/fileio.h"
#include "orz/orzhistory.h"


/******************************************************************************
*                                                                             *
* ja:キーファイル関数群                                                       *
*                                                                             *
******************************************************************************/
/*  ja:キーファイルから読み込む
    file,ファイル名(NULL:読み込まない)                                      */
void
keyfile_read (const gchar *file)
{
  GKeyFile *key_file;

  key_file = g_key_file_new ();
  if (g_key_file_load_from_file (key_file, file, G_KEY_FILE_NONE, NULL))
    {
      gchar *utf8str;
      gint i;

      license = g_key_file_get_integer (key_file, "license", "hash", NULL);
      orz_history_set_num (ORZ_HISTORY (history),
                g_key_file_get_integer (key_file, "pref", "history", NULL));
      newfile = g_key_file_get_boolean (key_file, "pref", "newfile", NULL);
      second = g_key_file_get_boolean (key_file, "pref", "second", NULL);
      def_width = g_key_file_get_integer (key_file, "pref", "width", NULL);
      def_height = g_key_file_get_integer (key_file, "pref", "height", NULL);
      n_pos = g_key_file_get_integer (key_file, "pref", "npos", NULL);
      ftnum = g_key_file_get_integer (key_file, "tmaid", "ftnum", NULL);
#if GTK_CHECK_VERSION(2,4,0) || defined (G_OS_WIN32)
      findex = g_key_file_get_integer (key_file, "tmaid", "findex", NULL);
#endif /* GTK_CHECK_VERSION(2,4,0) || defined (G_OS_WIN32) */
      utf8str = g_key_file_get_string (key_file, "tmaid", "openpath", NULL);
      open_path = utf8str
                ? g_filename_from_utf8 (utf8str, -1, NULL, NULL, NULL) : NULL;
      g_free (utf8str);
      utf8str = g_key_file_get_string (key_file, "tmaid", "savepath", NULL);
      save_path = utf8str
                ? g_filename_from_utf8 (utf8str, -1, NULL, NULL, NULL) : NULL;
      g_free (utf8str);
      /* ja:ファイルの履歴 */
      for (i = orz_history_get_num (ORZ_HISTORY (history)) - 1; i >= 0; i--)
        {
          gchar *key;

          key = g_strdup_printf ("file%d", i);
          utf8str = g_key_file_get_string (key_file, "history", key, NULL);
          g_free (key);
          if (utf8str)
            {
              gchar *file;

              file = g_filename_from_utf8 (utf8str, -1, NULL, NULL, NULL);
              g_free (utf8str);
              orz_history_add_file (ORZ_HISTORY (history), file);
              g_free (file);
            }
        }
      /* ja:検索 */
      find_arrow = g_key_file_get_boolean (key_file, "find", "arrow", NULL);
      find_ignorecase = g_key_file_get_boolean (key_file,
                                                "find", "ignorecase", NULL);
      for (i = 0; i < FIND_TEXT; i++)
        {
          gchar *key;

          key = g_strdup_printf ("text%02d", i);
          find_text[i] = g_key_file_get_string (key_file, "find", key, NULL);
          g_free (key);
          if (!find_text[i] || *find_text[i] == '\0')
            {
              g_free (find_text[i]);
              find_text[i] = NULL;
              break;
            }
        }
      /* ja:置換 */
      for (i = 0; i < REPLACE_TEXT; i++)
        {
          gchar *key;

          key = g_strdup_printf ("text%02d", i);
          replace_text[i] = g_key_file_get_string (key_file, "replace", key,
                                                                        NULL);
          g_free (key);
          if (!replace_text[i] || *replace_text[i] == '\0')
            {
              g_free (replace_text[i]);
              replace_text[i] = NULL;
              break;
            }
        }
      /* ja:印刷 */
#ifdef G_OS_UNIX
      printer.mode = g_key_file_get_boolean (key_file,
                                                    "printer", "mode", NULL);
      utf8str = g_key_file_get_string (key_file, "printer", "printer", NULL);
      if (utf8str)
        {
          printer.printer
                        = g_filename_from_utf8 (utf8str, -1, NULL, NULL, NULL);
          g_free (utf8str);
        }
      if (!printer.printer)
        printer.printer = g_strdup ("lpr");
#endif /* G_OS_UNIX */
#if defined (G_OS_UNIX) || defined (G_OS_WIN32)
      printer.copies = g_key_file_get_integer (key_file, "printer", "copies",
                                                                        NULL);
#endif /* defined (G_OS_UNIX) || defined (G_OS_WIN32) */
#ifndef G_OS_WIN32
      utf8str = g_key_file_get_string (key_file, "printer", "file", NULL);
      if (utf8str)
        {
          printer.file = g_filename_from_utf8 (utf8str, -1, NULL, NULL, NULL);
          g_free (utf8str);
        }
      if (!printer.file)
        printer.file = g_strdup ("tmaid.ps");
      printer.width = g_key_file_get_integer (key_file, "printer", "width",
                                                                        NULL);
      printer.height = g_key_file_get_integer (key_file, "printer", "height",
                                                                        NULL);
      printer.left = g_key_file_get_integer (key_file, "printer", "left",
                                                                        NULL);
      printer.top = g_key_file_get_integer (key_file, "printer", "top", NULL);
      printer.right = g_key_file_get_integer (key_file, "printer", "right",
                                                                        NULL);
      printer.bottom = g_key_file_get_integer (key_file, "printer", "bottom",
                                                                        NULL);
#endif /* not G_OS_WIN32 */
      /* ja:ファイルタイプ */
      if (ftnum > 0)
        ftype = g_malloc (ftnum * sizeof (FileType));
      for (i = 0; i < ftnum; i++)
        {
          gsize length;
          gboolean success[12];
          gchar *group;
          gint *list;

          group = g_strdup_printf ("filetype%04d", i);
          ftype[i].ft_id = tcount++;
          ftype[i].text = g_key_file_get_string (key_file,
                                                        group, "text", NULL);
          if (!ftype[i].text)
            ftype[i].text = g_strdup ("UNKNOW");
          ftype[i].ext = g_key_file_get_string (key_file, group, "ext", NULL);
          if (!ftype[i].ext)
            ftype[i].ext = g_strdup ("*");
          ftype[i].crlf = g_key_file_get_integer (key_file,
                                                    group, "crlf", NULL);
          ftype[i].margin = g_key_file_get_integer (key_file,
                                                    group, "margin", NULL);
          ftype[i].tab = g_key_file_get_integer (key_file, group, "tab", NULL);
          ftype[i].writemode = g_key_file_get_integer (key_file,
                                                    group, "writemode", NULL);
          ftype[i].associate = g_key_file_get_boolean (key_file,
                                                    group, "associate", NULL);
          ftype[i].autoindent = g_key_file_get_boolean (key_file,
                                                    group, "autoindent", NULL);
          ftype[i].code = g_key_file_get_boolean (key_file,
                                                    group, "code", NULL);
          ftype[i].eof = g_key_file_get_boolean (key_file, group, "eof", NULL);
          ftype[i].hexcase = g_key_file_get_boolean (key_file,
                                                    group, "hexcase", NULL);
          ftype[i].html = g_key_file_get_boolean (key_file,
                                                    group, "html", NULL);
          ftype[i].esc = g_key_file_get_boolean (key_file, group, "esc", NULL);
          ftype[i].limit = g_key_file_get_boolean (key_file, 
                                                    group, "limit", NULL);
          ftype[i].overwrite = g_key_file_get_boolean (key_file,
                                                    group, "overwrite", NULL);
          ftype[i].space = g_key_file_get_boolean (key_file, 
                                                    group, "space", NULL);
          ftype[i].syscol = g_key_file_get_boolean (key_file,
                                                    group, "syscol", NULL);
          ftype[i].tabconv = g_key_file_get_boolean (key_file,
                                                    group, "tabconv", NULL);
          ftype[i].gline = g_key_file_get_boolean (key_file, 
                                                    group, "gline", NULL);
          ftype[i].mline = g_key_file_get_boolean (key_file, 
                                                    group, "mline", NULL);
          ftype[i].nline = g_key_file_get_boolean (key_file, 
                                                    group, "nline", NULL);
          ftype[i].uline = g_key_file_get_boolean (key_file, 
                                                    group, "uline", NULL);
          ftype[i].vline = g_key_file_get_boolean (key_file, 
                                                    group, "vline", NULL);
          /* ja:キャラクターセット */
          ftype[i].charset = g_key_file_get_string (key_file,
                                                    group, "charset", NULL);
          if (ftype[i].charset)
            {
              gchar *tmp;

              tmp = charuty_strip (ftype[i].charset);
              g_free (ftype[i].charset);
              ftype[i].charset = tmp;
            }
          if (!ftype[i].charset)
            {
              const gchar *charset;

              ftype[i].charset = g_get_charset (&charset)
                                        ? g_strdup_printf ("%s,UTF-8", charset)
                                                        : g_strdup ("UTF-8");
            }
          /* ja:フォント */
          ftype[i].font_name = g_key_file_get_string (key_file,
                                                    group, "font", NULL);
          /* ja:色 */
          list = g_key_file_get_integer_list (key_file,
                                                group, "color", &length, NULL);
          if (list && length * sizeof (gint) >= sizeof (GdkColor) * 12)
            {
              g_memmove (ftype[i].color, list, sizeof (GdkColor) * 12);
              g_free (list);
            }
          else
            {
              g_memmove (ftype[i].color, system_color, sizeof (GdkColor) * 12);
            }
#if ! GTK_CHECK_VERSION(3,0,0)
          gdk_colormap_alloc_colors (gdk_colormap_get_system (),
                                    ftype[i].color, 12, FALSE, TRUE, success);
#endif /* not GTK_CHECK_VERSION(3,0,0) */
          g_free (group);
        }
    }
  else
    {
#ifdef G_OS_UNIX
      printer.mode = TRUE;
      printer.printer = g_strdup ("lpr");
#endif /* G_OS_UNIX */
#ifndef G_OS_WIN32
      printer.file = g_strdup ("tmaid.ps");
#endif /* not G_OS_WIN32 */
#if defined (G_OS_UNIX) || defined (G_OS_WIN32)
      printer.copies = 1;
#endif /* defined (G_OS_UNIX) || defined (G_OS_WIN32) */
#ifndef G_OS_WIN32
      printer.width = 594;
      printer.height = 840;
      printer.left = printer.top = printer.right = printer.bottom = 20;
#endif /* not G_OS_WIN32 */
    }
  g_key_file_free (key_file);

  /* ja:ファイルタイプ */
  if (ftnum <= 0)
    {
      gboolean success[12];
      const gchar *charset;

      ftnum = 1;
      ftype = g_malloc0 (ftnum * sizeof (FileType));
      ftype[0].text = g_strdup (_("All Files"));
      ftype[0].ext = g_strdup ("*");
      ftype[0].crlf = 3;
      ftype[0].margin = 80;
      ftype[0].tab = 8;
      ftype[0].associate = TRUE;
      ftype[0].syscol = TRUE;
      /* ja:キャラクターセット */
      ftype[0].charset = !g_get_charset (&charset) && charset
                ? g_strconcat (charset, ",UTF-8", NULL) : g_strdup ("UTF-8");
      /* ja:色 */
      g_memmove (ftype[0].color, system_color, sizeof (GdkColor) * 12);
#if ! GTK_CHECK_VERSION(3,0,0)
      gdk_colormap_alloc_colors (gdk_colormap_get_system (),
                                    ftype[0].color, 12, FALSE, TRUE, success);
#endif /* not GTK_CHECK_VERSION(3,0,0) */
    }
}


/*  ja:キーファイルに書き込む
    file,ファイル名(NULL:書き込まない)                                      */
void
keyfile_write (const gchar *file)
{
  if (file)
    {
      gsize length;
      gchar *data, *group, *utf8str;
      gint i;
      GList *glist;
      GKeyFile *key_file;

      key_file = g_key_file_new ();
      g_key_file_load_from_file (key_file, file, G_KEY_FILE_NONE, NULL);
      g_key_file_set_integer (key_file, "license", "hash", license);
      g_key_file_set_integer (key_file, "pref", "history",
                                orz_history_get_num (ORZ_HISTORY (history)));
      g_key_file_set_boolean (key_file, "pref", "newfile", newfile);
      g_key_file_set_boolean (key_file, "pref", "second", second);
      g_key_file_set_integer (key_file, "pref", "width", def_width);
      g_key_file_set_integer (key_file, "pref", "height", def_height);
      g_key_file_set_integer (key_file, "pref", "npos", n_pos);
      g_key_file_set_integer (key_file, "tmaid", "ftnum", ftnum);
#if GTK_CHECK_VERSION(2,4,0) || defined (G_OS_WIN32)
      for (i = ftnum - 1; i >= 0; i--)
        if (ftype[i].ft_id == findex)
          break;
      g_key_file_set_integer (key_file, "tmaid", "findex", i);
#endif /* GTK_CHECK_VERSION(2,4,0) || defined (G_OS_WIN32) */
      if (open_path)
        {
          utf8str = g_filename_to_utf8 (open_path, -1, NULL, NULL, NULL);
          if (utf8str)
            {
              g_key_file_set_string (key_file, "tmaid", "openpath", utf8str);
              g_free (utf8str);
            }
        }
      if (save_path)
        {
          utf8str = g_filename_to_utf8 (save_path, -1, NULL, NULL, NULL);
          if (utf8str)
            {
              g_key_file_set_string (key_file, "tmaid", "savepath", utf8str);
              g_free (utf8str);
            }
        }
      /* ja:ファイルの履歴 */
      glist = orz_history_get_file (ORZ_HISTORY (history));
      for (i = 0; i < g_list_length (glist); i++)
        {
          gchar *key;

          key = g_strdup_printf ("file%d", i);
          utf8str = g_filename_to_utf8 (g_list_nth_data (glist, i), -1,
                                                            NULL, NULL, NULL);
          if (utf8str)
            {
              g_key_file_set_string (key_file, "history", key, utf8str);
              g_free (utf8str);
            }
          g_free (key);
        }
      g_list_free (glist);
      while (i <= ORZ_HISTORY_NUM_MAX)
        {
          gchar *key;

          key = g_strdup_printf ("file%d", i++);
          g_key_file_remove_key (key_file, "history", key, NULL);
          g_free (key);
        }
      /* ja:検索 */
      g_key_file_set_boolean (key_file, "find", "arrow", find_arrow);
      g_key_file_set_boolean (key_file, "find", "ignorecase", find_ignorecase);
      for (i = 0; i < FIND_TEXT && find_text[i]; i++)
        {
          gchar *key;

          key = g_strdup_printf ("text%02d", i);
          g_key_file_set_string (key_file, "find", key, find_text[i]);
          g_free (key);
        }
      /* ja:置換 */
      for (i = 0; i < REPLACE_TEXT && replace_text[i]; i++)
        {
          gchar *key;

          key = g_strdup_printf ("text%02d", i);
          g_key_file_set_string (key_file, "replace", key, replace_text[i]);
          g_free (key);
        }
      /* ja:印刷 */
#ifdef G_OS_UNIX
      g_key_file_set_boolean (key_file, "printer", "mode", printer.mode);
      utf8str = g_filename_to_utf8 (printer.printer, -1, NULL, NULL, NULL);
      if (utf8str)
        {
          g_key_file_set_string (key_file, "printer", "printer", utf8str);
          g_free (utf8str);
        }
#endif /* not G_OS_UNIX */
#if defined (G_OS_UNIX) || defined (G_OS_WIN32)
      g_key_file_set_integer (key_file, "printer", "copies", printer.copies);
#endif /* defined (G_OS_UNIX) || defined (G_OS_WIN32) */
#ifndef G_OS_WIN32
      utf8str = g_filename_to_utf8 (printer.file, -1, NULL, NULL, NULL);
      if (utf8str)
        {
          g_key_file_set_string (key_file, "printer", "file", utf8str);
          g_free (utf8str);
        }
      g_key_file_set_integer (key_file, "printer", "width", printer.width);
      g_key_file_set_integer (key_file, "printer", "height", printer.height);
      g_key_file_set_integer (key_file, "printer", "left", printer.left);
      g_key_file_set_integer (key_file, "printer", "top", printer.top);
      g_key_file_set_integer (key_file, "printer", "right", printer.right);
      g_key_file_set_integer (key_file, "printer", "bottom", printer.bottom);
#endif /* not G_OS_WIN32 */
      /* ja:ファイルタイプ */
      for (i = 0; i < ftnum; i++)
        {
          gint list[(sizeof (GdkColor) * 12 + sizeof (gint) - 1)
                                                            / sizeof (gint)];

          group = g_strdup_printf ("filetype%04d", i);
          g_key_file_set_string (key_file, group, "text", ftype[i].text);
          g_key_file_set_string (key_file, group, "ext", ftype[i].ext);
          g_key_file_set_integer (key_file, group, "crlf", ftype[i].crlf);
          g_key_file_set_integer (key_file, group, "margin", ftype[i].margin);
          g_key_file_set_integer (key_file, group, "tab", ftype[i].tab);
          g_key_file_set_integer (key_file, group, "writemode",
                                                        ftype[i].writemode);
          g_key_file_set_boolean (key_file, group, "associate",
                                                        ftype[i].associate);
          g_key_file_set_boolean (key_file, group, "autoindent",
                                                        ftype[i].autoindent);
          g_key_file_set_boolean (key_file, group, "code", ftype[i].code);
          g_key_file_set_boolean (key_file, group, "eof", ftype[i].eof);
          g_key_file_set_boolean (key_file, group, "hexcase",
                                                            ftype[i].hexcase);
          g_key_file_set_boolean (key_file, group, "html", ftype[i].html);
          g_key_file_set_boolean (key_file, group, "esc", ftype[i].esc);
          g_key_file_set_boolean (key_file, group, "limit", ftype[i].limit);
          g_key_file_set_boolean (key_file, group, "overwrite",
                                                        ftype[i].overwrite);
          g_key_file_set_boolean (key_file, group, "space", ftype[i].space);
          g_key_file_set_boolean (key_file, group, "syscol", ftype[i].syscol);
          g_key_file_set_boolean (key_file, group, "tabconv",
                                                            ftype[i].tabconv);
          g_key_file_set_boolean (key_file, group, "gline", ftype[i].gline);
          g_key_file_set_boolean (key_file, group, "mline", ftype[i].mline);
          g_key_file_set_boolean (key_file, group, "nline", ftype[i].nline);
          g_key_file_set_boolean (key_file, group, "uline", ftype[i].uline);
          g_key_file_set_boolean (key_file, group, "vline", ftype[i].vline);
          /* ja:キャラクターセット */
          g_key_file_set_string (key_file, group, "charset", ftype[i].charset);
          /* ja:フォント */
          if (ftype[i].font_name)
            g_key_file_set_string (key_file, group, "font",
                                                        ftype[i].font_name);
          /* ja:色 */
          list[(sizeof (GdkColor) * 12 + sizeof (gint) - 1)
                                                    / sizeof (gint) - 1] = 0;
          g_memmove (list, ftype[i].color, sizeof (GdkColor) * 12);
          g_key_file_set_integer_list (key_file, group, "color", list,
                (sizeof (GdkColor) * 12 + sizeof (gint) - 1) / sizeof (gint));
          g_free (group);
        }
      while (group = g_strdup_printf ("filetype%04d", i++),
                                        g_key_file_has_group (key_file, group))
        {
          g_key_file_remove_group (key_file, group, NULL);
          g_free (group);
        }
      g_free (group);
      data = g_key_file_to_data (key_file, &length, NULL);
      g_key_file_free (key_file);
      if (data)
        {
          fileio_save (file, data, length);
          g_free (data);
        }
    }
}
