/*
    fileio
    copyright (c) 1998-2013 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef __FILEIO_H__
#define __FILEIO_H__


#include "gcommon/gcommon.h"
#ifdef G_OS_WIN32
# include <windows.h>
# include <tchar.h>
#endif /* G_OS_WIN32 */


G_BEGIN_DECLS


#ifdef G_OS_WIN32
# define FILEIO_ACCESS_READ         GENERIC_READ
# define FILEIO_ACCESS_WRITE        GENERIC_WRITE
# define FILEIO_SHARE_READ          FILE_SHARE_READ
# define FILEIO_SHARE_WRITE         FILE_SHARE_WRITE
# define FILEIO_MODE_CREATE_NEW     CREATE_NEW
# define FILEIO_MODE_CREATE_ALWAYS  CREATE_ALWAYS
# define FILEIO_MODE_OPEN_EXISTING  OPEN_EXISTING
# define FILEIO_MODE_OPEN_ALWAYS    OPEN_ALWAYS
# define FILEIO_SEEK_SET            FILE_BEGIN
# define FILEIO_SEEK_CUR            FILE_CURRENT
# define FILEIO_SEEK_END            FILE_END
#else /* not G_OS_WIN32 */
# define FILEIO_ACCESS_READ         1
# define FILEIO_ACCESS_WRITE        2
# define FILEIO_SHARE_READ          1
# define FILEIO_SHARE_WRITE         2
# define FILEIO_MODE_CREATE_NEW     0
# define FILEIO_MODE_CREATE_ALWAYS  1
# define FILEIO_MODE_OPEN_EXISTING  2
# define FILEIO_MODE_OPEN_ALWAYS    3
# define FILEIO_SEEK_SET            SEEK_SET
# define FILEIO_SEEK_CUR            SEEK_CUR
# define FILEIO_SEEK_END            SEEK_END
#endif /* not G_OS_WIN32 */
#define FILEIO_SHARE_DENY           0
#define FILEIO_ACCESS_ALL           (FILEIO_ACCESS_READ|FILEIO_ACCESS_WRITE)
#define FILEIO_SHARE_ALL            (FILEIO_SHARE_READ|FILEIO_SHARE_WRITE)


typedef struct _FileIO FileIO;
#ifdef G_OS_WIN32
typedef FILETIME FileIOTime;
#else /* not G_OS_WIN32 */
typedef time_t FileIOTime;
#endif /* not G_OS_WIN32 */
typedef struct _FileIOSysTime
{
  gint year;    /* ja:年(1900...) */
  gint mon;     /* ja:月(1...12) */
  gint mday;    /* ja:日(1..31) */
  gint hour;    /* ja:時間(0..23) */
  gint min;     /* ja:分(0..59) */
  gint sec;     /* ja:秒(0..61) */
  gint wday;    /* ja:曜日(0:日曜日,1:月曜日,...) */
} FileIOSysTime;


/******************************************************************************
*                                                                             *
* ja:低レベルファイル入出力関数群                                             *
*                                                                             *
******************************************************************************/
/*  ja:ファイルの作成
      file,ファイル名
    access,アクセス
     share,共有
      mode,モード
       RET,ファイルポインタ,NULL:エラー                                     */
FileIO *
fileio_open (const gchar *file,
             const guint  access,
             const guint  share,
             const guint  mode);


/*  ja:ファイルポインタの参照数を増やす
    fio,ファイルポインタ
    RET,TRUE:正常終了,FALSE:エラー                                          */
gboolean
fileio_ref (FileIO *fio);


/*  ja:ファイルポインタの参照数を減らす
    fio,ファイルポインタ
    RET,TRUE:正常終了,FALSE:エラー                                          */
gboolean
fileio_unref (FileIO *fio);


/*  ja:ファイルポインタを閉じる
    fio,ファイルポインタ
    RET,TRUE:正常終了,FALSE:エラー                                          */
#define fileio_close(fio) fileio_unref(fio)


/*  ja:ファイルの名前を取得する
    fio,ファイルポインタ
    RET,名前(フルパス)                                                      */
const gchar *
fileio_getname (FileIO *fio);


/*  ja:ファイルから読み込む
       fio,ファイルポインタ
      data,バッファ
    length,バイト数
       RET,バイト数,-1:エラー                                               */
gssize
fileio_read (FileIO       *fio,
             gpointer      data,
             const gssize  length);


/*  ja:ファイルに書き込む
       fio,ファイルポインタ
      data,バッファ
    length,バイト数
       RET,バイト数,-1:エラー                                               */
gssize
fileio_write (FileIO        *fio,
              gconstpointer  data,
              const gssize   length);


/*  ja:ファイルをシークする
       fio,ファイルポインタ
    offset,位置
    whence,モード
       RET,バイト数,-1:エラー                                               */
goffset
fileio_seek (FileIO        *fio,
             const goffset  offset,
             const guint    whence);


/*  ja:ファイルサイズを求める
    fio,ファイルポインタ
    RET,ファイルサイズ,-1:エラー                                            */
goffset
fileio_size (FileIO *fio);


/*  ja:ファイルの終端を設定する
    fio,ファイルポインタ
    RET,TRUE:正常終了,FALSE:エラー                                          */
gboolean
fileio_eof (FileIO *fio);


/******************************************************************************
*                                                                             *
* ja:高レベルファイル入出力関数群                                             *
*                                                                             *
******************************************************************************/
/*  ja:ファイルをメモリに読み込む
      file,ファイル名
    length,ファイルサイズ
       RET,メモリ,NULL:エラー                                               */
gpointer
fileio_load (const gchar *file,
             gssize      *length);


/*  ja:ファイルにメモリから書き込む
      file,ファイル名
      data,メモリ
    length,サイズ
       RET,TRUE:正常終了,FALSE:エラー                                       */
gboolean
fileio_save (const gchar   *file,
             gconstpointer  data,
             const gssize   length);


/*  ja:1文字読み込む
    fio,ファイルポインタ
    RET,文字,EOF:エラー                                                     */
gint
fileio_getc (FileIO *fio);


/*  ja:1文字書き込む
      c,文字
    fio,ファイルポインタ
    RET,文字,EOF:エラー                                                     */
gint
fileio_putc (gint    c,
             FileIO *fio);


/*  ja:1行読み込む
      data,バッファ
    length,バイト数
       fio,ファイルポインタ
       RET,data:正常終了,NULL:エラー                                        */
gchar *
fileio_gets (gchar       *data,
             const gsize  length,
             FileIO      *fio);


/*  ja:1行書き込む
      data,バッファ
    length,バイト数
       fio,ファイルポインタ
       RET,正数:正常終了,EOF:エラー                                         */
gint
fileio_puts (const gchar *data,
             FileIO      *fio);


/******************************************************************************
*                                                                             *
* ja:ディレクトリ関数群                                                       *
*                                                                             *
******************************************************************************/
/*  ja:ディレクトリを作る
    dir,ディレクトリ名
    RET,TRUE:正常終了,FALSE,エラー                                          */
gboolean
fileio_mkdir (const gchar *dir);


/*  ja:ファイル/ディレクトリを削除する
    path,ファイル/ディレクトリ名
     RET,TRUE:正常終了,FALSE,エラー                                         */
gboolean
fileio_remove (const gchar *path);


/******************************************************************************
*                                                                             *
* ja:ファイル時刻関数群                                                       *
*                                                                             *
******************************************************************************/
/*  ja:ファイルの時刻を取得する
     file,ファイル名
    atime,最終アクセス時刻
    mtime,最終修正時刻
      RET,TRUE:正常終了,FALSE,エラー                                        */
gboolean
fileio_gettime (const gchar *file,
                FileIOTime  *atime,
                FileIOTime  *mtime);


/*  ja:ファイルの時刻を設定する
     file,ファイル名
    atime,最終アクセス時刻
    mtime,最終修正時刻
      RET,TRUE:正常終了,FALSE,エラー                                        */
gboolean
fileio_settime (const gchar *file,
                FileIOTime  *atime,
                FileIOTime  *mtime);


/*  ja:システム時刻を取得する
    ftime,ファイルの時刻
    stime,システム時間                                                      */
void
fileio_getsystime (FileIOTime    *ftime,
                   FileIOSysTime *stime);


/*  ja:システム時刻を設定する
    ftime,ファイルの時刻
    stime,システム時間                                                      */
void
fileio_setsystime (FileIOTime    *ftime,
                   FileIOSysTime *stime);


/******************************************************************************
*                                                                             *
* ja:応用ファイル関数群                                                       *
*                                                                             *
******************************************************************************/
#ifndef USE_GTK_EMULATE
/*  ja:ファイルのアイコンを取得する
         file,ファイル名
    icon_size,アイコンサイズ
          RET,pixbuf,NULL:エラー                                            */
GdkPixbuf *
fileio_extract_icon (const gchar *file,
                     GtkIconSize  icon_size);
#endif /* not USE_GTK_EMULATE */


G_END_DECLS


#endif /* __FILEIO_H__ */
