/*
    Text maid for Windows
    copyright (c) 1998-2018 Kazuki Iwamoto https://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "edit.h"
#include "wcommon.h"


/******************************************************************************
*                                                                             *
* ja:編集関数群                                                               *
*                                                                             *
******************************************************************************/
/*  ja:文字の種類を求める
     lpbType,文字の属性を格納するバッファ,0:ANK,1:SJIS1バイト目,2:SJIS2バイト目
    lpszText,文字列
     nLength,データサイズ                                                   */
void
mCharType (LPBYTE lpbType,
           LPCSTR lpszText,
           int    nLength)
{
  int i = 0;

  while (i < nLength)
    if (i + 1 < nLength && IsDBCSLeadByteEx (CP_ACP, lpszText[i]))
      {
        lpbType[i++] = 1;
        lpbType[i++] = 2;
      }
    else
      {
        lpbType[i++] = 0;
      }
}


/*  ja:ラインバッファを求める
    lpStart,ラインバッファ
     lpnOff,現在のy座標
          y,y座標
        RET,ラインバッファ                                                  */
LPLINEBUF
GetLineBuffer (LINEBUF **lpStart,
               int      *lpnOff,
               int       y)
{
  while (y < *lpnOff && (*lpStart)->prev)
    {
      *lpStart = (*lpStart)->prev;
      (*lpnOff)--;
    }
  while (*lpnOff < y && (*lpStart)->next)
    {
      *lpStart = (*lpStart)->next;
      (*lpnOff)++;
    }
  return *lpStart;
}


/*  ja:桁数を取得する
    lpStart,ラインバッファ
     lpnOff,現在のy座標
          y,行
       nTab,タブ
        RET,桁数                                                            */
int
GetWidth (LINEBUF **lpStart,
          int      *lpnOff,
          int       y,
          int       nTab)
{
  int nDataPos = 0, nWidth = 0;
  LPLINEBUF p;

  p = GetLineBuffer (lpStart, lpnOff, y);
  if (p->fMargin)
    while (nDataPos < p->nLength - 1)
      if (p->lpszText[nDataPos] == '\t')
        {
          nWidth = (nWidth / nTab + 1) * nTab;
          nDataPos++;
        }
      else if (IsDBCSLeadByteEx (CP_ACP, p->lpszText[nDataPos]))
        {
          if (nDataPos == p->nLength - 2)
            break;
          nWidth += 2;
          nDataPos += 2;
        }
      else
        {
          nWidth++;
          nDataPos++;
        }
  else
    while (nDataPos < p->nLength)
      if (p->lpszText[nDataPos] == '\t')
        {
          nWidth = (nWidth / nTab + 1) * nTab;
          nDataPos++;
        }
      else if (nDataPos == p->nLength - 1
                        || !IsDBCSLeadByteEx (CP_ACP, p->lpszText[nDataPos]))
        {
          nWidth++;
          nDataPos++;
        }
      else
        {
          nWidth += 2;
          nDataPos += 2;
        }
  return nWidth;
}


/*  ja:最大桁数を取得する
    ptw,TXTウインドウ情報
    RET,桁数                                                                */
int
GetWidthMax (LPTEXTWND ptw)
{
  int y, nMax = 0;
  LPLINEBUF p;

  for (y = 0, p = GetLineBuffer (&ptw->lpStart, &ptw->nOff, ptw->ptTop.y);
                y < ptw->siWnd.cy && p; y += ptw->nFontSize * 2, p = p->next)
    {
      int nDataPos, nWidth = 0;

      for (nDataPos = 0; nDataPos < p->nLength; nDataPos++)
        if (p->lpszText[nDataPos] == '\t')
          nWidth = (nWidth / ptw->nTab + 1) * ptw->nTab;
        else
          nWidth++;
      if (nWidth > nMax)
        nMax = nWidth;
    }
  return nMax;
}


/*  ja:キャレットのアライメントされたx座標を求める
    lpStart,ラインバッファ
     lpnOff,現在のy座標
          x,x座標
          y,y座標
       nTab,タブ
     fAlign,TRUE:右に寄せる,FALSE:左に寄せる
        RET,座標                                                            */
int
GetAlignPos (LINEBUF **lpStart,
             int      *lpnOff,
             int       x,
             int       y,
             int       nTab,
             BOOL      fAlign)
{
  int nDataPos = 0, nScreenPos = 0;
  LPLINEBUF p;

  p = GetLineBuffer (lpStart, lpnOff, y);
  if (p->fMargin)
    while (nScreenPos < x && nDataPos < p->nLength - 1)
      if (p->lpszText[nDataPos] == '\t')
        {
          if (x < (nScreenPos / nTab + 1) * nTab)
            return fAlign ? (nScreenPos / nTab + 1) * nTab : nScreenPos;
          nScreenPos = (nScreenPos / nTab + 1) * nTab;
          nDataPos++;
        }
      else if (!IsDBCSLeadByteEx (CP_ACP, p->lpszText[nDataPos]))
        {
          nScreenPos++;
          nDataPos++;
        }
      else if (nDataPos == p->nLength - 2)
        {
          break;
        }
      else if (x < nScreenPos + 2)
        {
          return fAlign ? nScreenPos + 2 : nScreenPos;
        }
      else
        {
          nScreenPos += 2;
          nDataPos += 2;
        }
  else
    while (nScreenPos < x && nDataPos < p->nLength)
      if (p->lpszText[nDataPos] == '\t')
        {
          if (x < (nScreenPos / nTab + 1) * nTab)
            return fAlign ? (nScreenPos / nTab + 1) * nTab : nScreenPos;
          nScreenPos = (nScreenPos / nTab + 1) * nTab;
          nDataPos++;
        }
      else if (nDataPos == p->nLength - 1
                        || !IsDBCSLeadByteEx (CP_ACP, p->lpszText[nDataPos]))
        {
          nScreenPos++;
          nDataPos++;
        }
      else if (x < nScreenPos + 2)
        {
          return fAlign ? nScreenPos + 2 : nScreenPos;
        }
      else
        {
          nScreenPos += 2;
          nDataPos += 2;
        }
  return nScreenPos;
}


/*  ja:画面上の座標からラインバッファ上のアドレスを求める
    lpStart,ラインバッファ
     lpnOff,現在のy座標
          x,x座標
          y,y座標
       nTab,タブ
     fAlign,TRUE:右に寄せる,FALSE:左に寄せる
        RET,アドレス                                                        */
int
GetDataPos (LINEBUF **lpStart,
            int      *lpnOff,
            int       x,
            int       y,
            int       nTab,
            BOOL      fAlign)
{
  int nDataPos = 0, nScreenPos = 0;
  LPLINEBUF p;

  p = GetLineBuffer (lpStart, lpnOff, y);
  while (nScreenPos < x && nDataPos < p->nLength)
    if (p->lpszText[nDataPos] == '\t')
      {
        nScreenPos = (nScreenPos / nTab + 1) * nTab;
        if (x < nScreenPos)
          return fAlign ? nDataPos + 1 : nDataPos;
        nDataPos++;
      }
    else if (nDataPos == p->nLength - 1
                        || !IsDBCSLeadByteEx (CP_ACP, p->lpszText[nDataPos]))
      {
        nScreenPos++;
        nDataPos++;
      }
    else if (x < nScreenPos + 2)
      {
        return fAlign ? nDataPos + 2 : nDataPos;
      }
    else
      {
        nScreenPos += 2;
        nDataPos += 2;
      }
  return min (nDataPos, p->nLength);
}


/*  ja:ラインバッファ上のアドレスから画面上の座標を求める
    lpStart,ラインバッファ
     lpnOff,現在のy座標
          x,x座標
          y,y座標
       nTab,タブ
        RET,座標                                                            */
int
GetScreenPos (LINEBUF **lpStart,
              int      *lpnOff,
              int       x,
              int       y,
              int       nTab)
{
  int nDataPos, nScreenPos = 0;
  LPLINEBUF p;

  p = GetLineBuffer (lpStart, lpnOff, y);
  for (nDataPos = 0; nDataPos < x; nDataPos++)
    if (p->lpszText[nDataPos] == '\t')
      nScreenPos = (nScreenPos / nTab + 1) * nTab;
    else
      nScreenPos++;
  return nScreenPos;
}


/*  ja:文字の分類
    lpszText,文字列
     nLength,文字数
         RET,分類                                                           */
DWORD
GetCharType (LPCSTR lpszText,
             int    nLength)
{
  WORD ct1, ct3;

  if (!GetStringTypeExA (LOCALE_USER_DEFAULT,
                                        CT_CTYPE1, lpszText, nLength, &ct1)
        || !GetStringTypeExA (LOCALE_USER_DEFAULT,
                                        CT_CTYPE3, lpszText, nLength, &ct3))
    return TMAID_CTYPE_ERROR;
  switch (nLength)
    {
      case 2:
        if (ct3 & C3_KASHIDA)
          return TMAID_CTYPE_KASHIDA;
        if (ct3 & C3_IDEOGRAPH)
          return TMAID_CTYPE_IDEOGRAPH;
        if ((ct3 & (C3_KATAKANA | C3_HIRAGANA)) == (C3_KATAKANA | C3_HIRAGANA))
          return TMAID_CTYPE_KANA;
        if (ct3 & C3_KATAKANA)
          return TMAID_CTYPE_KATAKANA;
        if (ct3 & C3_HIRAGANA)
          return TMAID_CTYPE_HIRAGANA;
        if (ct1 & C1_DIGIT)
          return TMAID_CTYPE_DIGIT;
        if (ct1 & C1_ALPHA)
          return TMAID_CTYPE_ALPHA;
        if (ct1 & C1_SPACE)
          return TMAID_CTYPE_SPACE;
        break;
      case 1:
        if (ct3 & C3_KATAKANA)
          return TMAID_CTYPE_H_KATAKANA;
        if (ct1 & C1_DIGIT)
          return TMAID_CTYPE_H_DIGIT;
        if (ct1 & C1_ALPHA)
          return TMAID_CTYPE_H_ALPHA;
    }
  return TMAID_CTYPE_NULL;
}


/*  ja:区切りを求める
    lpStart,ラインバッファ
     lpnOff,現在のy座標
          x,x座標
          y,y座標
       nTab,タブ
      fMove,TRUE:右に移動する,FALSE:左に移動する
        RET,座標                                                            */
int
GetMovePos (LINEBUF **lpStart,
            int      *lpnOff,
            int       x,
            int       y,
            int       nTab,
            int       fMove)
{
  int nDataPos;
  LPLINEBUF p;

  p = GetLineBuffer (lpStart, lpnOff, y);
  if (fMove)
    {
      int nLength;
      DWORD ctype;

      /* ja:右へ移動する */
      nDataPos = GetDataPos (lpStart, lpnOff, x, y, nTab, FALSE);
      nLength = nDataPos < p->nLength - 1
                && IsDBCSLeadByteEx (CP_ACP, p->lpszText[nDataPos]) ? 2 : 1;
      ctype = GetCharType (p->lpszText + nDataPos, nLength);
      if (ctype == TMAID_CTYPE_ERROR)
        {
          nDataPos = p->nLength;
        }
      else
        {
          nDataPos += nLength;
          while (nDataPos < p->nLength)
            {
              DWORD ct;

              nLength = nDataPos < p->nLength - 1
                && IsDBCSLeadByteEx (CP_ACP, p->lpszText[nDataPos]) ? 2 : 1;
              ct = GetCharType (p->lpszText + nDataPos, nLength);
              if (ct == TMAID_CTYPE_ERROR)
                {
                  nDataPos = p->nLength;
                  break;
                }
              if (!(ctype & ct & TMAID_CTYPE_KANA) && ctype != ct)
                {
                  if (ct != TMAID_CTYPE_NULL)
                    break;
                  ctype = TMAID_CTYPE_NULL;
                }
              nDataPos += nLength;
            }
        }
    }
  else
    {
      int i = 0, nLength;
      DWORD ctype;
      LPBYTE lpbType; /* ja:0:1バイト文字,1:1バイト目,2バイト目 */

      /* ja:左へ移動する */
      nDataPos = GetDataPos (lpStart, lpnOff, x, y, nTab, FALSE);
      lpbType = MemoryAlloc (nDataPos * sizeof (BYTE));
      mCharType (lpbType, p->lpszText, nDataPos);
      nLength = lpbType[nDataPos - 1] == 2 ? 2 : 1;
      nDataPos -= nLength;
      ctype = GetCharType (p->lpszText + nDataPos, nLength);
      if (ctype == TMAID_CTYPE_ERROR)
        nDataPos = 0;
      else
        while (--nDataPos >= 0)
          if (lpbType[nDataPos] < 2)
            {
              DWORD ct;

              nLength = lpbType[nDataPos] == 1 ? 2 : 1;
              ct = GetCharType (p->lpszText + nDataPos, nLength);
              if (ct == TMAID_CTYPE_ERROR)
                {
                  nDataPos = 0;
                  break;
                }
              if (!(ctype & ct & TMAID_CTYPE_KANA) && ctype != ct)
                {
                  if (ctype != TMAID_CTYPE_NULL)
                    {
                      nDataPos += nLength;
                      break;
                    }
                  ctype = ct;
                }
            }
      MemoryFree (lpbType);
    }
  return GetScreenPos (lpStart, lpnOff, nDataPos, y, nTab);
}


/*  ja:指定範囲のバイト数を求める
      lpStart,ラインバッファ
       lpnOff,現在のy座標
    lpptStart,選択範囲
      lpptEnd,選択範囲
         nTab,タブ
          RET,バイト数                                                      */
int
GetSelByte (LINEBUF **lpStart,
            int      *lpnOff,
            LPPOINT   lpptStart,
            LPPOINT   lpptEnd,
            int       nTab)
{
  int nStart, nEnd, nLength;
  LPLINEBUF p;
  POINT ptStart, ptEnd;

  if (lpptStart->y < lpptEnd->y
                    || lpptStart->y == lpptEnd->y && lpptStart->x < lpptEnd->x)
    {
      ptStart = *lpptStart;
      ptEnd = *lpptEnd;
    }
  else
    {
      ptStart = *lpptEnd;
      ptEnd = *lpptStart;
    }
  nStart = GetDataPos (lpStart, lpnOff, ptStart.x, ptStart.y, nTab, FALSE);
  nEnd = GetDataPos (lpStart, lpnOff, ptEnd.x, ptEnd.y, nTab, FALSE);
  if (ptStart.y == ptEnd.y)
    {
      /* ja:同じ行 */
      nLength = nEnd - nStart;
    }
  else
    {
      /* ja:違う行 */
      int i;

      p = GetLineBuffer (lpStart, lpnOff, ptStart.y);
      nLength = p->nLength - nStart + nEnd + (p->fMargin ? 0 : 2);
      for (i = ptStart.y + 1, p = p->next; i < ptEnd.y && p; i++, p = p->next)
        nLength += p->nLength + (p->fMargin ? 0 : 2);
    }
  return nLength;
}


/*  ja:指定範囲をメモリにコピーする
     lpStart,ラインバッファ
      lpnOff,現在のy座標
    pptStart,選択範囲
     lpptEnd,選択範囲
        nTab,タブ
    lpszText,メモリ                                                         */
void
CpySelMem (LINEBUF **lpStart,
           int      *lpnOff,
           LPPOINT   lpptStart,
           LPPOINT   lpptEnd,
           int       nTab,
           LPSTR     lpszText)
{
  int nStart, nEnd;
  LPLINEBUF p;
  POINT ptStart, ptEnd;

  if (lpptStart->y < lpptEnd->y
                    || lpptStart->y == lpptEnd->y && lpptStart->x < lpptEnd->x)
    {
      ptStart = *lpptStart;
      ptEnd = *lpptEnd;
    }
  else
    {
      ptStart = *lpptEnd;
      ptEnd = *lpptStart;
    }
  nStart = GetDataPos (lpStart, lpnOff, ptStart.x, ptStart.y, nTab, FALSE);
  nEnd = GetDataPos (lpStart, lpnOff, ptEnd.x, ptEnd.y, nTab, FALSE);
  p = GetLineBuffer (lpStart, lpnOff, ptStart.y);
  if (ptStart.y == ptEnd.y)
    {
      /* ja:同じ行 */
      MemoryCopy (lpszText, p->lpszText + nStart,
                                            (nEnd - nStart) * sizeof (CHAR));
    }
  else
    {
      /* ja:違う行 */
      int i, nLength;

      nLength = p->nLength - nStart;
      MemoryCopy (lpszText, p->lpszText + nStart, nLength * sizeof (CHAR));
      if (!p->fMargin)
        {
          MemoryCopy (lpszText + nLength, cCRLF, sizeof (CHAR) * 2);
          nLength += 2;
        }
      for (i = ptStart.y + 1, p = p->next; i < ptEnd.y && p; i++, p = p->next)
        {
          MemoryCopy (lpszText + nLength, p->lpszText,
                                                p->nLength * sizeof (CHAR));
          nLength += p->nLength;
          if (!p->fMargin)
            {
              /* ja:改行 */
              MemoryCopy (lpszText + nLength, cCRLF, sizeof (CHAR) * 2);
              nLength += 2;
            }
        }
      MemoryCopy (lpszText + nLength, p->lpszText, nEnd * sizeof (CHAR));
    }
}


/*  ja:データを取得する
     lpStart,ラインバッファ
      lpnOff,現在のy座標
     lpptGet,取得をはじめるデータ座標
    lpszText,メモリ
     nLength,バイト数
         RET,取得できたバイト数                                             */
int
GetMem (LINEBUF **lpStart,
        int      *lpnOff,
        LPPOINT   lpptGet,
        LPSTR     lpszText,
        int       nLength)
{
  int nMove, nSize, nOff;
  LPLINEBUF p;

  p = GetLineBuffer (lpStart, lpnOff, lpptGet->y);
  nOff = *lpnOff;
  nSize = nLength;
  /* ja:最初の行 */
  nMove = min (max (p->nLength - lpptGet->x, 0), nSize);
  MemoryCopy (lpszText, p->lpszText + lpptGet->x, nMove * sizeof (CHAR));
  nSize -= nMove;
  lpszText += nMove;
  if (!p->fMargin)
    {
      /* ja:改行 */
      nMove = min (nSize, 2);
      MemoryCopy (lpszText + nLength, cCRLF, nMove * sizeof (CHAR));
      nSize -= nMove;
      lpszText += nMove;
    }
  p = p->next;
  while (nSize > 0 && p)
    {
      /* ja:2行目以降 */
      nMove = min (p->nLength, nSize);
      MemoryCopy (lpszText, p->lpszText, nMove * sizeof (CHAR));
      nSize -= nMove;
      lpszText += nMove;
      if (!p->fMargin)
        {
          /* ja:改行 */
          nMove = min (nSize, 2);
          MemoryCopy (lpszText + nLength, cCRLF, nMove * sizeof (CHAR));
          nSize -= nMove;
          lpszText += nMove;
        }
      p = p->next;
    }
  return nLength - nSize;
}


/*  ja:メモリの内容を貼り付ける
     lpStart,ラインバッファ
      lpnOff,現在のy座標
     lpptPut,貼り付ける座標
     lpptNew,新しい座標
        nTab,タブ
    lpszText,メモリ
     nLength,バイト数
     nMargin,右マージン(0ならば折り返しなし)
         RET,挿入された行数                                                 */
int
PutMem (LINEBUF **lpStart,
        int      *lpnOff,
        LPPOINT   lpptPut,
        LPPOINT   lpptNew,
        int       nTab,
        LPCSTR    lpszText,
        int       nLength,
        int       nMargin)
{
  int nTemp, nInsert = 0, nDataPos;
  LPLINEBUF p;
  LPSTR lpszTemp;

  nDataPos = GetDataPos (lpStart, lpnOff, lpptPut->x, lpptPut->y, nTab, FALSE);
  p = GetLineBuffer (lpStart, lpnOff, lpptPut->y);
  if (nDataPos <= 0 && p->prev && p->prev->fMargin && nLength >= 2
                    && MemoryCompare (lpszText, cCRLF, sizeof (CHAR) * 2) == 0)
    {
      /* ja:行頭への挿入で前の行が折り返されており、挿入する文字が改行のとき */
      p = p->prev;
      lpptPut->x = 0;
      for (nDataPos = 0; nDataPos < p->nLength; nDataPos++)
        if (p->lpszText[nDataPos] == '\t')
          lpptPut->x = (lpptPut->x / nTab + 1) * nTab;
        else
          lpptPut->x++;
      lpptPut->y--;
      /* ja:前の行の右端へ移動 */
    }
  if (nDataPos < p->nLength)
    {
      /* ja:挿入位置以降の文字列をバッファに保存する */
      nTemp = p->nLength - nDataPos;
      lpszTemp = MemoryAlloc (nTemp * sizeof (CHAR));
      MemoryCopy (lpszTemp, p->lpszText + nDataPos, nTemp * sizeof (CHAR));
      p->nLength = nDataPos;
    }
  else
    {
      lpszTemp = NULL;
    }
  if (nMargin > 0)
    {
      /* ja:右マージンで折り返す */
      int i = 0, nScreenPos = 0;

      for (nDataPos = 0; nDataPos < p->nLength; nDataPos++)
        if (p->lpszText[nDataPos] == '\t')
          nScreenPos = (nScreenPos / nTab + 1) * nTab;
        else
          nScreenPos++;
      while (i < nLength)
        {
          nDataPos = i;
          while (nDataPos < nLength && (lpszText[nDataPos] != cCRLF[0]
                                        || lpszText[nDataPos + 1] != cCRLF[1]))
            if (lpszText[nDataPos] == '\t')
              {
                nScreenPos = (nScreenPos / nTab + 1) * nTab;
                if (nMargin < nScreenPos)
                  break;
                nDataPos++;
              }
            else if (nDataPos == nLength - 1
                            || !IsDBCSLeadByteEx (CP_ACP, lpszText[nDataPos]))
              {
                nScreenPos++;
                if (nMargin < nScreenPos)
                  break;
                nDataPos++;
              }
            else
              {
                nScreenPos += 2;
                if (nMargin < nScreenPos)
                  break;
                nDataPos += 2;
              }
          p->lpszText = MemoryReAlloc (p->lpszText,
                                (p->nLength + nDataPos - i) * sizeof (CHAR));
          MemoryCopy (p->lpszText + p->nLength, lpszText + i,
                                            (nDataPos - i) * sizeof (CHAR));
          p->nLength += nDataPos - i;
          if (nLength <= nDataPos)
            {
              i = nDataPos;
            }
          else
            {
              LPLINEBUF q;

              q = MemoryAlloc (sizeof (LINEBUF));
              q->nLength = 0;
              q->fMargin = p->fMargin;
              q->lpszText = NULL;
              q->prev = p;
              q->next = p->next;
              p->next = q;
              if (q->next)
                q->next->prev = q;
              if (nDataPos <= nLength - 2 && lpszText[nDataPos] == cCRLF[0]
                                        && lpszText[nDataPos + 1] == cCRLF[1])
                {
                  /* ja:改行 */
                  i = nDataPos + 2;
                  p->fMargin = FALSE;
                }
              else
                {
                  /* ja:擬似改行 */
                  i = nDataPos;
                  p->fMargin = TRUE;
                }
              p = q;
              nScreenPos = 0;
              nInsert++;
            }
        }
    }
  else
    {
      /* ja:右マージンで折り返さない */
      int i = 0;

      while (i < nLength)
        {
          for (nDataPos = i;
                        nDataPos < nLength && (lpszText[nDataPos] != cCRLF[0]
                        || lpszText[nDataPos + 1] != cCRLF[1]); nDataPos++);
          p->lpszText = MemoryReAlloc (p->lpszText,
                                (p->nLength + nDataPos - i) * sizeof (CHAR));
          MemoryCopy (p->lpszText + p->nLength, lpszText + i,
                                            (nDataPos - i) * sizeof (CHAR));
          p->nLength += nDataPos - i;
          if (nLength <= nDataPos)
            {
              i = nDataPos;
            }
          else
            {
              LPLINEBUF q;

              i = nDataPos + 2;
              q = MemoryAlloc (sizeof (LINEBUF));
              q->nLength = 0;
              q->fMargin = FALSE;
              q->lpszText = NULL;
              q->prev = p;
              q->next = p->next;
              p->next = q;
              if (q->next)
                q->next->prev = q;
              p = q;
              nInsert++;
            }
        }
    }
  if (lpptNew)
    {
      lpptNew->x = 0;
      for (nDataPos = 0; nDataPos < p->nLength; nDataPos++)
        if (p->lpszText[nDataPos] == '\t')
          lpptNew->x = (lpptNew->x / nTab + 1) * nTab;
        else
          lpptNew->x++;
      lpptNew->y = lpptPut->y + nInsert;
      if (nMargin > 0 && p->fMargin && (nMargin <= lpptNew->x
                                || lpszTemp && nMargin - 1 <= lpptNew->x
                                && IsDBCSLeadByteEx (CP_ACP, lpszTemp[0])))
        {
          lpptNew->x = 0;
          lpptNew->y++;
        }
    }
  if (lpszTemp)
    {
      /* ja:挿入位置以降の文字列を戻す */
      p->lpszText = MemoryReAlloc (p->lpszText, p->nLength + nTemp);
      MemoryCopy (p->lpszText + p->nLength, lpszTemp, nTemp * sizeof (CHAR));
      p->nLength += nTemp;
      MemoryFree (lpszTemp);
    }
  if (nMargin > 0)
    /* ja:右マージンで折り返す処理 */
    while (p)
      {
        int nScreenPos = 0;

        nDataPos = 0;
        while (nDataPos < p->nLength)
          if (p->lpszText[nDataPos] == '\t')
            {
              nScreenPos = (nScreenPos / nTab + 1) * nTab;
              if (nMargin < nScreenPos)
                break;
              nDataPos++;
            }
          else if (nDataPos == p->nLength - 1
                        || !IsDBCSLeadByteEx (CP_ACP, p->lpszText[nDataPos]))
            {
              nScreenPos++;
              if (nMargin < nScreenPos)
                break;
              nDataPos++;
            }
          else
            {
              nScreenPos += 2;
              if (nMargin < nScreenPos)
                break;
              nDataPos += 2;
            }
        if (nMargin < nScreenPos)
          {
            /* ja:右マージンを超えているとき、擬似改行する */
            LPLINEBUF q;

            q = MemoryAlloc (sizeof (LINEBUF));
            q->nLength = p->nLength - nDataPos;
            q->fMargin = p->fMargin;
            q->lpszText = MemoryAlloc (q->nLength * sizeof (CHAR));
            q->prev = p;
            q->next = p->next;
            p->next = q;
            if (q->next)
              q->next->prev = q;
            MemoryCopy (q->lpszText, p->lpszText + nDataPos,
                                                q->nLength * sizeof (CHAR));
            p->nLength = nDataPos;
            p->fMargin = TRUE;
            p->lpszText = MemoryReAlloc (p->lpszText,
                                                p->nLength * sizeof (CHAR));
            p = q;
            nInsert++;
          }
        else if (nScreenPos < nMargin && p->fMargin && p->next)
          {
            /* ja:右マージンを下回り擬似改行のとき、次の行とあわせる */
            LPLINEBUF q;

            nDataPos = p->nLength;
            q = p->next;
            if (q->next)
              q->next->prev = p;
            p->next = q->next;
            p->nLength += q->nLength;
            p->fMargin = q->fMargin;
            p->lpszText = MemoryReAlloc (p->lpszText,
                                                p->nLength * sizeof (CHAR));
            MemoryCopy (p->lpszText + nDataPos, q->lpszText,
                                                q->nLength * sizeof (CHAR));
            MemoryFree (q->lpszText);
            MemoryFree (q);
            nInsert--;
          }
        else
          {
            break;
          }
      }
  return nInsert;
}


/*  指定範囲を削除する
      lpStart,ラインバッファ
       lpnOff,現在のy座標
    lpptStart,選択範囲
      lpptEnd,選択範囲
         nTab,タブ
     nMargin,右マージン(0ならば折り返しなし)
          RET,削除された行数                                                */
int
DelSelMem (LINEBUF **lpStart,
           int      *lpnOff,
           LPPOINT   lpptStart,
           LPPOINT   lpptEnd,
           int       nTab,
           int       nMargin)
{
  int i, nStart, nEnd, nDelete;
  LPLINEBUF p, q;
  POINT ptStart, ptEnd;

  if (lpptStart->y < lpptEnd->y
                    || lpptStart->y == lpptEnd->y && lpptStart->x < lpptEnd->x)
    {
      ptStart = *lpptStart;
      ptEnd = *lpptEnd;
    }
  else
    {
      ptStart = *lpptEnd;
      ptEnd = *lpptStart;
    }
  nStart = GetDataPos (lpStart, lpnOff, ptStart.x, ptStart.y, nTab, FALSE);
  nEnd = GetDataPos (lpStart, lpnOff, ptEnd.x, ptEnd.y, nTab, FALSE);
  p = GetLineBuffer (lpStart, lpnOff, ptStart.y);
  if (ptStart.y == ptEnd.y)
    {
      /* ja:同じ行 */
      MemoryCopy (p->lpszText + nStart, p->lpszText + nEnd,
                                        (p->nLength - nEnd) * sizeof (CHAR));
      p->nLength -= nEnd - nStart;
      p->lpszText = MemoryReAlloc (p->lpszText, p->nLength * sizeof (CHAR));
      nDelete = 0;
    }
  else
    {
      /* ja:違う行 */
      LPLINEBUF r;

      for (i = ptStart.y + 1, q = p->next; i < ptEnd.y && q; i++, q = r)
        {
          MemoryFree (q->lpszText);
          r = q->next;
          MemoryFree (q);
        }
      if (q->next)
        q->next->prev = p;
      p->next = q->next;
      p->nLength = nStart + q->nLength - nEnd;
      p->fMargin = q->fMargin;
      p->lpszText = MemoryReAlloc (p->lpszText, p->nLength * sizeof (CHAR));
      MemoryCopy (p->lpszText + nStart, q->lpszText + nEnd,
                                        (q->nLength - nEnd) * sizeof (CHAR));
      MemoryFree (q->lpszText);
      MemoryFree (q);
      nDelete = ptEnd.y - ptStart.y;
    }
  q = p->prev;
  if (q && q->fMargin && ptStart.x <= 0)
    {
      /* ja:前の行が擬似改行で削除開始が行頭のとき */
      int nScreenPos = 0;

      if (p->nLength > 0)
        {
          int nDataPos = 0;

          while (nDataPos < q->nLength)
            if (q->lpszText[nDataPos] == '\t')
              {
                nScreenPos = (nScreenPos / nTab + 1) * nTab;
                nDataPos++;
              }
            else if (IsDBCSLeadByteEx (CP_ACP, q->lpszText[nDataPos]))
              {
                nScreenPos += 2;
                nDataPos += 2;
              }
            else
              {
                nScreenPos++;
                nDataPos++;
              }
          if (p->lpszText[0] == '\t')
            nScreenPos = (nScreenPos / nTab + 1) * nTab;
          else if (IsDBCSLeadByteEx (CP_ACP, p->lpszText[0]))
            nScreenPos += 2;
          else
            nScreenPos++;
        }
      if (p->nLength <= 0 || nScreenPos <= nMargin)
        {
          /* ja:右マージンを下回り擬似改行のとき、次の行とあわせる */
          int nDataPos;

          nDataPos = q->nLength;
          if (p->next)
            p->next->prev = q;
          q->next = p->next;
          q->nLength += p->nLength;
          q->fMargin = p->fMargin;
          q->lpszText = MemoryReAlloc (q->lpszText,
                                                q->nLength * sizeof (CHAR));
          MemoryCopy (q->lpszText + nDataPos, p->lpszText,
                                                p->nLength * sizeof (CHAR));
          MemoryFree (p->lpszText);
          MemoryFree (p);
          nDelete++;
          p = q;
          if (lpptStart->y < lpptEnd->y
                    || lpptStart->y == lpptEnd->y && lpptStart->x < lpptEnd->x)
            {
              lpptStart->x = nScreenPos;
              lpptStart->y--;
            }
          else
            {
              lpptEnd->x = nScreenPos;
              lpptEnd->y--;
            }
        }
    }
  if (nMargin > 0)
    /* ja:右マージンで折り返す処理 */
    while (p)
      {
        int nDataPos = 0, nScreenPos = 0;

        while (nDataPos < p->nLength)
          if (p->lpszText[nDataPos] == '\t')
            {
              nScreenPos = (nScreenPos / nTab + 1) * nTab;
              if (nMargin < nScreenPos)
                break;
              nDataPos++;
            }
          else if (nDataPos == p->nLength - 1
                        || !IsDBCSLeadByteEx (CP_ACP, p->lpszText[nDataPos]))
            {
              nScreenPos++;
              if (nMargin < nScreenPos)
                break;
              nDataPos++;
            }
          else
            {
              nScreenPos += 2;
              if (nMargin < nScreenPos)
                break;
              nDataPos += 2;
            }
        if (nMargin < nScreenPos)
          {
            /* ja:右マージンを超えているとき、擬似改行する */
            q = MemoryAlloc (sizeof (LINEBUF));
            q->nLength = p->nLength - nDataPos;
            q->fMargin = p->fMargin;
            q->lpszText = MemoryAlloc (q->nLength * sizeof (CHAR));
            q->prev = p;
            q->next = p->next;
            p->next = q;
            if (q->next)
              q->next->prev = q;
            MemoryCopy (q->lpszText, p->lpszText + nDataPos,
                                                q->nLength * sizeof (CHAR));
            p->nLength = nDataPos;
            p->fMargin = TRUE;
            p->lpszText = MemoryReAlloc (p->lpszText,
                                                p->nLength * sizeof (CHAR));
            p = q;
            nDelete--;
          }
        else if (nScreenPos < nMargin && p->fMargin && p->next)
          {
            /* ja:右マージンを下回り擬似改行のとき、次の行とあわせる */
            nDataPos = p->nLength;
            q = p->next;
            if (q->next)
              q->next->prev = p;
            p->next = q->next;
            p->nLength += q->nLength;
            p->fMargin = q->fMargin;
            p->lpszText = MemoryReAlloc (p->lpszText,
                                                p->nLength * sizeof (CHAR));
            MemoryCopy (p->lpszText + nDataPos, q->lpszText,
                                                q->nLength * sizeof (CHAR));
            MemoryFree (q->lpszText);
            MemoryFree (q);
            nDelete++;
          }
        else if (!p->fMargin)
          {
            break;
          }
        else
          {
            p = p->next;
          }
      }
  return nDelete;
}
